package ch.sahits.game.javafx.control;

import ch.sahits.game.event.NavigationStateChange;
import ch.sahits.game.event.data.NewGameClient;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.image.IImageUtilities;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.model.IMap;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.util.IRebinabable;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.application.Platform;
import javafx.geometry.Dimension2D;
import javafx.geometry.Rectangle2D;
import javafx.scene.Group;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;

/**
 * Mini map control
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Nov 1, 2013
 *
 */
@LazySingleton
@ClassCategory({EClassCategory.JAVAFX, EClassCategory.SINGLETON_BEAN})
public class MiniMap extends Group implements IRebinabable {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private IMap map;
    @Autowired
    private IImageUtilities imageUtilities;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    private ImageView imgView;
    private boolean navigationDisabled = false;

    @PostConstruct
    private void initializeControl() {
        clientServerEventBus.register(this);
        clientEventBus.register(this);
        setManaged(false);
        imgView = new ImageView();
        imgView.setFitWidth(236);
        imgView.setFitHeight(192);
        imgView.setPreserveRatio(false);
        // scale the map to the correct size add white dots on cities that have ship in them
        // clicking on the city changes into the other city (if there is a ship or kontor
        // right clicking on a point on the map sends the active ship to that location
        getChildren().addAll(imgView);
    }

    /**
     * Initialize the player that goes with this client. The player does not change later on,
     * @param newGameDTO
     */
    @Subscribe
    public void initializeState(NewGameClient newGameDTO) {
        final IPlayer player = newGameDTO.getPlayer();
        initializeMapImage(player);
     }

    private void initializeMapImage(IPlayer player) {
        Image mapImage = imageUtilities.createMapWithCities(map, player);
        mapImage =  imageUtilities.createImageWithShipsInCities(map, player, mapImage);
        Dimension2D dim = map.getDimension();
        double factor = 192 / dim.getHeight();
        double widthOrigImg = 236 / factor;
        double xlocation = player.getHometown().getCoordinates().getX();
        double x = Math.max(0, xlocation - widthOrigImg/2);
        Rectangle2D viewport = new Rectangle2D(x, 0, widthOrigImg, dim.getHeight());
        final Image transferableMap = mapImage;

        Platform.runLater(() -> {
            imgView.setViewport(viewport);
            imgView.setImage(transferableMap);
        });
    }

    @Subscribe
    public void handleNAvigationChange(NavigationStateChange event) {
        switch (event.getChange()) {
            case DISABLE_NAVIGATION:
                navigationDisabled = true;
                break;
            case ENABLE_NAVIGATION:
                navigationDisabled = false;
                break;
            default:
                logger.info("Not interested in navigation change event: "+event.getChange());
        }
    }

    @Override
    public void rebind() {
        initializeMapImage(viewState.getPlayer());
    }
}
