package ch.sahits.game.event.handler.impl;

import ch.sahits.game.event.data.ShipEntersPortEvent;
import ch.sahits.game.event.data.CourierPatrolStateInternal;
import ch.sahits.game.event.handler.ShipEntersPortEventListener;
import ch.sahits.game.graphic.event.task.ClientTaskFactory;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.Optional;
import java.util.Random;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 16, 2014
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class NextCourierPatrolStation  extends ShipEntersPortEventListener {
    /** Next station may be null if it is the last */
    private final CourierPatrolStateInternal state;
    private final IPlayer player;
    private final DateTime deadline;
    @Autowired
    private Date date;
    @Autowired
    @XStreamOmitField
    private ClientTaskFactory taskFactory;
    @Autowired
    private Random rnd;

    @Autowired
    @Qualifier("clientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientEventBus;
    @Autowired
    @XStreamOmitField
    private Locale locale;
    @Autowired
    @XStreamOmitField
    private MessageSource messageSource;

    /**
     * Create the next patrol station listener. This listener adds itself to the event queue and removes the first city from the
     * patrol state
     * @param ship
     * @param city
     * @param state
     * @param player
     * @param deadline
     */
    public NextCourierPatrolStation(IShip ship, ICity city, CourierPatrolStateInternal state, IPlayer player, DateTime deadline) {
        super(ship, city);
        this.state = state;
        this.player = player;
        this.deadline = deadline;
        state.removeFirstCity();
    }
    @PostConstruct
    private void initializeEventBus() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void unregisterEventBus() { clientServerEventBus.unregister(this);}

    @Override
    public void handleShipEntersPort(ShipEntersPortEvent event) {
        super.handleShipEntersPort(event);
        DateTime now = date.getCurrentDate();
        if (now.isAfter(deadline)) {
            clientEventBus.post(new DisplayMessage("ch.sahits.game.event.handler.impl.NextCourierPatrolStation.unreliable"));
            ((IShip)getShip()).leavePassanger();
            clientServerEventBus.unregister(this);
        }
    }

    @Override
    public boolean execute() {
        Optional<ICity> nextCity = state.nextDestination();
        if (!nextCity.isPresent()){ // last leg
            int premium = state.getPremium();
            player.updateCash(premium);
            ((IShip)getShip()).leavePassanger();
            clientEventBus.post(new DisplayMessage("ch.sahits.game.event.handler.impl.NextCourierPatrolStation.salary", new Object[]{premium}));
            unregisterEventBus();
        } else { // next leg
            ICity next = nextCity.get();
            state.removeFirstCity();
            int minutes = rnd.nextInt(600);

            DateTime postMessage = date.getCurrentDate().plusMinutes(minutes);
            taskFactory.getPostponedDisplayMessage(postMessage, "ch.sahits.game.event.handler.impl.NextCourierPatrolStation.nextStation", new Object[]{next.getName()});
            DateTime deadline = date.getCurrentDate().plusDays(6);
            new NextCourierPatrolStation(((IShip)getShip()), state.nextDestination().get(), state, player, deadline);
        }
        return true;
    }
}
