package ch.sahits.game.graphic.display.dialog.cityhall;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.event.data.MapUpdateCityAdd;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.graphic.event.task.ClientTaskFactory;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.dialog.DialogTemplate;
import ch.sahits.game.openpatrician.engine.land.city.CityHallEngine;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IMap;
import ch.sahits.game.openpatrician.model.city.CityFactory;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.cityhall.ECityViolationPunishment;
import ch.sahits.game.openpatrician.model.city.cityhall.IAldermanTask;
import ch.sahits.game.openpatrician.model.city.cityhall.IBuildLandPassage;
import ch.sahits.game.openpatrician.model.city.cityhall.ICapturePirateNest;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityHall;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityViolation;
import ch.sahits.game.openpatrician.model.city.cityhall.ICustomsViolation;
import ch.sahits.game.openpatrician.model.city.cityhall.IFoundNewSettlement;
import ch.sahits.game.openpatrician.model.city.cityhall.IHelpCity;
import ch.sahits.game.openpatrician.model.city.cityhall.IHuntPirate;
import ch.sahits.game.openpatrician.model.city.cityhall.IPirateSupportViolation;
import ch.sahits.game.openpatrician.model.city.cityhall.IPlunderTradingOfficesViolation;
import ch.sahits.game.openpatrician.model.city.cityhall.ISpecialTaxViolation;
import ch.sahits.game.openpatrician.model.city.cityhall.ITreasury;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.AldermanOffice;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.BuildLandPassage;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.CapturePirateNest;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.CityHall;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.CityViolation;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.FoundNewSettlement;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.HelpCity;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.sea.PirateNest;
import ch.sahits.game.openpatrician.server.MapLocationDetectionModel;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.css.CssMetaData;
import javafx.css.SimpleStyleableObjectProperty;
import javafx.css.StyleConverter;
import javafx.css.Styleable;
import javafx.css.StyleableObjectProperty;
import javafx.css.StyleableProperty;
import javafx.event.EventHandler;
import javafx.scene.control.Control;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.HBox;
import javafx.scene.layout.VBox;
import javafx.scene.text.Font;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Apr 05, 2015
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class AldermanOfficeDialog extends CloseButtonDialog {
    @Autowired
    private ClientViewState viewState;
    @Autowired
    private CityHallEngine cityHallEngine;
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private IMap map;
    @Autowired
    private Date date;
    @Autowired
    private ClientTaskFactory taskFactory;
    @Autowired
    private MapLocationDetectionModel mapLocationDetectionModel;
    @Autowired
    private CityFactory cityFactory;


    private final IntegerProperty currentIndex = new SimpleIntegerProperty(0);
    private final ICityPlayerProxyJFX city;
    @ListType(IAldermanTask.class)
    private final List<IAldermanTask> tasks = new ArrayList<>();
    private VBox content;

    public AldermanOfficeDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
        getStylesheets().add(this.getClass().getResource("/styles/base.css").toExternalForm());
        getStyleClass().add("dialog");
    }

    @PostConstruct
    private void initializeDialog() {
        ICityHall cityHall = cityHallEngine.findCityHall(city.getCity());
        final AldermanOffice aldermanOffice = (AldermanOffice) cityHall.getAldermanOffice().get();
        tasks.addAll(aldermanOffice.getTasks());
        final int actionButtonX = (WIDTH - 124) / 2;

        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.aldermanOffice", new Object[]{}, locale.getCurrentLocal()));
        content = new VBox(24);
        String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.taskSelection", new Object[]{city.getCity().getName()}, locale.getCurrentLocal());
        DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
        DecoratedText decoratedTask = createCurrentTask();
        HBox navigationBar = new HBox(15);
        OpenPatricianSmallWaxButton back = new OpenPatricianSmallWaxButton("<");
        back.getStyleClass().add("actionButton");
        back.setId("back");
        back.setOnAction(createPreviousAction());
        OpenPatricianSmallWaxButton next = new OpenPatricianSmallWaxButton(">");
        next.getStyleClass().add("actionButton");
        next.setId("next");
        next.setOnAction(createNextAction());
        OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernBaseSideRoomPersonDialog.accept", new Object[]{}, locale.getCurrentLocal()));
        action.getStyleClass().add("actionButton");
        action.setId("action");
        action.setOnAction(createAction(tasks.get(currentIndex.get()), aldermanOffice));
        navigationBar.getChildren().addAll(back, action, next);
        navigationBar.setLayoutX((CloseButtonDialog.WIDTH - navigationBar.getWidth()) / 2);
        navigationBar.widthProperty().addListener((observable, oldWidth, newWidth) -> {
            navigationBar.setLayoutX((CloseButtonDialog.WIDTH - newWidth.doubleValue()) / 2);
        });
        navigationBar.setLayoutY(CLOSE_BTN_Y_POS - 64);

        content.getChildren().addAll(text, decoratedTask);

        OpenPatricianLargeWaxButton meetings = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.meetings", new Object[]{}, locale.getCurrentLocal()));
        meetings.getStyleClass().add("actionButton");
        meetings.setId("meetings");
        meetings.setOnAction(createMeetingsAction());
        meetings.setLayoutX(actionButtonX);
        meetings.setLayoutY(CLOSE_BTN_Y_POS - 24);
        if (city.getPlayer().getRank() != ESocialRank.ALDERMAN) {
            meetings.setVisible(false);
        }

        content.setLayoutY(text.getLayoutY());
        content.setLayoutX(text.getLayoutX());
        getContent().addAll(content, navigationBar, meetings);

    }

    private EventHandler<MouseEvent> createMeetingsAction() {
        return (mouseEvent) -> {
            CityHall cityHall = (CityHall) cityHallEngine.findCityHall(city.getCity());
            AldermanOffice office = (AldermanOffice) cityHall.getAldermanOffice().get();
            // remove content
            getContent().clear();
            if (!office.getViolation().isPresent()) {
                String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.noViolations", new Object[]{}, locale.getCurrentLocal());
                DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
                getContent().add(text);
            } else {
                ICityViolation violation = office.getViolation().get();
                VBox box = new VBox();
                if (violation instanceof ICustomsViolation) {
                    String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.customViolation", new Object[]{violation.getCity().getName()}, locale.getCurrentLocal());
                    DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
                    box.getChildren().add(text);
                }  else if (violation instanceof IPirateSupportViolation) {
                    String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.pirateSupport", new Object[]{violation.getCity().getName()}, locale.getCurrentLocal());
                    DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
                    box.getChildren().add(text);
                } else if (violation instanceof IPlunderTradingOfficesViolation) {
                    String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.plunderOffice", new Object[]{violation.getCity().getName()}, locale.getCurrentLocal());
                    DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
                    box.getChildren().add(text);
                } else if (violation instanceof ISpecialTaxViolation) {
                    String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.specialTaxViolation", new Object[]{violation.getCity().getName()}, locale.getCurrentLocal());
                    DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
                    box.getChildren().add(text);
                } else {
                    throw new IllegalStateException("Unhandled violation type: " + violation.getClass().getName());
                }
                String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.measureSelection", new Object[]{}, locale.getCurrentLocal());
                DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
                box.setLayoutY(text.getLayoutY());
                box.setLayoutX(text.getLayoutX());

                box.getChildren().add(text);
                template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.noPunishment", new Object[]{}, locale.getCurrentLocal());
                text = textFactory.createDecoratedText(template, new HashMap<>());
                text.setId("noPunishment");
                text.setOnMouseReleased((evt) -> {
                    office.setViolation(Optional.empty());
                    taskFactory.getPostponedDisplayMessage(violation.getDate(), "ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.displeasure", new Object[]{violation.getCity().getName()});
                    executeOnCloseButtonClicked();
                });

                box.getChildren().add(text);
                ITreasury otherTreasury = cityHallEngine.findCityHall(violation.getCity()).getTreasury();
                int fine = (int) (otherTreasury.getCash()/100);
                template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.smallFine", new Object[]{fine}, locale.getCurrentLocal());
                text = textFactory.createDecoratedText(template, new HashMap<>());
                text.setId("smallFine");
                text.setOnMouseReleased((evt) -> {
                    ((CityViolation) violation).setPunishment(ECityViolationPunishment.SMALL_FINE);
                    DateTime when = date.getCurrentDate().plusMonths(2);
                    cityHall.setHanseaticMeetingDate(Optional.of(when));
                    executeOnCloseButtonClicked();
                });
                box.getChildren().add(text);
                fine = (int) (otherTreasury.getCash()/10);
                template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.mediumFine", new Object[]{fine}, locale.getCurrentLocal());
                text = textFactory.createDecoratedText(template, new HashMap<>());
                text.setId("mediumFine");
                text.setOnMouseReleased((evt) -> {
                    ((CityViolation) violation).setPunishment(ECityViolationPunishment.MEDIUM_FINE);
                    DateTime when = date.getCurrentDate().plusMonths(2);
                    cityHall.setHanseaticMeetingDate(Optional.of(when));
                    executeOnCloseButtonClicked();
                });
                box.getChildren().add(text);
                fine = (int) (otherTreasury.getCash()*0.9);
                template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.largeFine", new Object[]{fine}, locale.getCurrentLocal());
                text = textFactory.createDecoratedText(template, new HashMap<>());
                text.setId("largeFine");
                text.setOnMouseReleased((evt) -> {
                    ((CityViolation) violation).setPunishment(ECityViolationPunishment.LARGE_FINE);
                    DateTime when = date.getCurrentDate().plusMonths(2);
                    cityHall.setHanseaticMeetingDate(Optional.of(when));
                    executeOnCloseButtonClicked();
                });
                box.getChildren().add(text);
                template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.boycott", new Object[]{fine}, locale.getCurrentLocal());
                text = textFactory.createDecoratedText(template, new HashMap<>());
                text.setId("boycott");
                text.setOnMouseReleased((evt) -> {
                    ((CityViolation)violation).setPunishment(ECityViolationPunishment.BLOCKADE);
                    DateTime when = date.getCurrentDate().plusMonths(2);
                    cityHall.setHanseaticMeetingDate(Optional.of(when));
                    executeOnCloseButtonClicked();
                });
                box.getChildren().add(text);
                getContent().addAll(box);
            }
        };
    }

    private EventHandler<MouseEvent> createAction(IAldermanTask task, AldermanOffice office) {
        return (mouseEvent) -> {
            if (task instanceof IBuildLandPassage) {
                BuildLandPassage concreteTask = (BuildLandPassage) task;
                DateTime deadline = date.getCurrentDate().plusDays(concreteTask.getDuration());
                Object[] args = new Object[]{concreteTask.getFromCity().getName(), concreteTask.getToCity().getName(), date.todisplayString(deadline.getDayOfMonth(), deadline.getMonthOfYear(), deadline.getYear())};
                DialogTemplate template = DialogTemplate.builder()
                        .closable(true)
                        .titleKey("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.buildLandPassageTitle")
                        .messageKey("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.buildLandPassageParagraph")
                        .messageArgs(args)
                        .build();
                taskFactory.getPostponedDisplayDialogMessage(date.getCurrentDate().plusHours(18), template);
                office.addTask(task, deadline, city.getPlayer());
            }
            if (task instanceof IFoundNewSettlement) {
                FoundNewSettlement concreteTask = (FoundNewSettlement) task;
                DateTime deadline = date.getCurrentDate().plusDays(concreteTask.getDuration());
                Object[] args = new Object[]{concreteTask.getName(), date.todisplayString(deadline.getDayOfMonth(), deadline.getMonthOfYear(), deadline.getYear())};
                DialogTemplate template = DialogTemplate.builder()
                        .closable(true)
                        .titleKey("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.foundNewSettlementTitle")
                        .messageKey("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.foundNewSettlementParagraph")
                        .messageArgs(args)
                        .build();
                taskFactory.getPostponedDisplayDialogMessage(date.getCurrentDate().plusHours(18), template);
                ICity foundingTown = cityFactory.createFutureCity(concreteTask);
                map.buildSettlement(foundingTown, city.getPlayer());
                clientEventBus.post(new MapUpdateCityAdd(foundingTown, city.getPlayer()));
                office.addTask(task, deadline, city.getPlayer());
            }
            if (task instanceof IHelpCity) {
                HelpCity concreteTask = (HelpCity) task;
                DateTime deadline = date.getCurrentDate().plusDays(concreteTask.getDuration());
                Object[] args = new Object[]{date.todisplayString(deadline.getDayOfMonth(), deadline.getMonthOfYear(), deadline.getYear()), concreteTask.getCity().getName()};
                DialogTemplate template = DialogTemplate.builder()
                        .closable(true)
                        .titleKey("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.helpCityTitle")
                        .messageKey("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.helpCityParagraph")
                        .messageArgs(args)
                        .build();
                taskFactory.getPostponedDisplayDialogMessage(date.getCurrentDate().plusHours(18), template);
                office.addTask(task, deadline, city.getPlayer());
            }
            if (task instanceof IHuntPirate) {
               IHuntPirate concreteTask = (IHuntPirate) task;
                DateTime deadline = date.getCurrentDate().plusDays(concreteTask.getDuration());
                Object[] args = new Object[]{concreteTask.getPirate().getName(), date.todisplayString(deadline.getDayOfMonth(), deadline.getMonthOfYear(), deadline.getYear())};
                DialogTemplate template = DialogTemplate.builder()
                        .closable(true)
                        .titleKey("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.huntPirateTitle")
                        .messageKey("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.huntPirateParagraph")
                        .messageArgs(args)
                        .build();
                taskFactory.getPostponedDisplayDialogMessage(date.getCurrentDate().plusHours(18), template);
                office.addTask(task, deadline, city.getPlayer());

            }
            if (task instanceof ICapturePirateNest) {
                CapturePirateNest concreteTask = (CapturePirateNest) task;
                DateTime deadline = date.getCurrentDate().plusDays(concreteTask.getDuration());
                Object[] args = new Object[]{date.todisplayString(deadline.getDayOfMonth(), deadline.getMonthOfYear(), deadline.getYear()), concreteTask.getPirateNestMap().getImageName()};
                DialogTemplate template = DialogTemplate.builder()
                        .closable(true)
                        .titleKey("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.destroyPiratesNestTitle")
                        .messageKey("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.destroyPiratesNestParagraph")
                        .messageArgs(args)
                        .build();
                taskFactory.getPostponedDisplayDialogMessage(date.getCurrentDate().plusHours(18), template);
                office.addTask(task, deadline, city.getPlayer());
                PirateNest nest = new PirateNest(concreteTask.getLocation());
                map.getPirateNests().add(nest);
                mapLocationDetectionModel.addSegment(concreteTask.getPirateNestMap());
            }
            System.out.println("Accepted task: "+task.getClass().getName());
            cityHallEngine.assignTask(city.getPlayer(), task);
            office.getTasks().remove(task);
            executeOnCloseButtonClicked();
        };
    }

    /**
     * Move the selection to the next index
     * @return .
     */
    private EventHandler<MouseEvent> createNextAction() {
        return (mouseEvent) -> {
            if (currentIndex.get() == tasks.size() - 1) {
                currentIndex.set(0);
            } else {
                currentIndex.set(currentIndex.get() + 1);
            }
            content.getChildren().remove(1);
            content.getChildren().add(1, createCurrentTask());
        };
    }

    /**
     * move the selection to the previous index.
     * @return  .
     */
    private EventHandler<MouseEvent> createPreviousAction() {
        return (mouseEvent)  -> {
            if (currentIndex.get() == 0) {
                currentIndex.set(tasks.size() - 1);
            } else {
                currentIndex.set(currentIndex.get() - 1);
            }
            content.getChildren().remove(1);
            content.getChildren().add(1, createCurrentTask());
        };
    }

    private DecoratedText createCurrentTask() {
        IAldermanTask task = tasks.get(currentIndex.get());
        if (task instanceof IBuildLandPassage) {
            IBuildLandPassage passage = (IBuildLandPassage) task;
            double width = map.getDimension().getWidth();
            double heigth = map.getDimension().getHeight();

            String orientation = messageSource.getMessage("orientation.west", new Object[]{city.getCity().getName()}, locale.getCurrentLocal());
            if (passage.getFromCity().getCoordinates().getY() < heigth/2) {
                orientation = messageSource.getMessage("orientation.north", new Object[]{city.getCity().getName()}, locale.getCurrentLocal());
            }
            if (passage.getFromCity().getCoordinates().getX() > width/2) {
                orientation = messageSource.getMessage("orientation.east", new Object[]{city.getCity().getName()}, locale.getCurrentLocal());
            }
            String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.taskNewPassage", new Object[]{orientation}, locale.getCurrentLocal());
            return textFactory.createDecoratedText(template, new HashMap<>());
        }
        if (task instanceof ICapturePirateNest) {
            String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.taskPirateNest", new Object[]{}, locale.getCurrentLocal());
            return textFactory.createDecoratedText(template, new HashMap<>());
        }
        if (task instanceof IFoundNewSettlement) {
            IFoundNewSettlement newSettlement = (IFoundNewSettlement) task;
            double width = map.getDimension().getWidth();
            double heigth = map.getDimension().getHeight();
            String orientation = messageSource.getMessage("orientation.west", new Object[]{city.getCity().getName()}, locale.getCurrentLocal());
            if (newSettlement.getLocation().getY() < heigth/2) {
                orientation = messageSource.getMessage("orientation.north", new Object[]{city.getCity().getName()}, locale.getCurrentLocal());
            }
            if (newSettlement.getLocation().getX() > width/2) {
                orientation = messageSource.getMessage("orientation.east", new Object[]{city.getCity().getName()}, locale.getCurrentLocal());
            }
            String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.taskNewTown", new Object[]{orientation}, locale.getCurrentLocal());
            return textFactory.createDecoratedText(template, new HashMap<>());
        }
        if (task instanceof IHelpCity) {
            String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.taskHelpCity", new Object[]{}, locale.getCurrentLocal());
            return textFactory.createDecoratedText(template, new HashMap<>());
        }
        if (task instanceof IHuntPirate) {
            IHuntPirate pirateTask = (IHuntPirate) task;
            String name = pirateTask.getPirate().getName();
            String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.AldermanOfficeDialog.taskPirateHunt", new Object[]{name}, locale.getCurrentLocal());
            return textFactory.createDecoratedText(template, new HashMap<>());
        }
        throw new IllegalStateException("Unknown task type: "+task.getClass().getName());
    }


    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_CITY_HALL_ALDERMAN);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }

    private StyleableObjectProperty<Font> font;

    public Font getFont() {
        return font == null ? Font.getDefault() : font.get();
    }
    public void setFont(Font font) {
        this.font.set(font);
    }
    public StyleableObjectProperty<Font> fontProperty() {
        if (font == null) {
            font = new SimpleStyleableObjectProperty<Font>(StyleableProperties.FONT, AldermanOfficeDialog.this, "font", Font.getDefault());
        }
        return font;
    }

    private static class StyleableProperties {
        private static final CssMetaData< AldermanOfficeDialog, Font> FONT =
                new CssMetaData<AldermanOfficeDialog, Font>("-fx-font",
                        StyleConverter.getFontConverter(), Font.getDefault()) {
                    @Override
                    public boolean isSettable(AldermanOfficeDialog control) {
                        return control.font == null || !control.font.isBound();
                    }
                    @Override
                    public StyleableProperty<Font> getStyleableProperty(AldermanOfficeDialog control) {
                        return control.fontProperty();
                    }
                };
        private static final List<CssMetaData<? extends Styleable, ?>> STYLEABLES;
        static {
            final List<CssMetaData<? extends Styleable, ?>> styleables =
                    new ArrayList<CssMetaData<? extends Styleable, ?>>(Control.getClassCssMetaData());
            Collections.addAll(styleables,
                    FONT
            );
            STYLEABLES = Collections.unmodifiableList(styleables);
        }
    }
    public List<CssMetaData<? extends Styleable, ?>> getControlCssMetaData() {
        return getClassCssMetaData();
    }
    public static List<CssMetaData<? extends Styleable, ?>> getClassCssMetaData() {
        return StyleableProperties.STYLEABLES;
    }
}
