package ch.sahits.game.graphic.display.dialog.loaner;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.graphic.image.impl.XMLImageLoader;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.engine.land.city.LoanerEngine;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.city.ILoanRequest;
import ch.sahits.game.openpatrician.model.city.impl.Debt;
import ch.sahits.game.openpatrician.model.city.impl.IDebt;
import ch.sahits.game.openpatrician.model.city.impl.LoanerState;
import ch.sahits.game.openpatrician.model.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.collections.ObservableList;
import javafx.event.EventHandler;
import javafx.scene.Node;
import javafx.scene.control.Label;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.StackPane;
import javafx.scene.layout.VBox;
import javafx.scene.text.Text;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * Dialog for giving a loan.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Jan 05, 2015
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class LoanerGiveLoanDialog extends CloseButtonDialog {
    private final int BUTTON_X = (WIDTH - 124) / 2;
    private final int BUTTON_Y = CLOSE_BTN_Y_POS - 24;
    public static final String DEFAULT_NON_SELECTED_STYLE = "-fx-fill: dimgray; -fx-text-fill: dimgray;";
    public static final String SELECTED_STYLE = "-fx-font-weight: bolder; -fx-fill: black;  -fx-text-fill: black;";


    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    private LoanerEngine loanerEngine;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    @Qualifier("mainScreenXMLImageLoader")
    private XMLImageLoader imageLoader;
    @Autowired
    private Date date;
    @Autowired
    private ClientViewState viewState;

    private LoanerState loaner;
    private final ICityPlayerProxyJFX city;
    @ObjectPropertyType(ILoanRequest.class)
    private ObjectProperty<ILoanRequest> selectedRequest = new SimpleObjectProperty<>(null);
    private VBox box;
    private OpenPatricianLargeWaxButton acceptBtn;
    private DecoratedText loanAgreement;
    @ListType(Double.class)
    private List<Double> interestLevels;
    private IntegerProperty selectedInterestLevel;

    public LoanerGiveLoanDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
    }

    @PostConstruct
    private void initializeDialog() {
        loaner = (LoanerState) loanerEngine.getLoaner(city.getCity());
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.giveLoan", new Object[]{}, locale.getCurrentLocal()));
        String header = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.subHeader", new Object[]{}, locale.getCurrentLocal());
        box = new VBox(5);
        box.setLayoutX(2 * FRAME_BORDER);
        box.setLayoutY(200);
        final DecoratedText subHeading = textFactory.createDecoratedText(header, new HashMap<String, Object>());
        box.getChildren().add(subHeading);
        int index = 0;
        for (final ILoanRequest request : loaner.getRequests()) {
            if (index < 4) {
                final ICitizen citizen = request.getCitizen();
                String rank = translator.getLocalDisplayName(citizen.getRank());
                String s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.request", new Object[]{rank, citizen.getName(), citizen.getLastName(), request.getSum(), request.getDurationInWeeks()}, locale.getCurrentLocal());
                final DecoratedText entry = textFactory.createDecoratedText(s, new HashMap<String, Object>());
                entry.setOnMouseReleased(mouseEvent -> {
                    selectedRequest.setValue(request);
                    deactivateAll();
                    entry.applyStyle(SELECTED_STYLE);
                });
                entry.applyStyle(DEFAULT_NON_SELECTED_STYLE);
                box.getChildren().add(entry);
                index++;
            }
        } // End for requests
        acceptBtn = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.btnLabel", new Object[]{}, locale.getCurrentLocal()));
        acceptBtn.getStyleClass().add("actionButton");
        acceptBtn.setLayoutX(BUTTON_X);
        acceptBtn.setLayoutY(BUTTON_Y);
        acceptBtn.setDisable(true);
        selectedRequest.addListener((observableValue, oldRequest, newRequest) -> {
            if (newRequest != null) {
                acceptBtn.setDisable(false);
            }
        });
        acceptBtn.setOnAction(createAcceptHandlerPage1());
        getContent().addAll(box, acceptBtn);
    }
    private void deactivateAll() {
        ObservableList<Node> children = box.getChildren();
        for (int i = 1; i < children.size(); i++) {
           Node node = children.get(i);
            if (node instanceof DecoratedText) {
                ((DecoratedText)node).applyStyle(DEFAULT_NON_SELECTED_STYLE);
            }
        }
    }

    private EventHandler<MouseEvent> createAcceptHandlerPage1() {
        return mouseEvent -> {
            getContent().remove(box);
            box = new VBox(5);
            box.setLayoutX(2 * FRAME_BORDER);
            box.setLayoutY(200);
            getContent().add(getContent().size() -2, box);

            String header = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.subHeader2", new Object[]{}, locale.getCurrentLocal());
            Text subHeading = new Text();
            subHeading.setText(header);
            subHeading.getStyleClass().add("dialogText");
            StackPane interestSelectionPane = new StackPane();
            Image minus = imageLoader.getImage("icons/minusIcon");
            Image plus = imageLoader.getImage("icons/plusIcon");
            Label reduceInterestLbl = new Label("", new ImageView(minus));
            Label increaseInterestLbl = new Label("", new ImageView(plus));
            interestLevels = new ArrayList<>();
            interestLevels.add(0.011); // 1.1%
            interestLevels.add(0.013); // 1.3%
            interestLevels.add(0.016); // 1.6%
            interestLevels.add(0.019); // 1.9%
            interestLevels.add(0.025); // 2.5%
            selectedInterestLevel = new SimpleIntegerProperty(2);
            final Text interstLbl = new Text();
            interstLbl.getStyleClass().add("dialogText");
            interstLbl.setText(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.interestLevel2", new Object[]{}, locale.getCurrentLocal()));
            BorderPane interestPane = new BorderPane();
            interestPane.setLeft(reduceInterestLbl);
            interestPane.setRight(increaseInterestLbl);
            interestPane.setCenter(interstLbl);
            interestSelectionPane.getChildren().add(interestPane);

            loanAgreement = createLoanAggreement(interestLevels, selectedInterestLevel);


            reduceInterestLbl.setOnMouseReleased(mouseEvent1 -> {
                final int selectedIndex = selectedInterestLevel.get();
                if (selectedIndex > 0) {
                    selectedInterestLevel.setValue(selectedIndex - 1);
                    box.getChildren().remove(loanAgreement);
                    loanAgreement = createLoanAggreement(interestLevels, selectedInterestLevel);
                    box.getChildren().add(loanAgreement);
                    interstLbl.setText(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.interestLevel" + selectedInterestLevel.get(), new Object[]{}, locale.getCurrentLocal()));
                }
            });
            increaseInterestLbl.setOnMouseReleased(mouseEvent1 -> {
                final int selectedIndex = selectedInterestLevel.get();
                if (selectedIndex < interestLevels.size() - 1) {
                    selectedInterestLevel.setValue(selectedIndex + 1);
                    box.getChildren().remove(loanAgreement);
                    loanAgreement = createLoanAggreement(interestLevels, selectedInterestLevel);
                    box.getChildren().add(loanAgreement);
                    interstLbl.setText(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.interestLevel" + selectedInterestLevel.get(), new Object[]{}, locale.getCurrentLocal()));
                }
            });

            box.getChildren().addAll(subHeading, interestSelectionPane, loanAgreement);



            acceptBtn.setText(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernBaseSideRoomPersonDialog.accept", new Object[]{}, locale.getCurrentLocal()));
            acceptBtn.setOnAction(createAcceptHandlerPage2());
        };
    }

    private DecoratedText createLoanAggreement(List<Double> interestLevels, IntegerProperty selectedInterestLevel) {
        final double interestSmall = interestLevels.get(selectedInterestLevel.get());
        String interest = String.format("%.1f", interestSmall * 100);
        final int sum = selectedRequest.get().getSum();
        int paybacksum = (int) ((1 + interestSmall) * sum);

        String s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.loanAgreement", new Object[]{sum, selectedRequest.get().getDurationInWeeks(), interest, paybacksum}, locale.getCurrentLocal());
        return textFactory.createDecoratedText(s, new HashMap<String, Object>());
    }

    private EventHandler<MouseEvent> createAcceptHandlerPage2() {
        return mouseEvent -> {
            final ILoanRequest request = selectedRequest.get();
            final double interest = interestLevels.get(selectedInterestLevel.get()) + 1;
            DateTime now = date.getCurrentDate();
            IDebt debt = Debt.builder()
                    .debitor(request.getCitizen())
                    .creditor(city.getPlayer())
                    .interest(interest)
                    .dueDate(now.plusWeeks(request.getDurationInWeeks()))
                    .amount(request.getSum())
                    .loanTakeOut(now)
                    .build();
            loaner.addDebt(debt);
            loaner.getRequests().remove(request);
            executeOnCloseButtonClicked();

        };
    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_LOANER);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }
}
