package ch.sahits.game.graphic.display.dialog.sea;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.event.data.ShipEntersPortEvent;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.graphic.image.impl.XMLImageLoader;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.engine.land.city.OutriggerService;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.service.ShipService;
import ch.sahits.game.openpatrician.model.ship.ConvoyList;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.ShipFactory;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.event.EventHandler;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.HBox;
import javafx.scene.layout.VBox;
import javafx.scene.text.Text;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.List;

/**
 * This dialog allows creating and dissolving a convoy.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jan 28, 2016
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ConvoyDialog extends CloseButtonDialog {
    private final Logger logger = LogManager.getLogger(getClass());
    /** Reference to the city view model */
    private final ICityPlayerProxyJFX city;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    @Qualifier("mainScreenXMLImageLoader")
    private XMLImageLoader imageLoader;
    @Autowired
    private OutriggerService outriggerService;
    @Autowired
    private ShipFactory shipFactory;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private ShipService shipService;
    @Autowired
    private ConvoyList convoyList;

    public ConvoyDialog(ICityPlayerProxyJFX city) {
        this.city = city;
    }

    @PostConstruct
    private void initializeDialog() {

        INavigableVessel vessel = viewState.getCurrentCityProxy().get().getActiveShip();
        if (vessel instanceof IShip) {
            initializeCreateConvoyDialog((IShip)vessel);
        } else {
            initializeDissolveConvoyDialog((IConvoy)vessel);
        }
    }

    private void initializeCreateConvoyDialog(IShip vessel) {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.createTitle", new Object[]{}, locale.getCurrentLocal()));
        if (canBecomeOrlegShip(vessel)) {
            String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.fullfilledRequirements.part1", new Object[]{vessel.getName()}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
            VBox box = new VBox(text);
            HBox hbox = new HBox(50);
            String s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.createPrivateConvoy", new Object[]{}, locale.getCurrentLocal());
            OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(s);
            action.getStyleClass().add("actionButton");
            action.setId("privateConvoy");
            action.setOnAction(createPrivateConvoy(vessel));
            hbox.getChildren().add(action);
            s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.createPublicConvoy", new Object[]{}, locale.getCurrentLocal());
            action = new OpenPatricianLargeWaxButton(s);
            action.getStyleClass().add("actionButton");
            action.setId("publicConvoy");
            action.setOnAction(createPublicConvoy(vessel));
            hbox.getChildren().add(action);
            box.getChildren().add(hbox);
            List<IConvoy> presentConvoys = convoyList.findConvoy(city.getCity());
            if (presentConvoys.isEmpty()) {
                template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.fullfilledRequirements.part2", new Object[]{vessel.getName()}, locale.getCurrentLocal());
                text = textFactory.createDecoratedText(template, new HashMap<>());
                box.getChildren().add(text);
            } else {
                template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.fullfilledRequirements.part2_2", new Object[]{vessel.getName()}, locale.getCurrentLocal());
                text = textFactory.createDecoratedText(template, new HashMap<>());
                box.getChildren().add(text);
                addConvoyList(box, presentConvoys, vessel);
            }

            box.setLayoutX(50);
            box.setLayoutY(CLOSE_BTN_Y_POS - 500);
            getContent().addAll(box);
        } else {
            String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.unfullfilledRequirements", new Object[]{vessel.getName()}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
            VBox box = new VBox(text);

            GridPane pane1 = new GridPane();
            pane1.setLayoutX(2*FRAME_BORDER);
            pane1.setLayoutY(200);
            pane1.getColumnConstraints().addAll(new ColumnConstraints(100), new ColumnConstraints(50), new ColumnConstraints(50), new ColumnConstraints(50));
            Image sailorIcon = imageLoader.getImage("icons/64/sailor-icon", 32, 32);
            Text captain = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.captain", new Object[]{}, locale.getCurrentLocal()));
            captain.getStyleClass().add("dialogText");
            pane1.add(captain, 0, 0);

            int amountSailors = getAmountSailors();

            Text sailors = new Text(String.valueOf(amountSailors));
            sailors.getStyleClass().add("dialogText");
            HBox hbox = new HBox(sailors, new ImageView(sailorIcon));
            pane1.add(hbox, 1, 0);

            int minStrength = outriggerService.getRequiredWeaponStrength(city.getCity());
            template = messageSource.getMessage("ch.sahits.game.openpatrician.model.ship.impl.Ship.weaponStrengthEven", new Object[]{minStrength}, locale.getCurrentLocal());
            text = textFactory.createDecoratedText(template, new HashMap<>());
            pane1.add(text, 2, 0);


            box.getChildren().add(pane1);

            List<IConvoy> presentConvoys = convoyList.findConvoy(city.getCity());
            if (! presentConvoys.isEmpty()) {
                template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.joinConvoy", new Object[]{vessel.getName()}, locale.getCurrentLocal());
                text = textFactory.createDecoratedText(template, new HashMap<>());
                box.getChildren().add(text);
                addConvoyList(box, presentConvoys, vessel);
            }

            box.setLayoutX(50);
            box.setLayoutY(CLOSE_BTN_Y_POS - 500);

            getContent().addAll(box);
        }
    }

    private void addConvoyList(VBox box, List<IConvoy> presentConvoys, IShip ship) {
        VBox innerBox = new VBox();
        int i = 0;
        for (IConvoy convoy : presentConvoys) {
            final String key = "ch.sahits.game.javafx.service.NoticeBoardShipSelectionMenuProvider.ship";
            String loadedText = messageSource.getMessage(key,
                    new Object[]{convoy.getName(), convoy.getCapacity()}, locale.getCurrentLocal());
            DecoratedText dt = textFactory.createDecoratedText(loadedText, new HashMap<>());
            dt.setId("convoy"+i);
            innerBox.getChildren().add(dt);
            dt.setOnMouseReleased(evt -> {
                convoy.addShip(ship);
                final ICityPlayerProxyJFX proxy = viewState.getCurrentCityProxy().get();
                proxy.leave(ship);
                proxy.getPlayer().removeSelectableVessel(ship);
                executeOnCloseButtonClicked();
            });
            i++;
        }
        innerBox.setLayoutX(50);
        box.getChildren().add(innerBox);
    }

    private EventHandler<MouseEvent> createPublicConvoy(IShip vessel) {
        return event -> {
            IConvoy convoy = shipFactory.getConvoy(vessel, true);
            final ICityPlayerProxyJFX proxy = viewState.getCurrentCityProxy().get();
            proxy.arrive(convoy);
            proxy.leave(vessel);
            proxy.activateShip(convoy);
            convoyList.add(convoy);
            IPlayer player = proxy.getPlayer();
            player.removeSelectableVessel(vessel);
            player.addSelectableVessel(convoy);
            executeOnCloseButtonClicked();
        };
    }

    private EventHandler<MouseEvent> createPrivateConvoy(IShip vessel) {
        return event -> {
          IConvoy convoy = shipFactory.getConvoy(vessel, false);
            final ICityPlayerProxyJFX proxy = viewState.getCurrentCityProxy().get();
            proxy.arrive(convoy);
            proxy.leave(vessel);
            proxy.activateShip(convoy);
            convoyList.add(convoy);
            IPlayer player = proxy.getPlayer();
            player.removeSelectableVessel(vessel);
            player.addSelectableVessel(convoy);
            executeOnCloseButtonClicked();
        };
    }

    private boolean canBecomeOrlegShip(IShip ship) {
        if (ship.getCaptian().isPresent()) {
            if (ship.getNumberOfSailors() >= 20) {
                final ICity city = viewState.getCurrentCityProxy().get().getCity();
                final int requiredWeaponStrength = outriggerService.getRequiredWeaponStrength(city);
                final int shipStrength = shipService.calculateShipsWeaponsStrength(ship);
                if (requiredWeaponStrength <= shipStrength) {
                    return true;
                }
            }
        }
        return false;
    }

    private void initializeDissolveConvoyDialog(IConvoy vessel) {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.dissolveTitle", new Object[]{}, locale.getCurrentLocal()));

        String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.areYouSure", new Object[]{vessel.getName()}, locale.getCurrentLocal());
        DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
        VBox box = new VBox(text);
        final int actionButtonX = (WIDTH - 124) / 2;

        String s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.dissolve", new Object[]{}, locale.getCurrentLocal());
        final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(s);
        action.getStyleClass().add("actionButton");
        action.setId("dissolve");
        ICity city = viewState.getCurrentCityProxy().get().getCity();
        action.setOnAction(Dissolve(vessel, city));
        action.setLayoutX(actionButtonX);
        action.setLayoutY(CLOSE_BTN_Y_POS - 24);

        box.setLayoutX(50);
        box.setLayoutY(CLOSE_BTN_Y_POS - 500);
        getContent().addAll(box, action);

    }

    private EventHandler<MouseEvent> Dissolve(IConvoy vessel, ICity city) {
        return event -> {
            for (IShip ship : vessel.getShips()) {
                IShipOwner owner = ship.getOwner();
                vessel.removeShip(ship);
                if (owner instanceof IHumanPlayer) {
                    DisplayMessage msg = new DisplayMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.convoyDissolveNotification", new Object[]{vessel.getName(), ship.getName(), city.getName()});
                    TargetedEvent targetMsg = new TargetedEvent((IHumanPlayer) owner, msg);
                    clientServerEventBus.post(targetMsg);
                }
                if (owner instanceof IPlayer) {
                    IPlayer player = (IPlayer) owner;
                    player.addSelectableVessel(ship);
                }
                clientServerEventBus.post(new ShipEntersPortEvent(ship, city));
            }
            viewState.getCurrentCityProxy().get().leave(vessel);
            convoyList.remove(vessel);
            IPlayer player = viewState.getCurrentCityProxy().get().getPlayer();
            player.removeSelectableVessel(vessel);
            executeOnCloseButtonClicked();
        };
    }


    private int getAmountSailors() {
        return 20;
    }

    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_SHIP_SELECTION);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }
}
