package ch.sahits.game.graphic.display.dialog.shipyard;

import ch.sahits.game.event.data.ShipyardOrderRepair;
import ch.sahits.game.graphic.display.dialog.util.DialogUtil;
import ch.sahits.game.javafx.control.BarrelAmount;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.model.ControlTableCell;
import ch.sahits.game.javafx.model.StaticTextTableCell;
import ch.sahits.game.javafx.model.Table;
import ch.sahits.game.javafx.model.TableHeader;
import ch.sahits.game.javafx.model.TableRow;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.EventHandler;
import javafx.scene.Group;
import javafx.scene.input.MouseEvent;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.HashMap;

/**
 * Dialog for repairing a ship
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 16, 2013
 */
@Component("shipRepairDialog")
@Scope(value = "prototype")
@ClassCategory({EClassCategory.DIALOG, EClassCategory.SINGLETON_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ShipRepairDialogV2 extends BaseShipyardDialog {
    @ObjectPropertyType(IShip.class)
    private ObjectProperty<IShip> currentShip;
    @Autowired
    private DialogUtil dialogUtil;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private Locale locale;

    @Autowired
    private MessageSource messageSource;

    public ShipRepairDialogV2(final ICityPlayerProxyJFX city) {
        super(city);
        this.currentShip = new SimpleObjectProperty<>(this, "currentShip", null);
        if (city.getActiveShip() instanceof IShip) {
            IShip ship = (IShip) city.getActiveShip();
            currentShip.setValue(ship);
        }
        for (int i = 0; i < shipTypes.length; i++) {
            if (shipTypes[i] == currentShip.get().getShipType()) {
                currentShipTypeIndexProperty().set(i);
                break;
            }
        }
    }
    @PostConstruct
    private void init() {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipRepairDialogV2.title", new Object[]{}, locale.getCurrentLocal()));
        this.enablePreviousNext = dialogUtil.enableShipCatalogueForShips(city);
    }
    @Override
    protected EventHandler<MouseEvent> getAction() {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                city.getPlayersShips().remove(currentShip.get());
                ShipyardOrderRepair event = new ShipyardOrderRepair( getCity().getCityState().getShipyardState(), currentShip.get(), city);
                clientServerEventBus.post(event);
                getPlayer().updateCash(-calculateRepairCosts());
                executeOnCloseButtonClicked();
            }
        };
    }

    @Override
    protected String getActionText() {
        return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipRepairDialogV2.repair", new Object[]{}, locale.getCurrentLocal());
    }

    @Override
    protected boolean hasShips() {
        if (city.getPlayersShips().size() > 0) {
            return city.getPlayersShips().stream().anyMatch(ship -> ship.isAvailable());
        } else {
            return false;
        }
    }

    @Override
    protected Group createFooterText() {
        int costs = calculateRepairCosts();
        Group g = new Group();
        String t = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.totalSum", new Object[]{}, locale.getCurrentLocal());
        HashMap<String, Object> parameters = new HashMap<>();
        parameters.put("price", costs);
        DecoratedText dt = decoratedTextFactory.createDecoratedText(t, parameters);
        g.getChildren().add(dt);
        return g;
    }

    private int calculateRepairCosts() {
        IShipyard state = getCity().getCityState().getShipyardState();
        int damage = (int)Math.rint((100-currentShip.get().getDamage())/100.0);
        int repairTime = state.calculateRepairTime(currentShip.get().getShipType());
        int materialCosts = state.calculateRepairMaterialCosts(currentShip.get(), damage);
        repairTime = repairTime*damage;
        return state.calculateRepairCosts(repairTime, materialCosts);
    }

    @Override
    protected Table createMainTable() {
        Table model = new Table();
        // Add dummy header which is not rendered
        TableHeader header = new TableHeader(1);

        model.setHeader(header);
        model.setColumnWidth(200);
        TableRow row = new TableRow();

        row.add(new StaticTextTableCell(currentShip.get().getName()));
        model.add(row);
        return model;
    }

    @Override
    protected Table createTopTable() {
        Table model = new Table();
        // Add dummy header which is not rendered
        TableHeader header = new TableHeader(4);
        model.setHeader(header);
        model.setColumnWidth(70, 70, 70, 70);
        TableRow row = new TableRow();
        BarrelAmount capacityAmount = new BarrelAmount();
        capacityAmount.setAmount(currentShip.get().getSize());
        row.add(new ControlTableCell(capacityAmount));

        row.add(new StaticTextTableCell(String.valueOf(100 - currentShip.get().getDamage())));
        row.add(new StaticTextTableCell(String.valueOf(currentShip.get().getNumberOfSailors())));
        row.add(new StaticTextTableCell("1"));
        model.add(row);
        return model;
    }

    @Override
    protected BooleanBinding actionEnabledBinding() {
        return new BooleanBinding() {
            {
                super.bind(getPlayer().getCompany().cashProperty());
            }
            @Override
            protected boolean computeValue() {
                return calculateRepairCosts() < getPlayer().getCompany().getCash();
            }
        };
    }

    @Override
    protected EventHandler<MouseEvent> createNextAction() {
        return dialogUtil.createNextActionForShips(city, currentShip);
     }

    @Override
    protected EventHandler<MouseEvent> createPreviousAction() {
        return dialogUtil.createPreviousActionForShips(city, currentShip);
    }
}
