package ch.sahits.game.graphic.display.model;

import ch.sahits.game.event.data.ShipEntersPortEvent;
import ch.sahits.game.event.data.ShipLeavingPort;
import ch.sahits.game.javafx.control.GameStatus;
import ch.sahits.game.javafx.control.MiniMap;
import ch.sahits.game.javafx.control.ViewStatus;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.annotation.UniquePrototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.spring.DependentAnnotationConfigApplicationContext;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.ListBinding;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import lombok.Getter;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.List;

/**
 * State information of the player, the city and the ships.
 * This instance is tied to the ClientViewState.
 */
@ClassCategory({EClassCategory.SINGLETON_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
@UniquePrototype
public class CityPlayerProxyJFX implements ICityPlayerProxyJFX {
    @Getter
    @Setter
	private ICity city;
    @Getter
	private IHumanPlayer player;
	/** List holding all ships owned by the player currently present in the city */
	@ListType(INavigableVessel.class)
	private ObservableList<INavigableVessel> playersVessels = FXCollections.observableArrayList();
	@ListType(IShip.class)
	@XStreamOmitField
	private ListBinding<IShip> playersShips;

    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    /** Reference the active ship of the player */
    @Getter
	private INavigableVessel activeShip=null;
    @Autowired
	@XStreamOmitField
    private DependentAnnotationConfigApplicationContext context;
	/**
	/**
	 * Constructor initializing the city and the player with one ship in port.
	 * @param city
	 * @param player
	 * @param activeShip
	 */
	public CityPlayerProxyJFX(ICity city, IHumanPlayer player, INavigableVessel activeShip) {
        this.city = city;
        this.player = player;
        if (activeShip != null) {
            this.activeShip = activeShip;
            playersVessels.add(activeShip);
        }
	}

    @PostConstruct
    private void register() {
		System.out.println("Initialize CityPlayerProxyJFX");
		clientServerEventBus.register(this);  // FIXME: 1/30/16 make sure this is called when loading
    }
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
    }

	/**
	 * Activate a ship. If the ship is not one of the players or not in port or not available
	 * an {@link IllegalArgumentException} will be thrown
	 * @param ship
	 * @throws IllegalArgumentException if the ship cannot be activated due to its non existence
	 */
	@Override
	public void activateShip(INavigableVessel ship){
//		for (INavigableVessel s : playersVessels) {
//			if ( s.equals(ship) && (!(s instanceof IShip) ||((IShip)s).isAvailable())){
				activeShip= ship;
//				break; // there can only be one
//			}
//		}
//		if (getActiveShip() == null) {
//			throw new IllegalArgumentException("The ship " + ship + " of the player is not in port");
//		}
	}

	@Override
	public ObservableList<INavigableVessel> getPlayersNavalVessels(){
		return playersVessels;
	}
	/**
	 * A ship arrives in the city
	 * @param ship
	 */
	public void arrive(INavigableVessel ship){
		if (player.getFleet().contains(ship)){
			playersVessels.add(ship);
			if (playersVessels.size()==1){
				activeShip=ship;
			}
		}
		if (ship instanceof IConvoy && ((IConvoy)ship).getPlayers().contains(player)) {
			playersVessels.add(ship);
		}
	}
	/**
	 * Ship leave the city
	 * @param ship
	 */
	public void leave(INavigableVessel ship){
        playersVessels.remove(ship);
		if (activeShip==ship){
			activeShip=null;
            if (!playersVessels.isEmpty()) {
                activeShip = playersVessels.get(0);
            }
		}
	}

	@Override
	public ObservableList<IShip> getPlayersShips() {
		if (playersShips == null) {
			playersShips = new ListBinding<IShip>() {
				{
					super.bind(playersVessels);
				}
				@Override
				protected ObservableList<IShip> computeValue() {
					ObservableList<IShip> ships = FXCollections.observableArrayList();
					for (INavigableVessel vessel : playersVessels) {
						 if (vessel instanceof IShip) {
							 ships.add((IShip) vessel);
						 }
					}
					return ships;
				}
			};
		}
		return playersShips;
	}
    @Subscribe
    public void handleShipLeavesPort(ShipLeavingPort event) {
        if (getCity().equals(event.getCity())) {
            leave(event.getShip());
        }
    }
    @Subscribe
    public void handleShipEntersPort(ShipEntersPortEvent event) {
        if (getCity().equals(event.getCity()) ) {
            arrive(event.getShip());
        }
    }

    /**
     * Execute after savegame was loaded triggered from the ClientViewState.
     */
	public void postLoad() {
		List<INavigableVessel> ships = player.findShips(this.city);
		this.playersVessels.clear();
		this.playersVessels.addAll(ships);
		if (!ships.isEmpty()) {
			this.activeShip = playersVessels.get(0);
		} else {
			this.activeShip = null;
		}
        GameStatus gameStatus = context.getBean(GameStatus.class);
        gameStatus.bindToPlayer(player);
        ViewStatus viewStatus = context.getBean(ViewStatus.class);
        viewStatus.setCity(city.getName());
        MiniMap miniMap = context.getBean(MiniMap.class);
        miniMap.initializeMapImage(player);
	}
}
