package ch.sahits.game.graphic.display.notice;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NavigationStateChange;
import ch.sahits.game.event.NoticeBoardClose;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.event.data.ClockTickPostDayChange;
import ch.sahits.game.event.data.NoticeBoardPersistentClose;
import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.graphic.javafx.display.MainGameView;
import ch.sahits.game.javafx.control.NoticeBoard;
import ch.sahits.game.javafx.model.EDialogType;
import ch.sahits.game.javafx.model.ENoticeBoardType;
import ch.sahits.game.javafx.model.NoticeBoardMenu;
import ch.sahits.game.javafx.model.NoticeBoardMenuEntry;
import ch.sahits.game.javafx.service.NoticeBoardMenuProvider;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.dialog.Dialog;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.event.PersonLeavesTavernEvent;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.IContractBroker;
import ch.sahits.game.openpatrician.model.people.IInformant;
import ch.sahits.game.openpatrician.model.people.IPerson;
import ch.sahits.game.openpatrician.model.people.IPirate;
import ch.sahits.game.openpatrician.model.people.ISailors;
import ch.sahits.game.openpatrician.model.people.ISideRoomPerson;
import ch.sahits.game.openpatrician.model.people.ITrader;
import ch.sahits.game.openpatrician.model.people.ITraveler;
import ch.sahits.game.openpatrician.model.people.IWeaponsDealer;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.application.Platform;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.Iterator;

/**
 * Actual implementation of the notice board. The updating of the board
 * happens through the client side asynchronous event bus.
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class OpenPatricianNoticeBoard extends NoticeBoard {
	static final Logger logger = LogManager.getLogger(OpenPatricianNoticeBoard.class);
    private ICityPlayerProxyJFX lastProxy; // FIXME: 12/20/15 this reference that is updated has to go
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    @Autowired
    private ApplicationContext context;
    private MainGameView mainGameView;
    private NoticeBoardMenuProvider menuProvider; // not autowired to avoid premature initialisationn of CitiesState.
    @PostConstruct
    private void initializeEventRegistration() {
        clientServerEventBus.register(this);
        clientEventBus.register(this);
        timerEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
        clientEventBus.unregister(this);
        timerEventBus.unregister(this);
    }

    /**
     * Event handler for updating the notice board
     * @param event object
     */
    @Subscribe
    public void handleNoticeBoardUpdate(NoticeBoardUpdate event) {
        ViewChangeCityPlayerProxyJFX proxy = (ViewChangeCityPlayerProxyJFX) event.getProxy();
        lastProxy = proxy;
        if (proxy.getViewChangeEvent() == EViewChangeEvent.NOTICE_HIDE) {
            clearNoticeBoardOfSceneTiedMenu(proxy);
        } else {
            updateNoticeBoard(proxy, map((EViewChangeEvent) ((ViewChangeCityPlayerProxyJFX) event.getProxy()).getViewChangeEvent()));
        }
    }
    @Subscribe
    public void handleNoticeBoardClosePersistent(NoticeBoardPersistentClose event) {
        boolean needsUpdate = false;
        switch (event.getBoardToBeClosed()) {
            case MESSAGES:
                for (Iterator<NoticeBoardMenu> iterator = noticeBoardMenu.iterator(); iterator.hasNext(); ) {
                    NoticeBoardMenu menu = iterator.next();
                    if (menu.getType() == ENoticeBoardType.MESSAGE) {
                        iterator.remove();
                        needsUpdate = true;
                        break;
                    }
                }
                break;
            default:
                throw new IllegalArgumentException("The persistent closing for "+event.getBoardToBeClosed()+" is not implemented");
        }
        ViewChangeCityPlayerProxyJFX proxy = (ViewChangeCityPlayerProxyJFX) event.getProxy();

        updateAfterStackClearing(proxy, needsUpdate);
    }
    @Subscribe
    public void updateDisplayMessages(DisplayMessage message) {
        if (lastProxy instanceof ViewChangeCityPlayerProxyJFX) {
            EViewChangeEvent viewChange = (EViewChangeEvent) ((ViewChangeCityPlayerProxyJFX) lastProxy).getViewChangeEvent();
            if (viewChange == EViewChangeEvent.MESSAGES) {
                Platform.runLater(() -> {
                    updateNoticeBoard(((ViewChangeCityPlayerProxyJFX) lastProxy), ENoticeBoardType.MESSAGE);
                });
            }
        }
    }
    @Subscribe
    public void handleNavigationChange(NavigationStateChange event) {
        switch (event.getChange()) {
            case DISABLE_NAVIGATION:
                disableNavigationItems();
                break;
            case ENABLE_NAVIGATION:
                enableNavigationItems();
                break;
            default:
                logger.info("Not interested in navigation change event: "+event.getChange());
        }
    }

    private void disableNavigationItems() {
        noticeBoardMenu.getLast().disableAll();
    }

    private void enableNavigationItems() {
        noticeBoardMenu.getLast().enableAll();
    }

    private void updateNoticeBoard(ViewChangeCityPlayerProxyJFX proxy, ENoticeBoardType newType) {
        if (menuProvider == null) {
            initializeMenuProvider();
        }
        if (noticeBoardMenu.isEmpty()) {
            NoticeBoardMenu newMenu = menuProvider.createMenu(newType, proxy);
            Platform.runLater(() -> addMenu(newMenu));
        } else {
            ENoticeBoardType type = noticeBoardMenu.getLast().getType();
            if (type == newType) {

                Platform.runLater(() -> {
                    NoticeBoardMenu updatedMenu = menuProvider.createMenu(type, proxy);
                    Dialog dialog = mainGameView.getDialog();
                    if (dialog != null) {
                        EDialogType dialogType = dialog.getDialogType();
                        for (NoticeBoardMenuEntry entry : updatedMenu.getMenuEntries()) {
                            if (entry.getDialogType() != null && entry.getDialogType() == dialogType) {
                                updatedMenu.deselctAll();
                                entry.setSelected(true);
                                break;
                            }
                        }
                    }
                    reset(updatedMenu);
                });
            } else {
                // switched to a different board
                NoticeBoardMenu newMenu = menuProvider.createMenu(newType, proxy);
                if (type.isTiedToScene() && newType.isTiedToScene()) {
                    // replace
                    closeNoticeBoard();
                    addMenu(newMenu);
                } else {
                    addMenu(newMenu);
                }
            }
        }
    }

    private void initializeMenuProvider() {
        menuProvider = context.getBean(NoticeBoardMenuProvider.class);
        mainGameView = context.getBean(MainGameView.class);
    }

    /**
     * Update the notice board when a person leaves
     * @param event
     */
    @Subscribe
    public void handlePersonLeaves(PersonLeavesTavernEvent event) {
        // last proxy might be null if the tavern dialog is not open, which is correct
        synchronized (noticeBoardMenu){
            if (!noticeBoardMenu.isEmpty() && lastProxy.getCity().equals(event.getCity())) {
                ENoticeBoardType type = noticeBoardMenu.getLast().getType();
                if (lastProxy != null && lastProxy instanceof ViewChangeCityPlayerProxyJFX && type == ENoticeBoardType.TAVERN) {
                    if (mainGameView != null) {
                        Dialog dialog = mainGameView.getDialog();
                        final EDialogType mappedDialogType = map(event.getPerson());
                        if (dialog != null && dialog.getDialogType() == mappedDialogType) {
                            ((CloseButtonDialog)dialog).executeOnCloseButtonClicked();
                        }
                    }
                    updateNoticeBoard((ViewChangeCityPlayerProxyJFX) lastProxy, type);
                }
            }
        }
    }

    private EDialogType map(IPerson person) {
        if (person instanceof ISideRoomPerson) {
            return EDialogType.TAVERN_SIDE_ROOM;
        }
        if (person instanceof ICaptain) {
            return EDialogType.TAVERN_CAPTAIN;
        }
        if (person instanceof ISailors) {
            return EDialogType.TAVERN_SAILORS;
        }
        if (person instanceof IWeaponsDealer) {
            return EDialogType.TAVERN_WEAPONS_DEALER;
        }
        if (person instanceof ITraveler) {
            return EDialogType.TAVERN_TRAVELER;
        }
        if (person instanceof ITrader) {
            return EDialogType.TAVERN_TRADER;
        }
        if (person instanceof IContractBroker) {
            return null;
        }
        if (person instanceof IPirate) {
            return EDialogType.TAVERN_PIRATE;
        }
        if (person instanceof IInformant) {
            return EDialogType.TAVERN_INFORMANT_1;
        }
        return null;
    }

    /**
     * Event handler for removing the content of the notice board.
     * @param event object
     */
    @Subscribe
    public void handleCloseNoticeBoard(NoticeBoardClose event) {
        closeNoticeBoard();
    }

    /**
     * Handle the day change event to update the notice board.
     * Actually this happens on the tick after the day change to ensure that all events
     * that happen on the day change are accounted for and executed.
     * @param event
     */
    @Subscribe
    public void handleDayChange(ClockTickPostDayChange event) {
        if (!noticeBoardMenu.isEmpty()) {
            ENoticeBoardType type = noticeBoardMenu.getLast().getType();
            if (lastProxy != null && type == ENoticeBoardType.TAVERN) {
                Platform.runLater(() -> updateNoticeBoard((ViewChangeCityPlayerProxyJFX) lastProxy, type));
            }
        }
    }

    /**
     * Clean out all menu entries that are tied to a scene.
     * @param proxy
     */
    private void clearNoticeBoardOfSceneTiedMenu(ViewChangeCityPlayerProxyJFX proxy) {
        boolean needsUpdate = false;
        for (Iterator<NoticeBoardMenu> iterator = noticeBoardMenu.iterator(); iterator.hasNext(); ) {
            NoticeBoardMenu menu = iterator.next();
            if (menu.getType().isTiedToScene()) {
                iterator.remove();
                needsUpdate = true;
            }
        }
        updateAfterStackClearing(proxy, needsUpdate);
    }

    /**
     * Update the view of the notice board after the menu stack was cleared.
     * @param proxy
     * @param needsUpdate
     */
    private void updateAfterStackClearing(ViewChangeCityPlayerProxyJFX proxy, boolean needsUpdate) {
        if (needsUpdate) {
            if (noticeBoardMenu.isEmpty()) {
                closeNoticeBoard();
            } else {
                ENoticeBoardType type = noticeBoardMenu.getLast().getType();
                Platform.runLater(() -> updateNoticeBoard(proxy, type));
            }
        }
    }

    /**
	 * Map the view change event to a notice board destinction. If the view change does not
	 * result in a view change null will be returned.
	 * @param viewChange
	 * @return
	 */
	private ENoticeBoardType map(EViewChangeEvent viewChange) {
		switch (viewChange) {
		case NOTICE_MARKET_BOOTH:
			return ENoticeBoardType.MARKET_PLACE;
		case NOTICE_SHIPYARD:
			return ENoticeBoardType.SHIPYARD;
		case NOTICE_TAVERN:
			return ENoticeBoardType.TAVERN;
		case NOTICE_TRADE:
			return ENoticeBoardType.TRADING;
		case NOTICE_TRADING_OFFICE:
			return ENoticeBoardType.TRADING_OFFICE;
        case NOTICE_LOANER:
            return ENoticeBoardType.LOANER;
        case MESSAGES:
            return ENoticeBoardType.MESSAGE;
        case NOTICE_CITY_HALL_BOARD:
            return ENoticeBoardType.CITY_HALL_NOTICE_BOARD;
        case NOTICE_CITY_HALL_TREASURY:
            return ENoticeBoardType.CITY_HALL_MAYORS_OFFICE;
        case NOTICE_CITY_HALL_ALDERMAN:
            return ENoticeBoardType.CITY_HALL_ALDERMAN_OFFICE;
        case NOTICE_CITY_HALL_MEETINGROOM:
            return ENoticeBoardType.CITY_HALL_MEETINGROOM;
        case MAIN_VIEW_CHURCH:
            return ENoticeBoardType.CHURCH;
        case NOTICE_SHIP_SELECTION:
            return ENoticeBoardType.SHIP_SELECTION;
		default:
			return null;
		}
	}
}
