package ch.sahits.game.graphic.javafx.display;

import ch.sahits.game.event.data.ShipArrivesAtDestinationEvent;
import ch.sahits.game.event.data.ShipEntersPortEvent;
import ch.sahits.game.event.data.ShipLeavingPort;
import ch.sahits.game.event.data.ShipNearingPortEvent;
import ch.sahits.game.event.data.ShipPositionUpdateEvent;
import ch.sahits.game.event.data.SwitchCity;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.graphic.image.ImageUtil;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.UniquePrototype;
import ch.sahits.game.openpatrician.clientserverinterface.model.PathInterpolatorMap;
import ch.sahits.game.openpatrician.engine.sea.AStarGraphProvider;
import ch.sahits.game.openpatrician.engine.sea.IPathConverter;
import ch.sahits.game.openpatrician.engine.sea.SeafaringService;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IMap;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.sea.TravellingVessel;
import ch.sahits.game.openpatrician.model.sea.TravellingVessels;
import ch.sahits.game.openpatrician.model.service.ShipService;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.animation.Animation;
import javafx.animation.KeyFrame;
import javafx.animation.Timeline;
import javafx.application.Platform;
import javafx.event.ActionEvent;
import javafx.geometry.Point2D;
import javafx.scene.Node;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.Pane;
import javafx.scene.paint.Color;
import javafx.scene.shape.Circle;
import javafx.scene.shape.CubicCurveTo;
import javafx.scene.shape.MoveTo;
import javafx.scene.shape.Path;
import javafx.scene.shape.PathElement;
import javafx.scene.shape.Rectangle;
import javafx.scene.shape.Shape;
import javafx.util.Duration;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 26, 2015
 */
@ClassCategory({EClassCategory.JAVAFX, EClassCategory.UNRELEVANT_FOR_DESERIALISATION, EClassCategory.SINGLETON_BEAN})
@UniquePrototype
public class SeamapImageView extends BaseMainGameImageView {

    private final Logger logger = LogManager.getLogger(getClass());

    private final ImageView imgView;
    private final Pane shipCanvas;  // draw the ships here
    private final Rectangle clip;
    private final Rectangle scrollLeft;
    private final Rectangle scrollRight;
    private final Timeline slideLeftAnimation;
    private final Timeline slideRightAnimation;

    private double scale;
    @Autowired
    private IMap map;
    @Autowired
    private SeafaringService seafaringService;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    private AStarGraphProvider aStarGraphService;
    @Autowired
    private IPathConverter pathConverter;
    @Autowired
    private TravellingVessels vessels;
    @Autowired
    private PathInterpolatorMap interpolators;
    @Autowired
    @Qualifier("mainScreenXMLImageLoader")
    private IDataImageLoader imageLoader;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("clientEventBus")
    protected AsyncEventBus clientEventBus;
    @Autowired
    private TimedUpdatableTaskList taskList;
    @Autowired
    private Date date;
    @Autowired
    private ShipService shipService;

    private Image mapImage;
    private final Rectangle slaveClip;
    private Point2D focus;

    private boolean displayed = false;

    public SeamapImageView(Image mapImage, double width, double height, Point2D focus, double scale) {
        super();
        imgView = new ImageView();
        shipCanvas = new Pane();
        clip = new Rectangle(0, 0, width, height);
        Color slideColor = new Color(1, 1, 1, 0.1);
        slaveClip = new Rectangle(0, 0, width, height);
        scrollLeft = new Rectangle(0, 0, 50, height);
        scrollLeft.setFill(slideColor);
        scrollRight = new Rectangle(width - 50, 0, 50, height);
        scrollRight.setFill(slideColor);
        slideLeftAnimation = new Timeline();
        slideLeftAnimation.setCycleCount(Animation.INDEFINITE);
        KeyFrame kf1 = new KeyFrame(Duration.millis(50),
                event -> {
                    resetClipXPosition(clip.getX() - 5);
                });
        slideLeftAnimation.getKeyFrames().add(kf1);
        slideRightAnimation = new Timeline();
        slideRightAnimation.setCycleCount(Animation.INDEFINITE);
        KeyFrame kf2 = new KeyFrame(Duration.millis(50),
                event -> {
                    resetClipXPosition(clip.getX() + 5);
                });
        slideRightAnimation.getKeyFrames().add(kf2);

        this.mapImage = mapImage;
        this.focus = focus;
        this.scale = scale;
    }
    @PostConstruct
    private void init() {
        clip.xProperty().addListener(
                (observable, oldValue, newValue) -> {
                    slaveClip.setX(newValue.doubleValue());
                    if (scrolledAllToTheLeft()) {
                        slideLeftAnimation.stop();
                        getChildren().remove(scrollLeft);
                    }
                    if (scrolledAllToTheRight()) {
                        slideRightAnimation.stop();
                        getChildren().remove(scrollRight);
                    }
                }
        );
        clip.yProperty().addListener(
                (observable, oldValue, newValue) -> slaveClip.setY(newValue.doubleValue()));
        clip.widthProperty().addListener((observable, oldValue, newValue) -> {
            slaveClip.setWidth(newValue.doubleValue());
            scrollRight.setX(newValue.doubleValue() - scrollRight.getWidth());
        });
        clip.heightProperty().addListener((observable, oldValue, newValue) -> {
            slaveClip.setHeight(newValue.doubleValue());
            scrollLeft.setHeight(newValue.doubleValue());
            scrollRight.setHeight(newValue.doubleValue());
        });
        resetImage(mapImage, slaveClip.getWidth(), slaveClip.getHeight(), scale);
        imgView.setClip(clip);
        shipCanvas.setClip(slaveClip);
        shipCanvas.setMouseTransparent(true);

        focusOnPoint(focus);

        imgView.addEventHandler(MouseEvent.MOUSE_RELEASED, this::handleMouseClick);


        scrollLeft.setOnMouseEntered(evt -> {
            if (scrolledAllToTheRight()&& !getChildren().contains(scrollRight)) {
                getChildren().add(scrollRight);
            }
            slideLeftAnimation.play();
        });
        scrollLeft.setOnMouseExited(evt -> {
            slideLeftAnimation.stop();
        });
        scrollRight.setOnMouseEntered(evt -> {
            if (scrolledAllToTheLeft() && !getChildren().contains(scrollLeft)) {
                getChildren().add(scrollLeft);
            }
            slideRightAnimation.play();
        });
        scrollRight.setOnMouseExited(evt -> {
            slideRightAnimation.stop();
        });

        if (!scrolledAllToTheLeft()) {
            getChildren().add(scrollLeft);
        }
        if (!scrolledAllToTheRight()) {
            getChildren().add(scrollRight);
        }
        clientServerEventBus.register(this);
    }
    @PreDestroy
    void unregister() {
        clientServerEventBus.unregister(this);
    }

    public void setDisplayed(boolean displayed) {
        this.displayed = displayed;
    }

    private void handleMouseClick(MouseEvent evt) {
        boolean travelTo = evt.getButton().equals(MouseButton.PRIMARY);
        double unscaledX = Math.round(evt.getX()/scale);
        double unscaledY = Math.round(evt.getY()/scale);
        ICity city = findCity(unscaledX, unscaledY);
        if (travelTo) {
            travelTo = viewState.getCurrentCityProxy().isPresent();
            if (travelTo) {
                if (viewState.getCurrentCityProxy().get().getCity().equals(city)) {
                    travelTo = false;
                } else {
                    travelTo = viewState.getCurrentCityProxy().get().getActiveShip() != null;
                }
            }
        }


        if (travelTo) {
            INavigableVessel vessel = viewState.getCurrentCityProxy().get().getActiveShip();
            if (vessel instanceof IConvoy && ((IConvoy)vessel).isPublicConvoy()) {
                TimedTask task = new TimedTask() {
                    {
                        setExecutionTime(date.getCurrentDate().plusDays(5));
                    }
                    @Override
                    public void run() {
                        travelToDestination(evt.getX(), evt.getY(), city, vessel);
                    }
                };
                taskList.add(task);
            } else {
                travelToDestination(evt.getX(), evt.getY(), city, vessel);
            }
        } else if (city != null) {
            List<ITradingOffice> office = city.findBuilding(ITradingOffice.class, Optional.of(viewState.getPlayer()));
            List<IShip> ships = viewState.getPlayer().getFleet();
            if (!office.isEmpty()) {
                switchToCity(city);
            } else {
                for (IShip ship : ships) {
                    if (ship.getLocation().equals(city.getCoordinates())) {
                        switchToCity(city);
                        break;
                    }
                }
            }
        } else {
            double x = evt.getX();
            double y = evt.getY();
            for (Node node : shipCanvas.getChildren()) {
                if (node instanceof ImageView) {
                    if (node.getBoundsInParent().contains(x, y)) {
                        String id = node.getId();
                        List<INavigableVessel> fleet = viewState.getPlayer().getSelectableVessels();
                        Optional<INavigableVessel> vessel = shipService.findShipByUuid(fleet, id);
                        if (vessel.isPresent()) {
                            viewState.getCurrentCityProxy().get().activateShip(vessel.get());
                        }
                    }
                }
            }
        }
    }

    private void travelToDestination(double x, double y, ICity destinationCity, INavigableVessel vessel) {
        double unscaledX = Math.round(x/scale);
        double unscaledY = Math.round(y/scale);
        ICity sourceCity = findCity(vessel.getLocation().getX(), vessel.getLocation().getY());
        List<Point2D> path = new ArrayList<>();
        if (destinationCity != null) {
            path = seafaringService.travelTo(vessel, destinationCity.getCoordinates());
        } else {
            final Point2D destination = new Point2D(unscaledX, unscaledY);
            if (aStarGraphService.isOnSea(destination)) {
                path = seafaringService.travelTo(vessel, destination);
            }
        }
        drawPath(vessel, path);
        if (destinationCity != null) {
            interpolators.get(vessel).setDestinationCity(true);
        } else {
            interpolators.get(vessel).setDestinationCity(false);
        }
        if (sourceCity != null) {
            interpolators.get(vessel).setSourceCity(true);
        } else {
            interpolators.get(vessel).setSourceCity(false);
        }
        vessels.getTravellingVessel(vessel).setDisplayVessel(true);
    }

    /**
     * Calculate a new path for the <code>vessel</code> and draw it on the map.
     * @param vessel that travels along the path
     * @param path series of points defining the path.
     */
    private void drawPath(INavigableVessel vessel, List<Point2D> path) {
        if (path != null) {
            clearLines();
            Path p = pathConverter.createPath(vessel, path, scale);
            drawPathOnMap(vessel, path, p);
        }
    }

    /**
     * Draw the path onto the map considering the proper scaling.
     * @param vessel for which the path is drawn
     * @param path series of points that define the corners of the path.
     * @param p path that is drawn.
     */
    private void drawPathOnMap(INavigableVessel vessel, List<Point2D> path, Path p) {
        shipCanvas.getChildren().add(p);
        Image shipIcon = getShipIcon(vessel);
        int scaleXFactor = 1;
        if (path.get(0).getX() < path.get(path.size() - 1).getX()) {
            // travel eastwards
            scaleXFactor = -1;
        }
        ImageView view = new ImageView(shipIcon);
        view.addEventFilter(ActionEvent.ANY, event -> {
            logger.debug("Activate ship {}", vessel.getName());
            viewState.getCurrentCityProxy().get().activateShip(vessel);
        });
        view.setScaleX(0.5 * scaleXFactor);
        view.setScaleY(0.5);
        view.setId(vessel.getUuid());
        updateShipPosition(vessel, shipIcon, view);
        ImageView oldView = findImageView(vessel);
        if (oldView != null) {
            shipCanvas.getChildren().remove(oldView);
        }
        shipCanvas.getChildren().add(view);
    }

    /**
     * Draw the path and ship icon for the vessel onto the map.
     * @param vessel for which the drawing should be done.
     */
    private void drawExistingPath(TravellingVessel vessel) {
        INavigableVessel ship = vessel.getVessel();
        Path path = vessel.getDrwawablePath();
        List<Point2D> pointedPath = vessel.getCalculatablePath();
        clearLines();
        drawPathOnMap(ship, pointedPath, path);
    }

    private void updateShipPosition(INavigableVessel vessel, Image shipIcon, ImageView view) {
        double x = vessel.getLocation().getX() * scale;
        double y = vessel.getLocation().getY() * scale;
        Platform.runLater(() -> {
            view.setLayoutX(x - shipIcon.getWidth() / 2);
            view.setLayoutY(y - shipIcon.getHeight() / 2);
            logger.trace("Update ship {} position to {},{}", vessel.getName(), view.getLayoutX(), view.getLayoutY());
        });
    }

    private ImageView findImageView(INavigableVessel vessel) {
        String id = vessel.getUuid();
        for (Node node : shipCanvas.getChildren()) {
            if (node instanceof ImageView && node.getId().equals(id)) {
                return (ImageView) node;
            }
        }
        return null;
    }

    private Image getShipIcon(INavigableVessel vessel) {
        if (vessel instanceof IShip) {
            switch (((IShip) vessel).getShipType()) {
                case COG:
                    return imageLoader.getImage("icons/64/cog_icon");
                case CRAYER:
                    return imageLoader.getImage("icons/64/crayer_icon");
                case HOLK:
                    return imageLoader.getImage("icons/64/holk_icon");
                case SNAIKKA:
                    return imageLoader.getImage("icons/64/schnikka_icon");
            }
        } else {
            IConvoy convoy = (IConvoy) vessel;
            return getShipIcon(convoy.getOrlegShip());
        }
        return null;
    }

    private void clearLines() {
        for (Iterator<Node> iterator = shipCanvas.getChildren().iterator(); iterator.hasNext(); ) {
            Node node = iterator.next();
            if (node instanceof Shape) {
                iterator.remove();
            }
        }
    }

    private void switchToCity(ICity city) {
        clientEventBus.post(new SwitchCity(city));
    }

    private ICity findCity(double unscaledX, double unscaledY) {
        Point2D p = new Point2D(unscaledX, unscaledY);
        for (ICity city : map.getCities()) {
            double distance = p.distance(city.getCoordinates());
            if (distance <= ImageUtil.CITY_RADIUS) {
                return city;
            }
        }
        return null;
    }

    private boolean scrolledAllToTheLeft() {
        return clip.getX() <= 0.0;
    }
    private boolean scrolledAllToTheRight() {
        return clip.getX() >= imgView.getImage().getWidth() - clip.getWidth();
    }


    private void focusOnPoint(Point2D focus) {
        double totalwidth = imgView.getImage().getWidth();
        double focusX = focus.getX();
        double clipX = clip.getX();
        double clipWidth = clip.getWidth();
        double centerX = clipX + clipWidth/2;
        if (clipX == 0.0 && focusX < centerX) {
            // we are all to the left
        } else if (clipX + clipWidth >= totalwidth) {
            // we are all to the right
        } else {
            double x = Math.min(focusX - clipWidth/2, totalwidth - clipWidth);
            resetClipXPosition(x);
        }
    }

    private void resetClipXPosition(double x) {
        if (x < 0) {
            x = 0;
        }
        if (x > imgView.getImage().getWidth()) {
            x = imgView.getImage().getWidth();
        }
        imgView.setLayoutX(-x);
        shipCanvas.setLayoutX(-x);
        clip.setX(x);
    }

    /**
     * Reset the image to accomodate the dimensions.
     * @param mapImage
     * @param width
     * @param height
     */
    public void resetImage(Image mapImage, double width, double height, double scale) {
        double oldScale = this.scale;
        this.scale = scale;
       imgView.setImage(mapImage);
        shipCanvas.setMaxWidth(mapImage.getWidth());
        shipCanvas.setMinWidth(mapImage.getWidth());
        shipCanvas.setMaxHeight(mapImage.getHeight());
        shipCanvas.setMinHeight(mapImage.getHeight());

        getChildren().removeAll(imgView, shipCanvas);
        double oldWidth = clip.getWidth();
        boolean widthChange = oldWidth != width;
        boolean heightChange = clip.getHeight() != height;

        double x = Math.max((width - mapImage.getWidth())/2,0);
        double y = Math.max((height - mapImage.getHeight())/2,0);
        if (!heightChange) {
            imgView.setLayoutX(x);
            shipCanvas.setLayoutX(x);
        }
        imgView.setLayoutY(y);
        shipCanvas.setLayoutY(y);
        double xx = x;
        if (widthChange && width < mapImage.getWidth()) {
            double focuspoint = clip.getX() + oldWidth/2;
            xx = Math.max(0, focuspoint - width/2);
            resetClipXPosition(xx);
        }
        for (Iterator<Node> iterator = shipCanvas.getChildren().iterator(); iterator.hasNext(); ) {
            Node node = iterator.next();
            if (node instanceof Circle) {
              iterator.remove();
            }
        }

        drawShipsInCities(mapImage, scale, xx);

        for (INavigableVessel vessel : vessels) {
            TravellingVessel traveling = vessels.getTravellingVessel(vessel);
            if (traveling.getDisplayVessel()) {
                drawExistingPath(traveling);
            }
        }

        redrawPath(oldScale);
        clip.setWidth(width);
        clip.setHeight(height);
        getChildren().add(0, shipCanvas);
        getChildren().add(0, imgView);
        scrollRight.setX(width - scrollRight.getWidth());



    }

    private void drawShipsInCities(Image mapImage, double scale, double x) {
        for (ICity city : map.getCities()) {
            if (city.getCoordinates().getX() >= x && city.getCoordinates().getX() <= x + mapImage.getWidth()) {
                List<INavigableVessel> ships = viewState.getPlayer().findShips(city);
                if (!ships.isEmpty()) {
                    drawShipPresenceInCity(city);
                }
            }
        }
    }

    private void drawShipPresenceInCity(ICity city) {
        int radius = 8;
        int cityX = (int) Math.rint(city.getCoordinates().getX());
        int cityY = (int) Math.rint(city.getCoordinates().getY());
        Circle c = new Circle(cityX * scale, cityY * scale, radius * scale, Color.WHITE);
        shipCanvas.getChildren().add(c);
    }

    @Subscribe
    public void handleShipLeavesCity(ShipLeavingPort event) {
        ICity city = event.getCity();
        final IShipOwner owner = event.getShip().getOwner();
        if (owner instanceof IHumanPlayer && owner.equals(viewState.getPlayer())) {
            List<INavigableVessel> ships = ((IHumanPlayer)owner).findShips(city);
            if (ships.isEmpty()) {
                double cityX = (int) Math.rint(city.getCoordinates().getX()) * scale;
                double cityY = (int) Math.rint(city.getCoordinates().getY()) * scale;
                for (Iterator<Node> iterator = shipCanvas.getChildren().iterator(); iterator.hasNext(); ) {
                    Node node = iterator.next();
                    if (node instanceof Circle) {
                        Circle c = (Circle) node;
                        if (c.getCenterX() == cityX && c.getCenterY() == cityY) {
                            Platform.runLater(() -> {
                                iterator.remove();
                            });
                            break;
                        }
                    }
                }
            }
        }
    }



    private void redrawPath(double oldScale) {
        double scaleCorrection = 1/oldScale;
        double scale = this.scale * scaleCorrection;
        logger.debug("Rescale path from {} to {}, which is a correction of {}", oldScale, this.scale, scale);
        for (Node node : shipCanvas.getChildren()) {
            if (node instanceof Path) {
                for (PathElement pathElement : ((Path) node).getElements()) {
                    rescale(pathElement, scale);
                }
            }
        }
    }

    private void rescale(MoveTo pathElement, double scale) {
         double value = pathElement.getX() * scale;
        pathElement.setX(value);
        value = pathElement.getY() * scale;
        pathElement.setY(value);
    }
    private void rescale(CubicCurveTo pathElement, double scale) {
        double value = pathElement.getX() * scale;
        pathElement.setX(value);
        value = pathElement.getY() * scale;
        pathElement.setY(value);
        value = pathElement.getControlX1() * scale;
        pathElement.setControlX1(value);
        value = pathElement.getControlY1() * scale;
        pathElement.setControlY1(value);
        value = pathElement.getControlX2() * scale;
        pathElement.setControlX2(value);
        value = pathElement.getControlY2() * scale;
        pathElement.setControlY2(value);
    }
    private void rescale(PathElement pathElement, double scale) {
        if (pathElement instanceof MoveTo) {
            rescale((MoveTo)pathElement, scale);
        } else if (pathElement instanceof CubicCurveTo) {
            rescale((CubicCurveTo) pathElement, scale);
        } else {
            throw new IllegalStateException("Rescaling for " + pathElement.getClass().getName() + " is not implemented");
        }
    }
    @Subscribe
    public void handleShipPositionUpdate(ShipPositionUpdateEvent event) {
      INavigableVessel vessel = event.getShip();
        TravellingVessel ship = vessels.getTravellingVessel(vessel);
        if (ship.getDisplayVessel()) {
            ImageView icon = findImageView(vessel);
            if (icon != null) {
                Image shipIcon = icon.getImage();
                updateShipPosition(vessel, shipIcon, icon);
            } else {
                logger.debug("Failed to find vessel with uuid {}", vessel.getUuid());
            }
        }
    }
    @Subscribe
    public void handleShipReachesDestination(ShipArrivesAtDestinationEvent event) {
        INavigableVessel vessel = event.getShip();
        TravellingVessel ship = vessels.getTravellingVessel(vessel);
        Platform.runLater(() -> shipCanvas.getChildren().remove(ship.getDrwawablePath()));
        vessels.remove(vessel);
    }
    @Subscribe
    public void handleShipNearsPort(ShipNearingPortEvent event) {
        INavigableVessel vessel = event.getShip();
        TravellingVessel ship = vessels.getTravellingVessel(vessel);
        if (ship.getDisplayVessel()) {
            ImageView icon = findImageView(vessel);
            if (icon != null) {
                Platform.runLater(() -> shipCanvas.getChildren().remove(icon));
                ship.setDisplayVessel(false);
            }
        }
    }
    @Subscribe
    public void handeShipReachesPort(ShipEntersPortEvent event) {
        INavigableVessel vessel = event.getShip();
        TravellingVessel ship = vessels.getTravellingVessel(vessel);
        ICity city = event.getCity();
        final IShipOwner owner = event.getShip().getOwner();
        Platform.runLater(() -> {
                shipCanvas.getChildren().remove(ship.getDrwawablePath());
                if (owner instanceof IHumanPlayer && owner.equals(viewState.getPlayer())) {
                    List<INavigableVessel> ships = ((IHumanPlayer)owner).findShips(city);
                    if (ships.size() == 1) {
                        drawShipPresenceInCity(city);
                    }
                }
            });
            vessels.remove(vessel);
    }

    /**
     * Remove all ship icons from the sea map.
     */
    public void removeShipIcons() {
        for (Iterator<Node> iterator = shipCanvas.getChildren().iterator(); iterator.hasNext(); ) {
            Node node = iterator.next();
            if (node instanceof ImageView) {
                iterator.remove();
            }
        }
    }
}
