package ch.sahits.game;

import ch.sahits.game.graphic.display.OpenPaticianApplicationWindow;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.plugin.BasePluginDefinition;
import ch.sahits.game.openpatrician.plugin.PluginConfiguration;
import ch.sahits.game.openpatrician.server.ServerLauncher;
import ch.sahits.game.openpatrician.spring.StandaloneConfiguration;
import ch.sahits.game.openpatrician.util.LocalStorage;
import ch.sahits.game.openpatrician.util.UnzipUtility;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.RollingFileAppender;

import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import java.io.File;
import java.io.IOException;
import java.util.Properties;

/**
 * Main class for starting the OpenPatrician game.
 * <br>
 * The following system properties are evaluated:
 * <ul>
 *     <li>javafx.polygon.shading: The polygons that are used to define an area are colored</li>
 * </ul>
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Dec 16, 2011
 *
 */
@ClassCategory(EClassCategory.STARTUP)
public class OpenPatrician {
	
	private static Logger logger = LogManager.getLogger(OpenPatrician.class);
	
//	private final static String START_CLIENT = "-c";
//	private final static String START_SERVER = "-s";
//	private final static String HELP = "?";
	

	private volatile static boolean startedServer = false;
	
	private final static Object lock = new Object();

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		String jreVersion = (String) System.getProperties().get("java.version");
		if (!jreVersion.startsWith("1.8")) {
			logger.error("JRE must be of version 1.8");
			System.out.println("JRE must be of version 1.8");
			System.exit(1);
		}
		logEnvironment();
		CommandLineArguments cmdHelper = new CommandLineArguments();
		Options opts = cmdHelper.createCommandLineOptions();
		CommandLine cmdLine = cmdHelper.parseCommandLine(opts, args);
		if (cmdLine.hasOption(CommandLineArguments.HELP_OPTION)){
			cmdHelper.printHelp(opts);
			System.exit(0);
		}
		if (cmdLine.hasOption(CommandLineArguments.VERSION_OPTION)) {
			System.out.println("OpenPatrician version: "+OpenPatrician.class.getPackage().getImplementationVersion());
			System.exit(0);
		}
		cmdHelper.persistAsPropertyFile(cmdLine);

		unpackPlugins();

		if (cmdHelper.isServer(cmdLine)){ //if (cmdHelper.isClient(cmdLine) || cmdHelper.isServer(cmdLine)) ??
			logger.error("Multiplayer game is not yet implemented, sorry.");
			System.exit(0);
		} else {
            if (cmdHelper.isStandalone(cmdLine)) {
                // also need to create the server
                 ServerLauncher.initializeApplicationContext(StandaloneConfiguration.class);
            }
            OpenPaticianApplicationWindow.startClientUI(new String[0]);
		}

	}

	private static void unpackPlugins() {
		LocalStorage localeStorage = new LocalStorage();
		File pluginDir = localeStorage.getPluginDirectory();
		String[] zipFiles = pluginDir.list((dir, name) -> name.endsWith(".zip"));
		UnzipUtility unzipper = new UnzipUtility();
		for (String zipFile : zipFiles) {

			String zipFileName = localeStorage.getPluginDirectoryPath()+File.separatorChar+zipFile;
			try {
				String pluginFilePath = unzipper.unzipPluginDescriptor(zipFileName);
				File pluginDesc = new File(pluginFilePath);
				PluginConfiguration pluginConfiguration = new PluginConfiguration();
				Unmarshaller um = pluginConfiguration.jaxbPluginUnMarshaller();
				BasePluginDefinition plugin = (BasePluginDefinition) um.unmarshal(pluginDesc);

				String outputdirectory = localeStorage.getPluginTypeDirecotryPath(plugin.getType(), zipFile);
				unzipper.unzip(zipFileName, outputdirectory);
				File f = new File(zipFileName);
				f.delete();
			} catch (IOException e) {
				logger.warn("Failed to extract plugin "+zipFile, e);
			} catch (JAXBException e) {
				e.printStackTrace();
			}
		}
	}

	private static void logEnvironment() {
		Properties sysprops = System.getProperties();
        LoggerContext logContext = (LoggerContext) LogManager.getContext();
        if (logger.isInfoEnabled()) {
			logger.info("Logging into file: "+((RollingFileAppender)logContext.getConfiguration().getAppender("FileAppender")).getFileName());
		}
		StringBuilder sb = new StringBuilder();
		sb.append("Java VM version").append(": ").append(sysprops.get("java.vm.version"));
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}
		sb = new StringBuilder();
		sb.append("Java runtime version").append(": ").append(sysprops.get("java.runtime.version"));
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}
		sb = new StringBuilder();
		sb.append("Java VM vendor").append(": ").append(sysprops.get("java.vm.vendor"));
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}
		sb = new StringBuilder();
		sb.append("Java VM name").append(": ").append(sysprops.get("java.vm.name"));
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}
		sb = new StringBuilder();
		sb.append("Java version").append(": ").append(sysprops.get("java.version"));
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}
		sb = new StringBuilder();
		sb.append("Java specification version").append(": ").append(sysprops.get("java.specification.version"));
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}
		sb = new StringBuilder();
		sb.append("Graphics environment").append(": ").append(sysprops.get("java.awt.graphicsenv"));
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}
		sb = new StringBuilder();
		sb.append("User country").append(": ").append(sysprops.get("user.country"));
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}
		sb = new StringBuilder();
		sb.append("User language").append(": ").append(sysprops.get("user.language"));
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}
		sb = new StringBuilder();
		sb.append("OS architecture").append(": ").append(sysprops.get("os.arch"));
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}
		sb = new StringBuilder();
		sb.append("OS name").append(": ").append(sysprops.get("os.name"));
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}
		sb = new StringBuilder();
		sb.append("OS version").append(": ").append(sysprops.get("os.version"));
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}
		sb = new StringBuilder();
		sb.append("Start command").append(": ").append(sysprops.get("sun.java.command")); // ?? is this generally availalbe
		if (logger.isInfoEnabled()) {
			logger.info(sb.toString());
		}

	}

//	/**
//	 * Create the server instance for a standalone game. If the server is already started no
//	 * second server will be started.
//	 * @param userGui Thread/Runnable that handels the users GUI rendering
//	 */
//	public static void createStandaloneServer(OpenPatricianFrame userGui){
//		synchronized (lock) {
//			if (startedServer) return; // only start the server once
//		}
//		// PRE: The server is not yet started
//		// TODO: create server with AI players and one human player
//	}
	
	// TODO add method connectToServer

}
