package ch.sahits.game.graphic.display.dialog.shipyard;

import ch.sahits.game.event.data.ShipyardOrderRefit;
import ch.sahits.game.javafx.control.BaleAmountAlwaysVisible;
import ch.sahits.game.javafx.control.BarrelAmount;
import ch.sahits.game.javafx.control.BarrelAmountAlwaysVisible;
import ch.sahits.game.javafx.control.CoinPriceAlwaysVisible;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.model.ControlTableCell;
import ch.sahits.game.javafx.model.ECellConstraint;
import ch.sahits.game.javafx.model.StaticTextTableCell;
import ch.sahits.game.javafx.model.Table;
import ch.sahits.game.javafx.model.TableHeader;
import ch.sahits.game.javafx.model.TableRow;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.IShipWeaponsLocation;
import ch.sahits.game.openpatrician.model.ship.ShipFactory;
import ch.sahits.game.openpatrician.model.ship.impl.ShipWeaponsLocationFactory;
import ch.sahits.game.openpatrician.model.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.scene.Group;
import javafx.scene.input.MouseEvent;
import javafx.scene.text.Text;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 13, 2013
 */
@Component("shipyardUpgradeDialog")
@Scope(value = "prototype")
@ClassCategory({EClassCategory.DIALOG, EClassCategory.SINGLETON_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ShipyardUpgradeDialogV2 extends BaseShipyardDialog {
    @ObjectPropertyType(IShip.class)
    private ObjectProperty<IShip> currentShip;
    @Autowired
    private ShipFactory shipFactory;
    @Autowired
    private ShipWeaponsLocationFactory weaponsLocationFactory;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private Locale locale;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private MessageSource messageSource;

    private static final EWare[] MATERIALS = new EWare[]{EWare.WOOD,
                                                         EWare.IRON,
                                                         EWare.PITCH};

    @MapType(key = IWare.class, value = IntegerProperty.class)
    private HashMap<IWare, IntegerProperty> availableMap = new HashMap<>();
    @MapType(key = IWare.class, value = IntegerBinding.class)
    private HashMap<IWare, IntegerBinding> priceMap = new HashMap<>();
    @MapType(key = IWare.class, value = BuyAmountBinding.class)
    private HashMap<IWare, BuyAmountBinding> buyAmountMap = new HashMap<>();

    public ShipyardUpgradeDialogV2(final ICityPlayerProxyJFX city) {
        super(city);
        this.currentShip = new SimpleObjectProperty<>(this, "currentShip", null);
        if (city.getActiveShip() instanceof IShip) {
            IShip ship = (IShip) city.getActiveShip();
            currentShip.setValue(ship);
        }

        this.enablePreviousNext = new BooleanBinding() {
            {
                super.bind(city.getPlayersShips());
            }
            @Override
            protected boolean computeValue() {
                return city.getPlayersShips().size() > 1;
            }
        };
        for (int i = 0; i < shipTypes.length; i++) {
            if (shipTypes[i] == currentShip.get().getShipType()) {
                currentShipTypeIndexProperty().set(i);
                break;
            }
        }
        mainTableYPosition = 250 + 36 + 48;
    }
    @PostConstruct
    private void intializeAdditionalText() {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardUpgradeDialogV2.title", new Object[]{}, locale.getCurrentLocal()));
        final Text name = new Text();
        name.getStyleClass().add("dialogText");
        name.setLayoutX(50);
        name.setLayoutY(250 + 36);
        name.setText(currentShip.get().getName());
        currentShip.addListener((observableValue, oldValue, newValue) -> {
            name.setText(newValue.getName());
        });
        final Text level = new Text();
        level.getStyleClass().add("dialogText");
        level.setLayoutX(50);
        level.setLayoutY(250 + 36 + 24);
        level.setText(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardUpgradeDialogV2.upgradeLevel", new Object[]{currentShip.get().getShipUpgradeLevel().nextLevel().name()}, locale.getCurrentLocal()));
        currentShip.addListener((observableValue, oldValue, newValue) -> {
            level.setText(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardUpgradeDialogV2.upgradeLevel", new Object[]{newValue.getShipUpgradeLevel().nextLevel().name()}, locale.getCurrentLocal()));
        });
        getContent().addAll(name, level);
    }
    @Override
    protected boolean hasShips() {
        if (city.getPlayersShips().size() > 0) {
            return city.getPlayersShips().stream().anyMatch(ship -> ship.isAvailable());
        } else {
            return false;
        }
    }


    @Override
    protected EventHandler<MouseEvent> getAction() {
        return mouseEvent -> {
            ITradingOffice office = getPlayer().findTradingOffice(getCity());
            for (EWare ware : MATERIALS) {
                int buyAmount = buyAmountMap.get(ware).get();
                getCity().move(ware, -buyAmount, getPlayer());
                int needed = shipFactory.getUpgradeAmount(currentShip.get().getShipType(), ware);
                int fromWarhouse = needed - buyAmount;
                office.move(ware, -fromWarhouse);
            }
            ShipyardOrderRefit event = new ShipyardOrderRefit(getCity().getCityState().getShipyardState(),currentShip.get(), currentShip.get().getShipUpgradeLevel().nextLevel(), city);
            clientServerEventBus.post(event);

            city.getPlayersShips().remove(currentShip.get());
            getPlayer().getCompany().updateCash(-calculateTotalUpgradeCosts());
            executeOnCloseButtonClicked();
        };
    }

    @Override
    protected String getActionText() {
        return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardUpgradeDialogV2.refit", new Object[]{}, locale.getCurrentLocal());
    }

    @Override
    protected Group createFooterText() {
        IntegerBinding totalPriceBinding = new IntegerBinding() {
            {
                for (EWare ware : MATERIALS) {
                    super.bind(getPriceBinding(ware, null));
                }
            }
            @Override
            protected int computeValue() {
                return calculateTotalUpgradeCosts();
            }
        };
        Group g = new Group();
        String t = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.totalSum", new Object[]{}, locale.getCurrentLocal());
        HashMap<String, Object> parameters = new HashMap<>();
        parameters.put("price", totalPriceBinding.get());
        DecoratedText dt = decoratedTextFactory.createDecoratedText(t, parameters);
        g.getChildren().add(dt);
        return g;
    }

    private int calculateTotalUpgradeCosts() {
        int total = getCity().getCityState().getShipyardState().calculateRefitCosts(currentShip.get().getShipType(), 1);
        for (EWare ware : MATERIALS) {
            total += getPriceBinding(ware, null).get();
        }
        return total;
    }

    @Override
    protected Table createMainTable() {
        Table model = new Table();
        TableHeader header = new TableHeader(5);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.materials", new Object[]{}, locale.getCurrentLocal())), ECellConstraint.COLSPAN2);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.stored", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.buy", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.costs", new Object[]{}, locale.getCurrentLocal())));
        header.setAligenment(0, HPos.CENTER);
        header.setAligenment(2, HPos.CENTER);
        header.setAligenment(3, HPos.CENTER);
        header.setAligenment(4, HPos.CENTER);
        model.setHeader(header);
        model.setAligenment(0, HPos.RIGHT);
        model.setAligenment(1, HPos.RIGHT);
        model.setAligenment(2, HPos.RIGHT);
        model.setAligenment(3, HPos.RIGHT);
        model.setAligenment(4, HPos.RIGHT);

        model.setColumnWidth(70, 70, 70, 70, 100);
        for (final EWare ware : MATERIALS) {
            TableRow row = new TableRow();
            row.add(new StaticTextTableCell(translator.getLocalDisplayName(ware)));

            final IntegerBinding needed = getNeededWareBinding(ware);
            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = new BarrelAmountAlwaysVisible();
                barrelAmount.amountProperty().bind(needed.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = new BaleAmountAlwaysVisible();
                baleAmount.amountProperty().bind(needed.asString());
                row.add(new ControlTableCell(baleAmount));
            }

            final IntegerProperty stored = getStoredAmount(ware);
            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = new BarrelAmountAlwaysVisible();
                barrelAmount.amountProperty().bind(stored.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = new BaleAmountAlwaysVisible();
                baleAmount.amountProperty().bind(stored.asString());
                row.add(new ControlTableCell(baleAmount));
            }

            final BuyAmountBinding buyAmount = new BuyAmountBinding(ware, needed);
            buyAmountMap.put(ware, buyAmount);
            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = new BarrelAmountAlwaysVisible();
                barrelAmount.amountProperty().bind(buyAmount.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = new BaleAmountAlwaysVisible();
                baleAmount.amountProperty().bind(buyAmount.asString());
                row.add(new ControlTableCell(baleAmount));
            }

            IntegerBinding priceBinding = getPriceBinding(ware, buyAmount);
            CoinPriceAlwaysVisible price = new CoinPriceAlwaysVisible();
            price.amountProperty().bind(priceBinding.asString());
            row.add(new ControlTableCell(price));

            model.add(row);
        }
        return model;
    }

    /**
     * Retrieve the stored amount from the cached map. Put it into the map when necessary.
     * @param ware for which the stored amount should be retrieved
     * @return stored amount
     */
    private IntegerProperty getStoredAmount(IWare ware) {
        if (!availableMap.containsKey(ware)) {
            final ITradingOffice office = getPlayer().findTradingOffice(getCity());

            final IntegerProperty stored = office.getWare(ware).amountProperty();
            availableMap.put(ware, stored);
        }
        return availableMap.get(ware);
    }

    /**
     * Retrieve the price from the cache map. Put it into the map when necessary.
     * @param ware for which the price should be retrieved
     * @param buyAmount that should be bought (required for the initialization)
     * @return price binding.
     */
    private IntegerBinding getPriceBinding(IWare ware, BuyAmountBinding buyAmount) {
        if (!priceMap.containsKey(ware)) {
            IntegerBinding priceBinding = new PriceBinding(ware, buyAmount);
            priceMap.put(ware, priceBinding);
        }
        return priceMap.get(ware);
    }
    private IntegerBinding getNeededWareBinding(final EWare ware) {
        return new IntegerBinding() {
            {
                super.bind(currentShipTypeIndexProperty());
            }
            @Override
            protected int computeValue() {
                return shipFactory.getConstructionAmount(getCurrentShipType(), ware);
            }
        };
    }
    @Override
    protected Table createTopTable() {
        int capacity = shipFactory.calculateInitialCapacity(getCurrentShipType(), getCity().getCoordinates().getX());
        IShipWeaponsLocation shipWeapons = weaponsLocationFactory.getShipWeaponsLocation(getCurrentShipType(), EShipUpgrade.NONE);
        Table model = new Table();
        // Add dummy header which is not rendered
        TableHeader header = new TableHeader(5);
        model.setHeader(header);
        model.setColumnWidth(70, 70, 70, 70, 70);
        TableRow row = new TableRow();
        BarrelAmount capacityAmount = new BarrelAmount();
        capacityAmount.setAmount(capacity);
        row.add(new ControlTableCell(capacityAmount));

        row.add(new StaticTextTableCell(String.valueOf(shipWeapons.getNumberOfSmallSlots())));
        row.add(new StaticTextTableCell(String.valueOf(shipWeapons.getNumberOfLargeSlots())));
        row.add(new StaticTextTableCell(String.valueOf(shipFactory.getShipSpeed(getCurrentShipType()))));
        row.add(new StaticTextTableCell(String.valueOf(shipFactory.getMinimalSailors(getCurrentShipType()))));
        model.add(row);
        return model;
    }

    @Override
    protected BooleanBinding actionEnabledBinding() {
        return new BooleanBinding() {
            {
                for (EWare ware : MATERIALS) {
                        IntegerProperty stored = getStoredAmount(ware);
                    IntegerBinding price = getPriceBinding(ware, null);
                    IntegerProperty available = getCity().getWare(ware).amountProperty();
                    super.bind(stored, price, available);
                }
                super.bind(getPlayer().getCompany().cashProperty());

            }
            @Override
            protected boolean computeValue() {
                long cash = getPlayer().getCompany().getCash();
                long buildCosts = calculateTotalUpgradeCosts();
                if (cash < buildCosts) {
                    return false;
                }
                for (EWare ware : MATERIALS) {
                    int storedAmount = getStoredAmount(ware).get();
                    int available = getCity().getWare(ware).getAmount();
                    final int needed = shipFactory.getUpgradeAmount(currentShip.get().getShipType(), ware);
                    if (needed > storedAmount + available) {
                        return false;
                    }
                }
                return true;
            }
        };
    }
    @Override
    protected EventHandler<MouseEvent> createNextAction() {
        return mouseEvent -> {
            List<IShip> ships = city.getPlayersShips();
            if (ships.size() > 1) {
                if (ships.get(ships.size()-1).equals(currentShip)) {
                    currentShip.set(ships.get(0)); // first one;
                } else {
                    for (int i=0;i<ships.size()-1; i++) {
                        if (ships.get(i).equals(currentShip)) {
                            currentShip.set(ships.get(i + 1));
                            break;
                        }
                    }
                }
            }
        };
    }

    @Override
    protected EventHandler<MouseEvent> createPreviousAction() {
        return mouseEvent -> {
            List<IShip> ships = city.getPlayersShips();
            if (ships.size() > 1) {
                if (ships.get(0).equals(currentShip)) {
                    currentShip.set(ships.get(ships.size()-1)); // last one;
                } else {
                    for (int i=1;i<ships.size(); i++) {
                        if (ships.get(i).equals(currentShip)) {
                            currentShip.set(ships.get(i - 1));
                            break;
                        }
                    }
                }
            }
        };
    }
}
