package ch.sahits.game.openpatrician.spring;

import ch.sahits.game.graphic.display.model.CityPlayerProxyJFX;
import ch.sahits.game.javafx.control.TextSizingUtility;
import ch.sahits.game.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.dialog.Dialog;
import ch.sahits.game.openpatrician.event.SubscriptionLoggingExceptionHandler;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.persistence.LoadAndSaveService;
import ch.sahits.game.openpatrician.persistence.PostLoadOperationProvider;
import ch.sahits.game.openpatrician.plugin.PluginConfiguration;
import ch.sahits.game.openpatrician.util.PausableAsyncEventBus;
import ch.sahits.game.util.UIFactory;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Scope;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.oxm.jaxb.Jaxb2Marshaller;

import java.util.Timer;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Nov 29, 2014
 */
@Configuration
@Import({ImageConfiguration.class, SoundConfiguration.class, PluginConfiguration.class})
@ComponentScan(basePackages = {"ch.sahits.game.graphic.image",
        "ch.sahits.game.graphic.layout",
        "ch.sahits.game.graphic.display",
        "ch.sahits.game.graphic.loaders",
        "ch.sahits.game.graphic.javafx",
        "ch.sahits.game.javafx.model",
        "ch.sahits.game.javafx.util",
        "ch.sahits.game.event.handler.impl",
        "ch.sahits.game.graphic.event.task",
        "ch.sahits.game.javafx.service"})
@ClassCategory(EClassCategory.STARTUP)
public class UICommonConfiguration {

    @Bean
    public DecoratedTextFactory decoratedDialogTextFactory() {
        return new DecoratedTextFactory(2* Dialog.FRAME_BORDER, 100, Dialog.WRAPPING_WIDTH);
    }
    @Bean
    public Jaxb2Marshaller jaxb2Marshaller() {
        Jaxb2Marshaller bean = new Jaxb2Marshaller();
        bean.setContextPath("ch.sahits.game.graphic.data");
        return bean;
    }
    @Bean
    public ExecutorService clientThreadPool() {
        return Executors.newSingleThreadExecutor();
    }
    @Bean
    public SubscriptionLoggingExceptionHandler subscriptionExceptionClientHandler() {
        return new SubscriptionLoggingExceptionHandler("eventBusException");
    }

    @Bean
    public PausableAsyncEventBus clientEventBus() {
        return new PausableAsyncEventBus(clientThreadPool(), subscriptionExceptionClientHandler());
    }
    @Bean
    public ReloadableResourceBundleMessageSource messageSource() {
        ReloadableResourceBundleMessageSource bundle = new ReloadableResourceBundleMessageSource();
        bundle.setDefaultEncoding("UTF-8");
        bundle.setBasename("i18n/text");
        bundle.setFallbackToSystemLocale(false);
        return bundle;
    }

    @Bean
    public ReloadableResourceBundleMessageSource resourceReference() {
        ReloadableResourceBundleMessageSource bundle = new ReloadableResourceBundleMessageSource();
        bundle.setBasename("/resources");
        bundle.setFallbackToSystemLocale(false);
        return bundle;
    }
    @Bean
    public UIFactory uiFactory() {
        return new UIFactory();
    }
    @Bean
    public Timer uiTimer() {
        return new Timer("UI-Timer");
    }
    @Bean
    @Lazy
    public TextSizingUtility textSizingUtility() {
        return new TextSizingUtility();
    }

    @Bean
    @Lazy
    public LoadAndSaveService loadAndSaveService() {
        return new LoadAndSaveService();
    }

    @Bean
    @Lazy
    public PostLoadOperationProvider postLoadOperationProvider() {
        return new PostLoadOperationProvider();
    }

    @Bean
    @Scope("prototype")
    public ICityPlayerProxyJFX cityPlayerProxy(ICity city, IHumanPlayer player, IShip ship) {
        return new CityPlayerProxyJFX(city, player, ship);
    }
}
