package ch.sahits.game.graphic.display.dialog.cityhall;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.event.data.ClockTickDayChange;
import ch.sahits.game.event.data.ElectionWinnerNotification;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.engine.land.city.CityHallEngine;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DateService;
import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.cityhall.EElectionType;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityHall;
import ch.sahits.game.openpatrician.model.personal.EEconomicCareer;
import ch.sahits.game.javafx.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.application.Platform;
import javafx.css.CssMetaData;
import javafx.css.SimpleStyleableObjectProperty;
import javafx.css.StyleConverter;
import javafx.css.Styleable;
import javafx.css.StyleableObjectProperty;
import javafx.css.StyleableProperty;
import javafx.scene.control.Control;
import javafx.scene.layout.VBox;
import javafx.scene.text.Font;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Apr 04, 2015
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class HanseaticInfoDialog extends CloseButtonDialog {
    @Autowired
    private ClientViewState viewState;
    @Autowired
    private CityHallEngine cityHallEngine;
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private ModelTranslations modelTranslator;
    @Autowired
    private Date date;
    @Autowired
    private DateService dateService;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    private final ICityPlayerProxyJFX city;
    private VBox box;

    private boolean displayAtElectionDay = false;

    public HanseaticInfoDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
        getStylesheets().add(this.getClass().getResource("/styles/base.css").toExternalForm());
        getStyleClass().add("dialog");
    }


    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.HanseaticInfoDialog.title", new Object[]{}, locale.getCurrentLocal()));
        resetContent();
        clientServerEventBus.register(this);
        timerEventBus.register(this);
     }
    @PreDestroy
    private void unregister() {
        clientEventBus.unregister(this);
        timerEventBus.unregister(this);
    }

    private void resetContent() {
        getContent().remove(box);
        ICity town = city.getCity();
        ICityHall cityHall = cityHallEngine.findCityHall(town);
        final DateTime electionDate = cityHall.getAldermanElectionDate();
        DateTime today = date.getCurrentDate();
        box = textFactory.createMultiParagraphContainer();

        ICitizen alderman = cityHall.getAlderman();
        boolean articleMale = true;
        if (alderman instanceof IHumanPlayer && !((IPlayer) alderman).isMale()) {
            articleMale = false;
        }
        String attribute = modelTranslator.getLocalDisplayName(EEconomicCareer.HONEST);
        if (alderman instanceof IHumanPlayer) {
            attribute = modelTranslator.getLocalDisplayName(((IPlayer) alderman).getCareerLevel());
        }
        System.out.println(alderman);
        Object[] args = {modelTranslator.getLocalDisplayDistinctArticle(articleMale, false), attribute, alderman.getName() + " " + alderman.getLastName(), alderman.getHometown().getName()};
        DecoratedText text;
        if (dateService.isSameDay(electionDate, today)) {
            String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.HanseaticInfoDialog.electionDay", new Object[]{modelTranslator.toDisplayString(electionDate), alderman.getHometown().getName()}, locale.getCurrentLocal());
            text = textFactory.createDecoratedText(template, new HashMap<>());
            box.getChildren().add(text);
            displayAtElectionDay = true;
        } else {
            String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.HanseaticInfoDialog.paragraph1", args, locale.getCurrentLocal());
            text = textFactory.createDecoratedText(template, new HashMap<>());
            box.getChildren().add(text);
            template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.HanseaticInfoDialog.paragraph2", new Object[]{modelTranslator.toDisplayString(electionDate), alderman.getHometown().getName()}, locale.getCurrentLocal());
            text = textFactory.createDecoratedText(template, new HashMap<>());
            box.getChildren().add(text);
            displayAtElectionDay = false;
        }
        List<ICitizen> candidates = cityHall.getAldermanCandidates().getAll();
        for (ICitizen candidate : candidates) {
            text = getCandidateRow(candidate);
            box.getChildren().add(text);

        }

        getContent().addAll(box);
    }
    private DecoratedText getCandidateRow(ICitizen candidate) {
        Object[] args;
        String template;
        DecoratedText text;
        if (candidate instanceof IPlayer) {
            IPlayer p = (IPlayer) candidate;
            args = new Object[]{modelTranslator.getLocalDisplayDistinctArticle(p.isMale(), true), modelTranslator.getLocalDisplayName(p.getCareerLevel()), modelTranslator.getLocalDisplayName(p.getRank()), p.getName()+" "+p.getLastName()};
        } else {
            args = new Object[]{modelTranslator.getLocalDisplayDistinctArticle(true, true), modelTranslator.getLocalDisplayName(EEconomicCareer.SMART), modelTranslator.getLocalDisplayName(candidate.getRank()), candidate.getName()+" "+candidate.getLastName()};
        }
        template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.TownInfoNotice.candidate", args, locale.getCurrentLocal());
        text = textFactory.createDecoratedText(template, new HashMap<>());
        return text;
    }

    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_CITY_HALL_ALDERMAN);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
        clientServerEventBus.unregister(this);
    }
    @Subscribe
    public void checkDayChangeUpdates(ClockTickDayChange dayChange) {
        ICity town = city.getCity();
        ICityHall cityHall = cityHallEngine.findCityHall(town);
        final DateTime electionDate = cityHall.getAldermanElectionDate();
        if (dateService.isSameDay(electionDate, date.getCurrentDate())) {
            Platform.runLater(() -> {
                resetContent();
            });
        }
    }
    @Subscribe
    public void checkPostDayChangeUpdates(ElectionWinnerNotification electionResult) {
        if (displayAtElectionDay && electionResult.getElectionType().equals(EElectionType.ALDERMAN) && electionResult.getCity().equals(city.getCity())) {
            ICity town = city.getCity();
            ICityHall cityHall = cityHallEngine.findCityHall(town);
            final DateTime electionDate = cityHall.getAldermanElectionDate();
            if (!dateService.isSameDay(electionDate, date.getCurrentDate())) {
                Platform.runLater(() -> {
                    resetContent();
                });
            }
        }
    }
    private StyleableObjectProperty<Font> font;

    public Font getFont() {
        return font == null ? Font.getDefault() : font.get();
    }

    public void setFont(Font font) {
        this.font.set(font);
    }

    public StyleableObjectProperty<Font> fontProperty() {
        if (font == null) {
            font = new SimpleStyleableObjectProperty<Font>(StyleableProperties.FONT, HanseaticInfoDialog.this, "font", Font.getDefault());
        }
        return font;
    }

    private static class StyleableProperties {
        private static final CssMetaData<HanseaticInfoDialog, Font> FONT =
                new CssMetaData<HanseaticInfoDialog, Font>("-fx-font",
                        StyleConverter.getFontConverter(), Font.getDefault()) {
                    @Override
                    public boolean isSettable(HanseaticInfoDialog control) {
                        return control.font == null || !control.font.isBound();
                    }

                    @Override
                    public StyleableProperty<Font> getStyleableProperty(HanseaticInfoDialog control) {
                        return control.fontProperty();
                    }
                };
        private static final List<CssMetaData<? extends Styleable, ?>> STYLEABLES;

        static {
            final List<CssMetaData<? extends Styleable, ?>> styleables =
                    new ArrayList<CssMetaData<? extends Styleable, ?>>(Control.getClassCssMetaData());
            Collections.addAll(styleables,
                    FONT
            );
            STYLEABLES = Collections.unmodifiableList(styleables);
        }
    }

    public List<CssMetaData<? extends Styleable, ?>> getControlCssMetaData() {
        return getClassCssMetaData();
    }

    public static List<CssMetaData<? extends Styleable, ?>> getClassCssMetaData() {
        return StyleableProperties.STYLEABLES;
    }
}
