package ch.sahits.game.graphic.javafx.display;

import ch.sahits.game.event.handler.impl.EventHandlerFactory;
import ch.sahits.game.javafx.control.DisplayMessageOverlayLabel;
import ch.sahits.game.javafx.control.TextSizingUtility;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.javafx.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.animation.FadeTransition;
import javafx.application.Platform;
import javafx.event.EventHandler;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.VBox;
import javafx.util.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.TimerTask;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Mar 04, 2015
 */
@LazySingleton
@ClassCategory(EClassCategory.HANDLER)
public class DisplayMessageOverlay extends VBox {

    private final static int DISPLAY_MESSAGE_FOR_SECONDS = 15;

    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("uiTimer")
    private ScheduledExecutorService uiTimer;
    @Autowired
    private TextSizingUtility sizeing;
    @Autowired
    private EventHandlerFactory eventHandlerFactory;
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private ModelTranslations translator;
    public DisplayMessageOverlay() {
    }
    @PostConstruct
    private void init() {
        clientEventBus.register(this);
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        clientEventBus.unregister(this);
        clientServerEventBus.unregister(this);
    }
    @Subscribe
    public void handleMessage(DisplayMessage message) {
        System.out.println("Caught display message: "+message.getMessageKey());
        Object[] args = message.getMessageArguments();
        for (int i = 0; i < args.length; i++) {
            if (args[i] instanceof EWare) {
                args[i] = translator.getLocalDisplayName((EWare) args[i]);
            }
        }
        String translatedMessage = messageSource.getMessage(message.getMessageKey(), args, locale.getCurrentLocal());
        DisplayMessageOverlayLabel label = new DisplayMessageOverlayLabel(translatedMessage, sizeing);
        if (message.hasDialogTemplate()) {
            label.setLinkText(true);
            EventHandler<MouseEvent> openDialogEvenHandler = createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(message.getDialogTemplate()));
            label.setOnMouseReleased(openDialogEvenHandler);
        }
        TimerTask task = new TimerTask() {
            @Override
            public void run() {
                Platform.runLater(() -> {
                    FadeTransition ft = new FadeTransition(Duration.millis(2000), label);
                    ft.setFromValue(1.0);
                    ft.setToValue(0);

                    ft.setAutoReverse(false);
                    ft.setOnFinished(event -> getChildren().remove(label));

                    ft.play();

                });
            }
        };

        uiTimer.schedule(task, DISPLAY_MESSAGE_FOR_SECONDS, TimeUnit.SECONDS);
        Platform.runLater(() -> {
            getChildren().add(0, label);
        });
    }

    /**
     * Helper method to convert the old event triggering into JavaFX events.
     * @param run
     * @return
     */
    private EventHandler<MouseEvent> createEventHandlerFromRunnable(final Runnable run) {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent arg0) {
                run.run();
            }
        };

    }
}
