package ch.sahits.game.graphic.display.dialog.cityhall;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.engine.land.city.CityEngine;
import ch.sahits.game.openpatrician.engine.land.city.CityHallEngine;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityHall;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityHallNotice;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.javafx.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.collections.ObservableList;
import javafx.css.CssMetaData;
import javafx.css.SimpleStyleableObjectProperty;
import javafx.css.StyleConverter;
import javafx.css.Styleable;
import javafx.css.StyleableObjectProperty;
import javafx.css.StyleableProperty;
import javafx.event.EventHandler;
import javafx.scene.control.Control;
import javafx.scene.input.MouseEvent;
import javafx.scene.text.Font;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Mar 18, 2015
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class PublicNoticeDialog extends CloseButtonDialog {
    private final Logger logger = LogManager.getLogger(getClass());

    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private ModelTranslations modelTranslator;
    @Autowired
    private Date date;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private CityEngine cityEngine;
    @Autowired
    private CityHallEngine cityHallEngine;
    @Autowired
    private ClientViewState viewState;
    @ListType(ICityHallNotice.class)
    private ObservableList<ICityHallNotice> notices;
    private final IntegerProperty currentIndex = new SimpleIntegerProperty(0);
    private  BooleanBinding enablePreviousNext;
    private final ICityPlayerProxyJFX city;

    public PublicNoticeDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
        getStylesheets().add(this.getClass().getResource("/styles/base.css").toExternalForm());
        getStyleClass().add("dialog");
    }

    @PostConstruct
    private void initializeDialog() {
        ICityHall cityHall = cityHallEngine.findCityHall(city.getCity());
        notices = cityHall.getNotices();
        enablePreviousNext = new BooleanBinding() {
            {
                super.bind(notices);
            }
            @Override
            protected boolean computeValue() {
                return notices.size() > 1;
            }
        };
        setupCurrentContent();
        final int actionButtonX = (WIDTH - 124) / 2;

        final OpenPatricianSmallWaxButton previous = new OpenPatricianSmallWaxButton("<");
        previous.getStyleClass().add("actionButton");
        previous.setLayoutX(actionButtonX - 62 - 4);
        previous.setLayoutY(CLOSE_BTN_Y_POS - 48);
        previous.setOnAction(createPreviousAction());
        previous.setDisable(notices.size() <= 1);
        final OpenPatricianSmallWaxButton next = new OpenPatricianSmallWaxButton(">");
        next.getStyleClass().add("actionButton");
        next.setLayoutX(actionButtonX + 124 + 4);
        next.setLayoutY(CLOSE_BTN_Y_POS - 48);
        next.setOnAction(createNextAction());
        next.setDisable(notices.size() <= 1);

        final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernBaseSideRoomPersonDialog.accept", new Object[]{}, locale.getCurrentLocal()));
        action.getStyleClass().add("actionButton");
        action.setOnAction(getAction());
        action.setLayoutX(actionButtonX);
        action.setLayoutY(CLOSE_BTN_Y_POS - 24);

        enablePreviousNext.addListener((observableValue, oldVBoolean, newValue) -> {
            next.setDisable(!newValue);
            previous.setDisable(!newValue);
        });
        getContent().addAll(previous, next, action);

    }

    private EventHandler<MouseEvent> getAction() {
        return (mouseEvent) -> {
              ICityHallNotice notice = notices.get(currentIndex.get());
            ICity destination = notice.getDestination();
            cityEngine.findTavernEngine().getTavernState(destination).setContractBrooker(notice.getContact());
            notices.remove(notice);
            if (notices.isEmpty()) {
                executeOnCloseButtonClicked();
            } else {
                if (currentIndex.get() >= notices.size()) {
                    currentIndex.set(currentIndex.get() - 1);
                }
                setupCurrentContent();
            }
        };
    }

    private void setupCurrentContent() {
        removeDecoratedContent();
        if (notices.size() > 0) {

            final ICityHallNotice notice = notices.get(currentIndex.get());
            EWare ware = (EWare) notice.getRequiredWare();
            ICity destination = notice.getDestination();
            setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.cityhall.PublicNoticeDialog.title", new Object[]{modelTranslator.getLocalDisplayName(ware), destination.getName()}, locale.getCurrentLocal()));
            String key;
            if (ware.isBarrelSizedWare()) {
                key = "ch.sahits.game.graphic.display.dialog.cityhall.PublicNoticeDialog.barrelMessage";
            } else {
                key = "ch.sahits.game.graphic.display.dialog.cityhall.PublicNoticeDialog.baleMessage";

            }
            AmountablePrice<IWare> amountPrice = notice.getWareAndAmount();
            final DateTime dueDate1 = notice.getDueDate();
            String dueDate = modelTranslator.toDisplayString(dueDate1);
            int premium = amountPrice.getAVGPrice() * amountPrice.getAmount();
            Object[] args = new Object[]{modelTranslator.getLocalDisplayName(ware), amountPrice.getAmount(), dueDate, premium, amountPrice.getAVGPrice(), 1};
            String template = messageSource.getMessage(key, args, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
            getContent().add(text);
        }
    }

    /**
     * Move the selection to the next index
     * @return .
     */
    private EventHandler<MouseEvent> createNextAction() {
        return (mouseEvent) -> {
            if (currentIndex.get() == notices.size() - 1) {
                currentIndex.set(0);
            } else {
                currentIndex.set(currentIndex.get() + 1);
            }
            setupCurrentContent();
        };
    }

    /**
     * move the selection to the previous index.
     * @return  .
     */
    private EventHandler<MouseEvent> createPreviousAction() {
        return (mouseEvent)  -> {
            if (currentIndex.get() == 0) {
                currentIndex.set(notices.size() - 1);
            } else {
                currentIndex.set(currentIndex.get() - 1);
            }
            setupCurrentContent();
        };
    }

    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_CITY_HALL_BOARD);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }

    private StyleableObjectProperty<Font> font;

    public Font getFont() {
        return font == null ? Font.getDefault() : font.get();
    }
    public void setFont(Font font) {
        this.font.set(font);
    }
    public StyleableObjectProperty<Font> fontProperty() {
        if (font == null) {
            font = new SimpleStyleableObjectProperty<Font>(StyleableProperties.FONT, PublicNoticeDialog.this, "font", Font.getDefault());
        }
        return font;
    }

    private static class StyleableProperties {
        private static final CssMetaData< PublicNoticeDialog, Font> FONT =
                new CssMetaData<PublicNoticeDialog, Font>("-fx-font",
                        StyleConverter.getFontConverter(), Font.getDefault()) {
                    @Override
                    public boolean isSettable(PublicNoticeDialog control) {
                        return control.font == null || !control.font.isBound();
                    }
                    @Override
                    public StyleableProperty<Font> getStyleableProperty(PublicNoticeDialog control) {
                        return control.fontProperty();
                    }
                };
        private static final List<CssMetaData<? extends Styleable, ?>> STYLEABLES;
        static {
            final List<CssMetaData<? extends Styleable, ?>> styleables =
                    new ArrayList<CssMetaData<? extends Styleable, ?>>(Control.getClassCssMetaData());
            Collections.addAll(styleables,
                    FONT
            );
            STYLEABLES = Collections.unmodifiableList(styleables);
        }
    }
    public List<CssMetaData<? extends Styleable, ?>> getControlCssMetaData() {
        return getClassCssMetaData();
    }
    public static List<CssMetaData<? extends Styleable, ?>> getClassCssMetaData() {
        return StyleableProperties.STYLEABLES;
    }


}
