package ch.sahits.game;

import ch.sahits.game.graphic.javafx.display.MainGameView;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.util.GamePropertyUtility;
import ch.sahits.game.openpatrician.util.PropertyFileKeys;
import ch.sahits.game.openpatrician.util.service.DisableProperties;
import com.google.common.base.Preconditions;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;

import java.util.Properties;

/**
 * Helper class to encapsulate the funcionality concerning command line arguments
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 16, 2013
 *
 */
@ClassCategory(EClassCategory.STARTUP)
public class CommandLineArguments {

	static final String VERSION_OPTION = "version";
	static final String HELP_OPTION = "h";
	static final String GAME_LOCALE = "l";
	private static final String WINDOWED_MODE = "w";
	private static final String FULLSCREEN_MODE = "f";
	public static final String CLIENT_OPTION = "c";
	public static final String SERVER_OPTION = "s";
	private CommandLineParser parser = new PosixParser();
	private GamePropertyUtility gameProperties = new GamePropertyUtility();

	@SuppressWarnings("static-access")
	public Options createCommandLineOptions() {
		Options opts = new Options();
		opts.addOption(HELP_OPTION, "help", false, "Display help");
		opts.addOption(OptionBuilder.withLongOpt(VERSION_OPTION)
				.withDescription("Version of this application")
				.create());
		opts.addOption(FULLSCREEN_MODE, "fullscreen", false, "fullscreen mode");
		opts.addOption(OptionBuilder.withArgName(WINDOWED_MODE)
				.withLongOpt("windowed")
				.hasOptionalArgs(1)
				.withArgName("widthxheight")
				.withDescription("Windowed mode with optional definition of window size like: 1280x780")
				.create());
		opts.addOption(DisableProperties.DISABLE_RANDOM_EVENTS, false, "Disable random events");
		opts.addOption(DisableProperties.DISABLE_AI_TRADING, false, "Disable AI trading");
		opts.addOption(GAME_LOCALE, "lang", true, "Specify the locale to use");
		// todo: andi 4/22/15: it might be a good idea to list all the available languages here,
        // this however would require do define the list of the available languages outside of ch.sahits.game.openpatrician.util.l10n.Locale
        // Perhaps create a simple standalone object holding the languages and using that object here and in ch.sahits.game.openpatrician.util.l10n.Locale.
		opts.addOption(CLIENT_OPTION, "client", false, "Start application in client mode. Currently unused. Either client or server must be specified");
		opts.addOption(SERVER_OPTION, "server", false, "Start application in server mode. Currently unused. Either client or server must be specified");
		return opts;

	}

	public CommandLine parseCommandLine(Options options, String[] args) {
		try {
			// parse the command line arguments
			return parser.parse( options, args );
		}
		catch( ParseException exp ) {
			printHelp(options);
			throw new IllegalArgumentException("Parsing of command line arguments failed", exp);
		}
	}

	public void printHelp(Options options) {
		HelpFormatter formatter = new HelpFormatter();
		formatter.printHelp( "OpenPatrician", options );
	}

	/**
	 * Check wether the commandline arguments for standalone are set.
	 * @param cmdLine command line object
	 * @return true if both client and server arguments are specified or neither.
	 */
	public boolean isStandalone(CommandLine cmdLine) {
		return !cmdLine.hasOption(CLIENT_OPTION) && !cmdLine.hasOption(SERVER_OPTION)
				|| cmdLine.hasOption(CLIENT_OPTION) && cmdLine.hasOption(SERVER_OPTION);
	}

	/**
	 * Check weather the application is started as server.
	 * @param cmdLine command line object
	 * @return true if only the server flag is set
	 */
	public boolean isServer(CommandLine cmdLine) {
		return !isStandalone(cmdLine) && cmdLine.hasOption(SERVER_OPTION);
	}

	/**
	 * Check weather the application is started as client.
	 * @param cmdLine command line object
	 * @return true if only the client flag is set
	 */
	public boolean isClient(CommandLine cmdLine) {
		return !isStandalone(cmdLine) && cmdLine.hasOption(CLIENT_OPTION);
	}

	/**
	 * Set the spring profiles and write some properties, which will be used by Spring.
	 * @param cmdLine command line object
	 */
	public void persistAsPropertyFile(CommandLine cmdLine) {
		Properties props = gameProperties.getProperties();
		if (isStandalone(cmdLine)) {
			gameProperties.addSpringProfile("standalone");
		} else if (cmdLine.hasOption(CLIENT_OPTION)) {
			gameProperties.addSpringProfile("client");
		} else if (cmdLine.hasOption(SERVER_OPTION)) {
			gameProperties.addSpringProfile("server");
		}
		if (cmdLine.hasOption(FULLSCREEN_MODE)) {
			props.setProperty(PropertyFileKeys.WINDOW_FULLSCREEN, "true");
		} else {
			props.setProperty(PropertyFileKeys.WINDOW_FULLSCREEN, "false");
		}

		if (cmdLine.hasOption(WINDOWED_MODE)) {
			props.setProperty(PropertyFileKeys.WINDOW_FULLSCREEN, "false");
			String windowedOption = cmdLine.getOptionValue(WINDOWED_MODE);
			if (windowedOption != null) {
				String[] values = windowedOption.split("x");
				try {
					Integer.parseInt(values[0]);
					Integer.parseInt(values[1]);
				} catch (NumberFormatException e) {
					throw new NumberFormatException("The window size parameter is invalid: "+windowedOption);
				}
				props.setProperty(PropertyFileKeys.WINDOW_WIDTH, values[0]);
				props.setProperty(PropertyFileKeys.WINDOW_HEIGHT, values[1]);
			} else {
				// compute default width and height
			}
		}
		int height;
		if (props.get(PropertyFileKeys.WINDOW_HEIGHT) == null) {
			height = MainGameView.MINMIMAL_DISPLAY_HEIGHT;
			props.setProperty(PropertyFileKeys.WINDOW_HEIGHT, String.valueOf(height));
		} else {
			height = Integer.parseInt((String) props.get(PropertyFileKeys.WINDOW_HEIGHT));
		}

		checkMinimalHeight(height);

		if (cmdLine.hasOption(GAME_LOCALE)){
			props.setProperty(PropertyFileKeys.GAME_LOCALE, cmdLine.getOptionValue(GAME_LOCALE));
		}

		// Disable properties
		if (cmdLine.hasOption(DisableProperties.DISABLE_RANDOM_EVENTS)) {
			System.setProperty(DisableProperties.DISABLE_RANDOM_EVENTS, "true");
		}
		if (cmdLine.hasOption(DisableProperties.DISABLE_AI_TRADING)) {
			System.setProperty(DisableProperties.DISABLE_AI_TRADING, "true");
		}

		gameProperties.persistProperties();
	}



	private void checkMinimalHeight(int height) {
		Preconditions.checkArgument(height>= MainGameView.MINMIMAL_DISPLAY_HEIGHT, "The frame in the display must have at least "+MainGameView.MINMIMAL_DISPLAY_HEIGHT+"px height");
	}
}
