package ch.sahits.game.graphic.display.dialog.action;

import ch.sahits.game.graphic.display.dialog.util.ITransferableJFX;
import ch.sahits.game.javafx.bindings.ConstantIntegerBinding;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.ComputablePriceV2;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import com.google.common.base.Preconditions;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.util.Pair;

import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;

/**
 * Action of buying stuff from the city onto the ship
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 23, 2011
 *
 */
@ClassCategory(EClassCategory.HANDLER)
class City2ShipJFXAction implements Runnable{
	private final IWare ware;
	private final ITransferableJFX transfer;

//	private static final Logger logger = Logger.getLogger(City2ShipAction.class);

	public City2ShipJFXAction(IWare ware, ITransferableJFX transfer) {
		super();
		this.ware = ware;
		this.transfer =transfer;
	}

	@Override
	public void run() {
		final ICity city = transfer.getCity();
		int availableAmountCity = city.getWare(ware).getAmount();
		if (availableAmountCity>0) {
			int amount2Move = transfer.getAmount(availableAmountCity); // This is ware specific size
			// consider available capacity
			final short sizeAsBarrels = ware.getSizeAsBarrels();
			final IPlayer player = transfer.getPlayer();
			Optional<ITradingOffice> optOffice = player.findTradingOffice(city);
			final INavigableVessel vessel = transfer.getVessel();
			final int capacity = vessel.getCapacity();
			if (sizeAsBarrels == 1) {
				if (!optOffice.isPresent()) {
					amount2Move = Math.min(amount2Move, capacity);
				}
			} else {
				int temp = Math.min(amount2Move * sizeAsBarrels, capacity);
				amount2Move = temp / sizeAsBarrels;
			}
			ComputablePriceV2 computablePrice = transfer.getComputablePrice();
			int avgPrice = computablePrice.buyPrice(ware, new SimpleIntegerProperty(availableAmountCity), new ConstantIntegerBinding(amount2Move));
			// check if this is afforable
			long cash = 0;
			ICompany company = player.getCompany();
			if (vessel instanceof IShip) {
				cash = company.getCash();
			} else {
				IConvoy convoy = (IConvoy) vessel;
				for (IPlayer p : convoy.getPlayers()) {
					cash += p.getCompany().getCash();
				}
			}
			Pair<Integer, Integer> affordable = calculateAffordableAmount(cash, amount2Move, avgPrice, availableAmountCity);

			amount2Move = affordable.getKey();
			avgPrice = affordable.getValue();
			if (amount2Move <= 0) {
				return; // cannot buy anything
			}
			int movedAmount = city.move(ware, -amount2Move, player);
			if (amount2Move != -movedAmount) { // not enough available in city
				avgPrice = computablePrice.buyPrice(ware, new SimpleIntegerProperty(city.getWare(ware).getAmount() + movedAmount),
						new ConstantIntegerBinding(movedAmount));
				amount2Move = -movedAmount;
			}
			int loaded;
			if (!optOffice.isPresent()){  // amount2Move is max the ships capacity
				loaded = vessel.load(ware, amount2Move, avgPrice);
			} else {
				ITradingOffice office = optOffice.get();
				if (amount2Move > capacity) {
				   int loadOntoShip = capacity;
					int storeInOffice = amount2Move - capacity;
					loaded = vessel.load(ware, loadOntoShip, avgPrice);
					loaded += office.move(ware, storeInOffice);
				} else {
					loaded = vessel.load(ware, amount2Move, avgPrice);
				}
			}
			if (vessel instanceof IShip) {
				company.updateCash(-avgPrice * loaded);
			} else {
				IConvoy convoy = (IConvoy) vessel;
				Map<IPlayer, Integer> capacityMap = convoy.getCapacityPerOwner();
				long totalCash = -avgPrice * loaded;
				double totalCapacity = capacity;
				for (Entry<IPlayer, Integer> entry : capacityMap.entrySet()) {
					double percentage = entry.getValue()/totalCapacity;
					long partialAmount = Math.round(totalCash*percentage);
					entry.getKey().getCompany().updateCash(partialAmount);
				}
			}
		} // end no ware available
	}

	/**
	 * Calculate how much can be bought due to monetary restrictions.
	 * @param cash available cash
	 * @param desiredAmount amount that is desired to be bought
	 * @param avgPrice average price for an item when buying the whole amount.
	 * @param availableAmountCity amount of the ware available in the city
	 * @return Pair containing the affordable amount and the average price.
	 */
	Pair<Integer, Integer> calculateAffordableAmount(long cash, int desiredAmount, int avgPrice, int availableAmountCity) {
		if (cash < (long) avgPrice * desiredAmount) {
			int amountAprox = (int) (cash / avgPrice); // approx amount we can afford
			// fixme: andi 12/13/14: this might still be above the available cash
			if (amountAprox > 0) {
				ComputablePriceV2 computablePrice = transfer.getComputablePrice();
				int tempPrice = computablePrice.buyPrice(ware, new SimpleIntegerProperty(availableAmountCity), new ConstantIntegerBinding(amountAprox));
				Preconditions.checkArgument(amountAprox*tempPrice<=cash, "Buying fewer items never results in a higher avg price");
				while (amountAprox * tempPrice + tempPrice < cash) {
					int newTempPrice = computablePrice.buyPrice(ware, new SimpleIntegerProperty(availableAmountCity), new ConstantIntegerBinding(++amountAprox));
					if (amountAprox * newTempPrice > cash) {
						// we cannot afford another item
						break;
					}
					tempPrice = newTempPrice;
				}
				return new Pair<>(amountAprox, tempPrice);
			} else {
				return new Pair<>(0, 0);
			}
		} else {
			return new Pair<>(desiredAmount, avgPrice);
		}
	}

}
