package ch.sahits.game.graphic.display.dialog.guild;

import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.graphic.display.dialog.util.DialogUtil;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.IItemNavigatorLabelWrapper;
import ch.sahits.game.javafx.control.ItemNavigator;
import ch.sahits.game.javafx.control.PlaceHolder;
import ch.sahits.game.javafx.service.DecoratedTextFactory;
import ch.sahits.game.javafx.service.ItemNavigatorLabelWrapperFactory;
import ch.sahits.game.javafx.util.ModelTranslations;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.building.TradingOfficeList;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.guild.GuildList;
import ch.sahits.game.openpatrician.model.city.guild.IGuild;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.personal.IPersonalData;
import ch.sahits.game.openpatrician.model.personal.ISpouseData;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.beans.property.IntegerProperty;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.control.Label;
import javafx.scene.layout.Pane;
import javafx.scene.layout.VBox;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

/**
 * Dialog to display details on the members of a guild
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Nov 21, 2016
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class GuildMembersDialog extends CloseButtonDialog {
    /** Reference to the city view model */
    private final ICityPlayerProxyJFX city;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private GuildList guildList;
    @Autowired
    private ItemNavigatorLabelWrapperFactory itemWrapperFactory;
    @Autowired
    private ModelTranslations modelTranslations;
    @Autowired
    private TradingOfficeList offices;
    @Autowired
    private Date date;
    @Autowired
    private IMap map;
    @Autowired
    private DialogUtil dialogHelper;
    private VBox box;

    public GuildMembersDialog(ICityPlayerProxyJFX city) {
        this.city = city;
    }
    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.guild.GuildMembersDialog.title",
                new Object[]{}, locale.getCurrentLocal()));

        ICity city = this.city.getCity();
        IGuild guild = guildList.findGuild(city).get();
        if (guild.getMembers().isEmpty()) {
            String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.guild.GuildMembersDialog.noMembers",
                    new Object[]{city.getName()}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
            text.setId("noMembers");
            box = new VBox(text);
            box.setLayoutY(CLOSE_BTN_Y_POS - 500);
            getContent().addAll(box);
        } else {
            List<IPlayer> members = guild.getMembers();
            List<IItemNavigatorLabelWrapper<IPlayer>> wrappedList = itemWrapperFactory.createNonLabeledListForPlayer(members);
            ItemNavigator<IPlayer> itemNavigator = new ItemNavigator<>(wrappedList);
            initializeContent(itemNavigator);
            if (members.size() > 1) {
                Group centerPane = dialogHelper.center(itemNavigator);
                box.getChildren().addAll(dialogHelper.createVerticalSpacer(50), centerPane);
            }
        }

    }

    private void initializeContent(final ItemNavigator<IPlayer> itemNavigator) {

        IntegerProperty currentPos = itemNavigator.currentPositionProperty();
        currentPos.addListener((observable, oldValue, newValue) -> {
            for (Iterator<Node> iterator = getContent().iterator(); iterator.hasNext(); ) {
                Node node = iterator.next();
                if (node instanceof VBox) {
                    iterator.remove();
                }
            }
            initializeContent(itemNavigator);
        });
        IPlayer member = itemNavigator.getCurrent();

        Label header = new Label(member.getName() + " " +member.getLastName());
        header.getStyleClass().addAll("subTitle");
        header.setId("playerName");
        Pane centerPane = dialogHelper.center(header);

        box = new VBox(centerPane, new PlaceHolder(1, 10));
        List<ITradingOffice> offices = this.offices.findOwnedBy(member);
        ITradingOffice oldest = null;
        DateTime foundingDate = DateTime.now();
        for (ITradingOffice office : offices) {
           DateTime date = office.getEstablishedDate();
           if (date.isBefore(foundingDate)) {
               foundingDate = date;
               oldest = office;
           }
        }

        String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.guild.GuildMembersDialog.summary",
                new Object[]{modelTranslations.getLocalDisplayName(member.getCareerLevel()),
                        modelTranslations.getLocalDisplayName(member.getRank()), member.getName(), member.getLastName(), member.getHometown().getName(),
                        modelTranslations.toDisplayString(foundingDate), oldest.getCity().getName()}, locale.getCurrentLocal());
        DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
        text.setId("summary");
        box.getChildren().add(text);
        box.getChildren().add(new PlaceHolder(1, 30));
        IPersonalData personalData = member.getPersonalData();
        template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.guild.GuildMembersDialog.age",
                new Object[]{personalData.getAge(date.getCurrentDate())}, locale.getCurrentLocal());
        text = textFactory.createDecoratedText(template, new HashMap<>());
        text.setId("age");
        box.getChildren().add(text);

        if (member.getSpouseData().isPresent()) {
            ISpouseData spouseData = member.getSpouseData().get();
            template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.guild.GuildMembersDialog.married",
                    new Object[]{spouseData.getName(), spouseData.getLastName(), spouseData.getBirthPlace().getName()}, locale.getCurrentLocal());
            text = textFactory.createDecoratedText(template, new HashMap<>());
            text.setId("married");
            box.getChildren().add(text);
        } else {
            template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.guild.GuildMembersDialog.notMarried",
                    new Object[]{personalData.getAge(date.getCurrentDate())}, locale.getCurrentLocal());
            text = textFactory.createDecoratedText(template, new HashMap<>());
            text.setId("notMarried");
            box.getChildren().add(text);
        }
        box.getChildren().add(new PlaceHolder(1, 20));

        template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.guild.GuildMembersDialog.details",
                new Object[]{offices.size(), member.getFleet().size()}, locale.getCurrentLocal());
        text = textFactory.createDecoratedText(template, new HashMap<>());
        text.setId("details");
        box.getChildren().add(text);

        List<ICity> cities = map.getCities(member);
        ICity bestCity = city.getPlayer().getHometown(); // default
        long reputation = 0;
        for (ICity c : cities) {
            if (c.getReputation(member).getPopularity() > reputation) {
                bestCity = c;
                reputation = c.getReputation(member).getPopularity();
            }
        }

        template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.guild.GuildMembersDialog.reputation",
                new Object[]{bestCity.getName()}, locale.getCurrentLocal());
        text = textFactory.createDecoratedText(template, new HashMap<>());
        text.setId("reputation");
        box.getChildren().add(text);

        box.setLayoutX(50);
        box.setLayoutY(CLOSE_BTN_Y_POS - 500);
        getContent().addAll(box);

    }
}
