package ch.sahits.game.graphic.display.dialog.loaner;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.dialog.DefaultSimpleDialog;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.graphic.image.impl.SelectiveCachableXMLImageLoader;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.javafx.util.ModelTranslations;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.LoanerService;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.city.ILoanRequest;
import ch.sahits.game.openpatrician.model.city.impl.Debt;
import ch.sahits.game.openpatrician.model.city.impl.IDebt;
import ch.sahits.game.openpatrician.model.city.impl.LoanerState;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.EventHandler;
import javafx.scene.Node;
import javafx.scene.control.Label;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.StackPane;
import javafx.scene.text.Text;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.List;

/**
 * Dialog for giving a loan.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Jan 05, 2015
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class LoanerGiveLoanDialog extends DefaultSimpleDialog {
    public static final String DEFAULT_NON_SELECTED_STYLE = "-fx-fill: dimgray; -fx-text-fill: dimgray;";
    public static final String SELECTED_STYLE = "-fx-font-weight: bolder; -fx-fill: black;  -fx-text-fill: black;";


    @Autowired
    private LoanerService loanerService;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    @Qualifier("xmlImageLoader")
    private SelectiveCachableXMLImageLoader imageLoader;
    @Autowired
    private Date date;
    @Autowired
    private ClientViewState viewState;

    private LoanerState loaner;
    private final ICityPlayerProxyJFX city;
    @ObjectPropertyType(ILoanRequest.class)
    private ObjectProperty<ILoanRequest> selectedRequest = new SimpleObjectProperty<>(null);
    @ListType(Node.class)
    private List<Node> loans = new ArrayList<>();
    private OpenPatricianLargeWaxButton acceptBtn;
    private DecoratedText loanAgreement;
    @ListType(Double.class)
    private List<Double> interestLevels;
    private IntegerProperty selectedInterestLevel;

    public LoanerGiveLoanDialog(ICityPlayerProxyJFX city) {
        super(5);
        this.city = city;
    }

    @PostConstruct
    private void initializeDialog() {
        loaner = (LoanerState) loanerService.findLoaner(city.getCity());
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.giveLoan", new Object[]{}, locale.getCurrentLocal()));
        addDecoratedText("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.subHeader", "intro");
        addVerticalSpacer(20);
        int index = 0;
        for (final ILoanRequest request : loaner.getRequests()) {
            if (index < 4) {
                final ICitizen citizen = request.getCitizen();
                String rank = translator.getLocalDisplayName(citizen.getRank());
                final DecoratedText entry = addDecoratedText("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.request", "request"+index, rank, citizen.getName(), citizen.getLastName(), request.getSum(), request.getDurationInWeeks());
                loans.add(entry);
                entry.setOnMouseReleased(mouseEvent -> {
                    selectedRequest.setValue(request);
                    deactivateAll();
                    entry.applyStyle(SELECTED_STYLE);
                });
                entry.applyStyle(DEFAULT_NON_SELECTED_STYLE);
                index++;
            }
        } // End for requests
        acceptBtn = addActionButton("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.btnLabel", "giveLoanBtn");
        acceptBtn.setDisable(true);
        selectedRequest.addListener((observableValue, oldRequest, newRequest) -> {
            if (newRequest != null) {
                acceptBtn.setDisable(false);
            }
        });
        acceptBtn.setOnAction(createAcceptHandlerPage1());
    }
    private void deactivateAll() {
        for (Node node : loans) {
            if (node instanceof DecoratedText) {
                ((DecoratedText)node).applyStyle(DEFAULT_NON_SELECTED_STYLE);
            }
        }
    }

    private EventHandler<MouseEvent> createAcceptHandlerPage1() {
        return mouseEvent -> {
            clearContent();
            addDecoratedText("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.subHeader2", "introText");
            StackPane interestSelectionPane = new StackPane();
            Image minus = imageLoader.getImage("icons/minusIcon");
            Image plus = imageLoader.getImage("icons/plusIcon");
            Label reduceInterestLbl = new Label("", new ImageView(minus));
            Label increaseInterestLbl = new Label("", new ImageView(plus));
            interestLevels = new ArrayList<>();
            interestLevels.add(0.011); // 1.1%
            interestLevels.add(0.013); // 1.3%
            interestLevels.add(0.016); // 1.6%
            interestLevels.add(0.019); // 1.9%
            interestLevels.add(0.025); // 2.5%
            selectedInterestLevel = new SimpleIntegerProperty(2);
            final Text interstLbl = new Text();
            interstLbl.getStyleClass().add("dialogText");
            interstLbl.setText(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.interestLevel2", new Object[]{}, locale.getCurrentLocal()));
            BorderPane interestPane = new BorderPane();
            interestPane.setLeft(reduceInterestLbl);
            interestPane.setRight(increaseInterestLbl);
            interestPane.setCenter(interstLbl);
            interestSelectionPane.getChildren().add(interestPane);
            addNode(interestSelectionPane, "interest");

            loanAgreement = createLoanAggreement(interestLevels, selectedInterestLevel);


            reduceInterestLbl.setOnMouseReleased(mouseEvent1 -> {
                final int selectedIndex = selectedInterestLevel.get();
                if (selectedIndex > 0) {
                    selectedInterestLevel.setValue(selectedIndex - 1);
                    remove(loanAgreement);
                    loanAgreement = createLoanAggreement(interestLevels, selectedInterestLevel);
                    interstLbl.setText(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.interestLevel" + selectedInterestLevel.get(), new Object[]{}, locale.getCurrentLocal()));
                }
            });
            increaseInterestLbl.setOnMouseReleased(mouseEvent1 -> {
                final int selectedIndex = selectedInterestLevel.get();
                if (selectedIndex < interestLevels.size() - 1) {
                    selectedInterestLevel.setValue(selectedIndex + 1);
                    remove(loanAgreement);
                    loanAgreement = createLoanAggreement(interestLevels, selectedInterestLevel);
                    interstLbl.setText(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.interestLevel" + selectedInterestLevel.get(), new Object[]{}, locale.getCurrentLocal()));
                }
            });




            acceptBtn = addActionButton("ch.sahits.game.graphic.display.dialog.TavernBaseSideRoomPersonDialog.accept", "acceptBtn");
            acceptBtn.setOnAction(createAcceptHandlerPage2());
        };
    }

    private DecoratedText createLoanAggreement(List<Double> interestLevels, IntegerProperty selectedInterestLevel) {
        final double interestSmall = interestLevels.get(selectedInterestLevel.get());
        String interest = String.format("%.1f", interestSmall * 100);
        final int sum = selectedRequest.get().getSum();
        int paybacksum = (int) ((1 + interestSmall) * sum);

        return addDecoratedText("ch.sahits.game.graphic.display.dialog.LoanerGiveLoanDialog.loanAgreement", "loanAgreement", sum, selectedRequest.get().getDurationInWeeks(), interest, paybacksum);
    }

    private EventHandler<MouseEvent> createAcceptHandlerPage2() {
        return mouseEvent -> {
            final ILoanRequest request = selectedRequest.get();
            final double interest = interestLevels.get(selectedInterestLevel.get()) + 1;
            DateTime now = date.getCurrentDate();
            IDebt debt = Debt.builder()
                    .debitor(request.getCitizen())
                    .creditor(city.getPlayer())
                    .interest(interest)
                    .dueDate(now.plusWeeks(request.getDurationInWeeks()))
                    .amount(request.getSum())
                    .loanTakeOut(now)
                    .build();
            loaner.addDebt(debt);
            loaner.getRequests().remove(request);
            IReputation reputation = city.getCity().getReputation(city.getPlayer());
            reputation.update(request.getSum() / 1000 * 10);
            executeOnCloseButtonClicked();

        };
    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_LOANER);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }
}
