package ch.sahits.game.graphic.display.dialog.loaner;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.dialog.DefaultSimpleDialog;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.LoanerService;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.impl.ILoanProposal;
import ch.sahits.game.openpatrician.model.city.impl.LoanerState;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.EventHandler;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.List;

/**
 * Dialog for taking up a loan.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Jan 04, 2015
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class LoanerTakeLoanDialog extends DefaultSimpleDialog {
    public static final String DEFAULT_NON_SELECTED_STYLE = "-fx-fill: dimgray; -fx-text-fill: dimgray;";
    public static final String SELECTED_STYLE = "-fx-font-weight: bolder; -fx-fill: black;  -fx-text-fill: black;";

    @Autowired
    private ClientViewState viewState;
    @Autowired
    private LoanerService loanerService;

    private LoanerState loaner;
    private final ICityPlayerProxyJFX city;

    @ObjectPropertyType(ILoanProposal.class)
    private ObjectProperty<ILoanProposal> selectedProposal = new SimpleObjectProperty<>(null);
    @ListType(Node.class)
    private List<Node> loans = new ArrayList<>();

    public LoanerTakeLoanDialog(ICityPlayerProxyJFX city) {
        super(false);
        this.city = city;
    }
    @PostConstruct
    private void initializeDialog() {
        loaner = (LoanerState) loanerService.findLoaner(city.getCity());
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.takeLoan", new Object[]{}, locale.getCurrentLocal()));
        addDecoratedText("ch.sahits.game.graphic.display.dialog.LoanerTakeLoanDialog.dialogSubheading", "introText");
        for (final ILoanProposal proposal : loaner.getLoanProposals()) {
            int amount = proposal.getAmount();
            int payback = (int) (amount * proposal.getInterest());
            int duration = proposal.getDurationInWeeks();
            String interest = String.format("%.1f", (proposal.getInterest() - 1) * 100);
            final DecoratedText entry = addDecoratedText("ch.sahits.game.graphic.display.dialog.LoanerTakeLoanDialog.loanProposal", "proposal", amount, duration, payback, interest);
            loans.add(entry);
            entry.setOnMouseReleased(mouseEvent -> {
                selectedProposal.setValue(proposal);
                deactivateAll();
                entry.applyStyle(SELECTED_STYLE);
            });
            entry.applyStyle(DEFAULT_NON_SELECTED_STYLE);
        }

        final OpenPatricianLargeWaxButton acceptBtn =  addActionButton("ch.sahits.game.graphic.display.dialog.LoanerTakeLoanDialog.btnLabel", "actionBtn");
        acceptBtn.setDisable(true);
        selectedProposal.addListener((observableValue, oldProposal, newProposal) -> {
            if (newProposal != null) {
                acceptBtn.setDisable(false);
            }
        });
        acceptBtn.setOnAction(createAcceptHandler());
    }
    private void deactivateAll() {
        for (Node node : loans) {
            if (node instanceof DecoratedText) {
                ((DecoratedText)node).applyStyle(DEFAULT_NON_SELECTED_STYLE);
            }
        }
    }
    private EventHandler<MouseEvent> createAcceptHandler() {
        return mouseEvent -> {
            ILoanProposal proposal = selectedProposal.get();
            IPlayer debitor = city.getPlayer();
            loanerService.takeOutLoan(loaner, proposal, debitor, city.getCity());
            executeOnCloseButtonClicked();
        };
    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_LOANER);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }
}
