package ch.sahits.game.graphic.display.dialog.shipyard;

import ch.sahits.game.event.data.ShipyardOrderBuild;
import ch.sahits.game.javafx.control.BaleAmountAlwaysVisible;
import ch.sahits.game.javafx.control.BarrelAmount;
import ch.sahits.game.javafx.control.BarrelAmountAlwaysVisible;
import ch.sahits.game.javafx.control.CoinPriceAlwaysVisible;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.model.ControlTableCell;
import ch.sahits.game.javafx.model.ECellConstraint;
import ch.sahits.game.javafx.model.StaticTextTableCell;
import ch.sahits.game.javafx.model.Table;
import ch.sahits.game.javafx.model.TableHeader;
import ch.sahits.game.javafx.model.TableRow;
import ch.sahits.game.javafx.util.ModelTranslations;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.ShipFactory;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.IShipWeaponsLocation;
import ch.sahits.game.openpatrician.model.ship.impl.ShipWeaponsLocationFactory;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.scene.Group;
import javafx.scene.input.MouseEvent;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 13, 2013
 */
@Component("shipyardConstructionDialog")
@Scope(value = "prototype")
@ClassCategory({EClassCategory.DIALOG, EClassCategory.SINGLETON_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ShipyardConstructionDialogV2 extends BaseShipyardDialog {
    @Autowired
    private ShipFactory shipFactory;
    @Autowired
    private ShipWeaponsLocationFactory weaponsLocationFactory;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private Locale locale;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private MessageSource messageSource;

    private static final EWare[] MATERIALS = new EWare[]{EWare.WOOD,
                                                         EWare.CLOTH,
                                                         EWare.IRON,
                                                         EWare.HEMP,
                                                         EWare.PITCH};
    @MapType(key = IWare.class, value = ReadOnlyIntegerProperty.class)
    private HashMap<IWare, ReadOnlyIntegerProperty> availableMap = new HashMap<>();
    @MapType(key = IWare.class, value = IntegerBinding.class)
    private HashMap<IWare, IntegerBinding> priceMap = new HashMap<>();
    @MapType(key = IWare.class, value = BuyAmountBinding.class)
    private HashMap<IWare, BuyAmountBinding> buyAmountMap = new HashMap<>();

    public ShipyardConstructionDialogV2(ICityPlayerProxyJFX city) {
        super(city);
    }
    @PostConstruct
    private void init() {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.title", new Object[]{}, locale.getCurrentLocal()));
    }


    @Override
    protected EventHandler<MouseEvent> getAction() {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                ICity city = getCity();
                Optional<ITradingOffice> office = getPlayer().findTradingOffice(getCity());
                Preconditions.checkArgument(office.isPresent(), "Trading office must be present");
                for (EWare ware : MATERIALS) {
                    int buyAmount = buyAmountMap.get(ware).get();
                    city.move(ware, -buyAmount, getPlayer());
                    int needed =  shipFactory.getConstructionAmount(getCurrentShipType(), ware);
                    int fromWarhouse = needed - buyAmount;
                    office.get().move(ware, -fromWarhouse);
                }
                ShipyardOrderBuild event = new ShipyardOrderBuild(city.getCityState().getShipyardState(), getCurrentShipType(), getPlayer());
                clientServerEventBus.post(event);
                getPlayer().getCompany().updateCash(-calculateTotalBuildCosts());
                executeOnCloseButtonClicked();
            }
        };
    }

    @Override
    protected String getActionText() {
        return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.build", new Object[]{}, locale.getCurrentLocal());
    }

    @Override
    protected Group createFooterText() {
        IntegerBinding totalPriceBinding = new IntegerBinding() {
            {
                for (EWare ware : MATERIALS) {
                    super.bind(getPriceBinding(ware, null));
                }
            }
            @Override
            protected int computeValue() {
                return calculateTotalBuildCosts();
            }
        };
        Group g = new Group();
        String t = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.totalSum", new Object[]{}, locale.getCurrentLocal());
        HashMap<String, Object> parameters = new HashMap<>();
        parameters.put("price", totalPriceBinding.get());
        DecoratedText dt = decoratedTextFactory.createDecoratedText(t, parameters);
        g.getChildren().add(dt);
        return g;
    }

    private int calculateTotalBuildCosts() {
        int total = getCity().getCityState().getShipyardState().calculateConstructionCosts(getCurrentShipType());
        for (EWare ware : MATERIALS) {
            total += getPriceBinding(ware, null).get();
        }
        return total;
    }

    @Override
    protected Table createMainTable() {
        Table model = new Table();
        TableHeader header = new TableHeader(5);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.materials", new Object[]{}, locale.getCurrentLocal())), ECellConstraint.COLSPAN2);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.stored", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.buy", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionDialogV2.costs", new Object[]{}, locale.getCurrentLocal())));
        header.setAligenment(0, HPos.CENTER);
        header.setAligenment(2, HPos.CENTER);
        header.setAligenment(3, HPos.CENTER);
        header.setAligenment(4, HPos.CENTER);
        model.setHeader(header);
        model.setAligenment(0, HPos.RIGHT);
        model.setAligenment(1, HPos.RIGHT);
        model.setAligenment(2, HPos.RIGHT);
        model.setAligenment(3, HPos.RIGHT);
        model.setAligenment(4, HPos.RIGHT);

        model.setColumnWidth(70, 70, 70, 70, 100);
        for (final EWare ware : MATERIALS) {
            final TableRow row = new TableRow();
            row.add(new StaticTextTableCell(translator.getLocalDisplayName(ware)));

            final IntegerBinding needed = getNeededWareBinding(ware);
            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = new BarrelAmountAlwaysVisible();
                barrelAmount.amountProperty().bind(needed.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = new BaleAmountAlwaysVisible();
                baleAmount.amountProperty().bind(needed.asString());
                row.add(new ControlTableCell(baleAmount));
            }


            final ReadOnlyIntegerProperty stored = getStoredAmount(ware);
            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = new BarrelAmountAlwaysVisible();
                barrelAmount.amountProperty().bind(stored.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = new BaleAmountAlwaysVisible();
                baleAmount.amountProperty().bind(stored.asString());
                row.add(new ControlTableCell(baleAmount));
            }

            final BuyAmountBinding buyAmount = new BuyAmountBinding(ware, needed);
            buyAmountMap.put(ware, buyAmount);
            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = new BarrelAmountAlwaysVisible();
                barrelAmount.amountProperty().bind(buyAmount.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = new BaleAmountAlwaysVisible();
                baleAmount.amountProperty().bind(buyAmount.asString());
                row.add(new ControlTableCell(baleAmount));
            }

            IntegerBinding priceBinding = getPriceBinding(ware, buyAmount);
            CoinPriceAlwaysVisible price = new CoinPriceAlwaysVisible();
            price.amountProperty().bind(priceBinding.asString());
            row.add(new ControlTableCell(price));

            model.add(row);
        }
        return model;
    }

    /**
     * Retrieve the stored amount from the cached map. Put it into the map when necessary.
     * @param ware for which the stored amount should be retrieved
     * @return stored amount
     */
    private ReadOnlyIntegerProperty getStoredAmount(IWare ware) {
        final Optional<ITradingOffice> office = getPlayer().findTradingOffice(getCity());
        if (!availableMap.containsKey(ware) && office.isPresent()) {

            final ReadOnlyIntegerProperty stored = office.get().getWare(ware).amountProperty();
            availableMap.put(ware, stored);
        }
        return availableMap.get(ware);
    }

    /**
     * Retrieve the price from the cache map. Put it into the map when necessary.
     * @param ware for which the price should be retrieved
     * @param buyAmount that should be bought (required for the initialization)
     * @return price binding.
     */
    private IntegerBinding getPriceBinding(IWare ware, BuyAmountBinding buyAmount) {
        if (!priceMap.containsKey(ware)) {
            IntegerBinding priceBinding = new PriceBinding(ware, buyAmount);
            priceMap.put(ware, priceBinding);
        }
        return priceMap.get(ware);
    }
    private IntegerBinding getNeededWareBinding(final EWare ware) {
        return new IntegerBinding() {
            {
                super.bind(currentShipTypeIndexProperty());
            }
            @Override
            protected int computeValue() {
                return shipFactory.getConstructionAmount(getCurrentShipType(), ware);
            }
        };
    }

    @Override
    protected Table createTopTable() {
        int capacity = shipFactory.calculateInitialCapacity(getCurrentShipType(), getCity().getCoordinates().getX());
        IShipWeaponsLocation shipWeapons = weaponsLocationFactory.getShipWeaponsLocation(getCurrentShipType(), EShipUpgrade.NONE);
        Table model = new Table();
        // Add dummy header which is not rendered
        TableHeader header = new TableHeader(5);
        model.setHeader(header);
        model.setColumnWidth(70, 70, 70, 70, 70);
        TableRow row = new TableRow();
        BarrelAmount capacityAmount = new BarrelAmount();
        capacityAmount.setAmount(capacity);
        row.add(new ControlTableCell(capacityAmount));

        row.add(new StaticTextTableCell(String.valueOf(shipWeapons.getNumberOfSmallSlots())));
        row.add(new StaticTextTableCell(String.valueOf(shipWeapons.getNumberOfLargeSlots())));
        row.add(new StaticTextTableCell(String.valueOf(shipFactory.getShipSpeed(getCurrentShipType()))));
        row.add(new StaticTextTableCell(String.valueOf(shipFactory.getMinimalSailors(getCurrentShipType()))));
        model.add(row);
        return model;
    }

    @Override
    protected BooleanBinding actionEnabledBinding() {
        return new BooleanBinding() {
            {
                for (EWare ware : MATERIALS) {
                    ReadOnlyIntegerProperty stored = getStoredAmount(ware);
                    IntegerBinding price = getPriceBinding(ware, null);
                    ReadOnlyIntegerProperty available = getCity().getWare(ware).amountProperty();
                    super.bind(stored, price, available);
                }
                super.bind(getPlayer().getCompany().cashProperty());
            }
            @Override
            protected boolean computeValue() {
                long cash = getPlayer().getCompany().getCash();
                long buildCosts = calculateTotalBuildCosts();
                if (cash < buildCosts) {
                    return false;
                }
                for (EWare ware : MATERIALS) {
                    int storedAmount = getStoredAmount(ware).get();
                    int available = getCity().getWare(ware).getAmount();
                    int needed = shipFactory.getConstructionAmount(getCurrentShipType(), ware);
                    if (needed > storedAmount + available) {
                        return false;
                    }
                }
                return true;
            }
        };
    }
}
