package ch.sahits.game.graphic.display.dialog.tavern;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.event.data.ClockTickDayChange;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.graphic.display.dialog.util.DialogUtil;
import ch.sahits.game.graphic.display.dialog.util.TavernDialogUtil;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.FightingSkillAlwaysVisible;
import ch.sahits.game.javafx.control.NavigationSkillAlwaysVisible;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.javafx.control.TradingSkillAlwaysVisible;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.city.impl.TavernState;
import ch.sahits.game.openpatrician.model.event.PersonLeavesTavernEvent;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.impl.CaptainsState;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.geometry.HPos;
import javafx.geometry.Orientation;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.FlowPane;
import javafx.scene.layout.GridPane;
import javafx.scene.text.Text;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 20, 2013
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class TavernCaptainDialog extends CloseButtonDialog {
    private final ICityPlayerProxyJFX city;
    @ObjectPropertyType(IShip.class)
    private final ObjectProperty<IShip> currentShip;
    @Autowired
    private CaptainsState captainsState;
    @Autowired
    private DialogUtil dialogUtil;
    @Autowired
    private TavernDialogUtil tavernDialogUtil;
    @Autowired
    private ClientViewState viewState;
    private ICaptain captain;
    @Autowired
    private Locale locale;
    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    @XStreamOmitField
    private AsyncEventBus timerEventBus;
    @Autowired
    private MessageSource messageSource;
    private final TavernState tavernState;

    public TavernCaptainDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
        this.currentShip = new SimpleObjectProperty<>(this, "currentShip", null);
        if (city.getActiveShip() instanceof IShip) {
            IShip ship = (IShip) city.getActiveShip();
            currentShip.setValue(ship);
        }
        tavernState = city.getCity().getCityState().getTavernState();
    }
    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernCaptainDialog.title", new Object[]{}, locale.getCurrentLocal()));
        captain = captainsState.getCaptain(city.getCity()).get();
        Text intro = new Text();
        intro.getStyleClass().add("dialogText");
        intro.setWrappingWidth(WRAPPING_WIDTH);
        intro.setLayoutX(2*FRAME_BORDER);
        intro.setLayoutY(100);
        intro.setText(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernCaptainDialog.dialog1", new Object[]{captain.getName()}, locale.getCurrentLocal()));
        getContent().add(intro);

        GridPane pane1 = new GridPane();
        pane1.setLayoutX(2*FRAME_BORDER);
        pane1.setLayoutY(200);
        pane1.getColumnConstraints().addAll(new ColumnConstraints(100), new ColumnConstraints(50), new ColumnConstraints(50), new ColumnConstraints(50));
        String template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernCaptainDialog.age", new Object[]{captain.getAge()}, locale.getCurrentLocal());
        DecoratedText age = decoratedTextFactory.createDecoratedText(template, new HashMap<>());
        TradingSkillAlwaysVisible trading = new TradingSkillAlwaysVisible();
        NavigationSkillAlwaysVisible navigation = new NavigationSkillAlwaysVisible();
        FightingSkillAlwaysVisible fighting = new FightingSkillAlwaysVisible();
        trading.setAmount(captain.getTradingSkillLevel());
        navigation.setAmount(captain.getNavigationSkillLevel());
        fighting.setAmount(captain.getFightSkillLevel());
        pane1.add(age, 0, 0);
        pane1.add(trading, 1, 0);
        pane1.add(navigation, 2, 0);
        pane1.add(fighting, 3, 0);
        getContent().add(pane1);

        FlowPane salery = new FlowPane(Orientation.HORIZONTAL);
        salery.setLayoutX(2*FRAME_BORDER);
        salery.setLayoutY(250);
        String loadedText = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernCaptainDialog.price", new Object[]{}, locale.getCurrentLocal());
        HashMap<String, Object> parameters = new HashMap<>();
        parameters.put("salary", captain.getSalary());
        DecoratedText dt = decoratedTextFactory.createDecoratedText(loadedText, parameters);

        getContent().add(dt);

        if (currentShip.get() == null) {
            template = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernCaptainDialog.noShip", new Object[]{}, locale.getCurrentLocal());
            DecoratedText noShip = decoratedTextFactory.createDecoratedText(template, new HashMap<>());
            noShip.setLayoutX(2*FRAME_BORDER);
            noShip.setLayoutY(300);
            getContent().add(noShip);
        } else {
            GridPane shipCatalogue = createShipCatalogue();

            getContent().add(shipCatalogue);

            final int actionButtonX = (WIDTH - 124) / 2;
            final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.AutomaticTradingOverviewDialog.hire", new Object[]{}, locale.getCurrentLocal()));
            action.getStyleClass().add("actionButton");
            action.setOnAction(mouseEvent -> {
                currentShip.get().setCaptain(captain);
                captainsState.hireCaptain(captain, city.getCity());
                clientServerEventBus.post(new PersonLeavesTavernEvent(city.getCity(), captain));
                executeOnCloseButtonClicked();
            });
            action.setLayoutX(actionButtonX);
            action.setLayoutY(CLOSE_BTN_Y_POS - 24);
            BooleanBinding enableAction = new BooleanBinding() {
                {
                    super.bind(currentShip);
                }
                @Override
                protected boolean computeValue() {
                    final IShip iShip = currentShip.get();
                    return iShip != null
                            && !iShip.getCaptian().isPresent();
                }
            };
            action.setDisable(!enableAction.get());
            enableAction.addListener((observableValue, oldValue, newValue) -> {
                action.setDisable(!newValue);
            });
            getContent().add(action);
            timerEventBus.register(this);

        }

    }

    private GridPane createShipCatalogue() {
        GridPane shipCatalogue = new GridPane();
        shipCatalogue.setLayoutX(2*FRAME_BORDER);
        shipCatalogue.setLayoutY(300);
        shipCatalogue.getColumnConstraints().addAll(new ColumnConstraints(64), new ColumnConstraints(WIDTH - 4*FRAME_BORDER - 2* 64), new ColumnConstraints(64));
        BooleanBinding enablePrevNext = dialogUtil.enableShipCatalogueForShips(city);

        final OpenPatricianSmallWaxButton prevShip = new OpenPatricianSmallWaxButton("<");
        prevShip.setOnAction(dialogUtil.createPreviousActionForShips(city, currentShip));
        prevShip.setDisable(!enablePrevNext.get());
        shipCatalogue.add(prevShip, 0, 0);
        final OpenPatricianSmallWaxButton nextShip = new OpenPatricianSmallWaxButton(">");
        nextShip.setOnAction(dialogUtil.createNextActionForShips(city, currentShip));
        nextShip.setDisable(!enablePrevNext.get());
        shipCatalogue.add(nextShip, 2, 0);
        enablePrevNext.addListener((observableValue, oldValue, newValue) -> {
            nextShip.setDisable(!newValue);
            prevShip.setDisable(!newValue);
        });
        Text ship = new Text();
        ship.getStyleClass().add("dialogText");
        ship.textProperty().bind(new StringBinding() {
            {
                super.bind(currentShip);
            }
            @Override
            protected String computeValue() {
                return currentShip.get().getName();
            }
        });
        shipCatalogue.add(ship, 1, 0);
        GridPane.setHalignment(ship, HPos.CENTER);
        return shipCatalogue;
    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_TAVERN);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }

    @Override
    public void close() {
        super.close();
        timerEventBus.unregister(this);
        tavernDialogUtil.stopTalkingToPerson(tavernState, city.getPlayer());
    }

    @Subscribe
    public void handleDayChange(ClockTickDayChange dayChange) {
        Optional<ICaptain> captain = captainsState.getCaptain(city.getCity());
        if (!captain.isPresent()) {
            executeOnCloseButtonClicked();
        }

    }

}
