package ch.sahits.game.graphic.display.dialog.util;

import ch.sahits.game.javafx.control.BarrelAmount;
import ch.sahits.game.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.javafx.control.PlaceHolder;
import ch.sahits.game.javafx.model.Table;
import ch.sahits.game.javafx.util.JavaFXUtils;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.dialog.IDialog;
import ch.sahits.game.openpatrician.model.ship.IShip;
import com.google.common.base.Preconditions;
import javafx.beans.Observable;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.collections.ObservableList;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.geometry.Pos;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.control.Control;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.FlowPane;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.Pane;
import javafx.scene.layout.RowConstraints;
import javafx.scene.text.Text;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 20, 2013
 */
@Component
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class DialogUtil {
    @Autowired
    private JavaFXUtils fxUtils;
    /**
     * Boolean binding for the navigation enabling of the ships cataloque.
     * Only the ships are considered (convoys are excluded)
     * @param city proxy
     * @return binding.
     */
    public BooleanBinding enableShipCatalogueForShips(final ICityPlayerProxyJFX city) {
        return new BooleanBinding() {
            {
                super.bind(city.getPlayersShips());
            }
            @Override
            protected boolean computeValue() {
                return city.getPlayersShips().size() > 1;
            }
        };
    }

    /**
     * Navigate to the next ship in the catalogue.
     * Only the ships are considered (convoys are excluded)
     * @param city proxy
     * @param currentShip current ship
     * @return Event handler for the navigation.
     */
    public EventHandler<MouseEvent> createNextActionForShips(final ICityPlayerProxyJFX city,
                                                             final ObjectProperty<IShip> currentShip) {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                List<IShip> ships = city.getPlayersShips();
                if (ships.size() > 1) {
                    if (ships.get(ships.size()-1).equals(currentShip.get())) {
                        currentShip.set(ships.get(0)); // first one;
                    } else {
                        for (int i=0;i<ships.size()-1; i++) {
                            if (ships.get(i).equals(currentShip.get())) {
                                final IShip nextShip = ships.get(i + 1);
                                currentShip.set(nextShip);
                                break;
                            }
                        }
                    }
                }
            }
        };
    }
    /**
     * Navigate to the previous ship in the catalogue.
     * Only the ships are considered (convoys are excluded)
     * @param city proxy
     * @param currentShip current ship
     * @return Event handler for the navigation.
     */

    public EventHandler<MouseEvent> createPreviousActionForShips(final ICityPlayerProxyJFX city,
                                                                 final ObjectProperty<IShip> currentShip) {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                List<IShip> ships = city.getPlayersShips();
                if (ships.size() > 1) {
                    if (ships.get(0).equals(currentShip.get())) {
                        currentShip.set(ships.get(ships.size()-1)); // last one;
                    } else {
                        for (int i=1;i<ships.size(); i++) {
                            if (ships.get(i).equals(currentShip.get())) {
                                final IShip prevShip = ships.get(i - 1);
                                currentShip.set(prevShip);
                                break;
                            }
                        }
                    }
                }
            }
        };
    }

    /**
     * Create a grid pane based on a model.
     * @param model base model
     * @return grid pane
     */
    public GridPane createGridPaneFromModel(Table model) {
        return fxUtils.createGridPaneFromModel(model);
    }
    /**
     * Create the grid pane for the ship selection on three lines.
     * Only the ships are considered (convoys are excluded)
     * @param city proxy for the player city
     * @param currentShip non empty current ship
     * @return grid pane
     */
    public GridPane createShipSelection3LinesForShips(final ICityPlayerProxyJFX city, final ObjectProperty<IShip> currentShip) {
        GridPane shipSelectionPane = new GridPane();
        RowConstraints rowConstraint = new RowConstraints(24);
        shipSelectionPane.getRowConstraints().addAll(rowConstraint, rowConstraint, rowConstraint);
        ColumnConstraints colConstraint = new ColumnConstraints(64);
        shipSelectionPane.getColumnConstraints().addAll(colConstraint, colConstraint, colConstraint, colConstraint, colConstraint);
        Text shipName = new Text();
        StringBinding shipNameBinding = new StringBinding() {
            {
                super.bind(currentShip);
            }
            @Override
            protected String computeValue() {
                final IShip ship = currentShip.get();
                return ship.getShipType().name()+" "+ ship.getName();
            }
        };
        shipName.getStyleClass().add("dialogText");
        shipName.textProperty().bind(shipNameBinding);
        shipSelectionPane.add(shipName,1, 0, 3, 1);
        GridPane.setHalignment(shipName, HPos.CENTER);

        BooleanBinding enablePrevNext = enableShipCatalogueForShips(city);

        final OpenPatricianSmallWaxButton prevShip = new OpenPatricianSmallWaxButton("<");
        prevShip.getStyleClass().add("actionButton");
        prevShip.setOnAction(createPreviousActionForShips(city, currentShip));
        prevShip.setDisable(!enablePrevNext.get());
        shipSelectionPane.add(prevShip, 0, 1);
        final OpenPatricianSmallWaxButton nextShip = new OpenPatricianSmallWaxButton(">");
        nextShip.getStyleClass().add("actionButton");
        nextShip.setOnAction(createNextActionForShips(city, currentShip));
        nextShip.setDisable(!enablePrevNext.get());
        shipSelectionPane.add(nextShip, 4, 1);
        enablePrevNext.addListener(new ChangeListener<Boolean>() {
            @Override
            public void changed(ObservableValue<? extends Boolean> observableValue,
                                Boolean oldValue, Boolean newValue) {
                nextShip.setDisable(!newValue);
                prevShip.setDisable(!newValue);
            }
        });

        BarrelAmount shipSize = new BarrelAmount();
        StringBinding shipSizeBinding = new StringBinding() {
            {
                super.bind(currentShip);
            }
            @Override
            protected String computeValue() {
                return String.valueOf(currentShip.get().getSize());
            }
        };
        shipSize.amountProperty().bind(shipSizeBinding);
        shipSelectionPane.add(shipSize, 1, 2);
        GridPane.setHalignment(shipSize, HPos.CENTER);

        Text damage = new Text();
        damage.getStyleClass().add("dialogText");
        StringBinding damageBinding = new StringBinding() {
            {
                super.bind(currentShip);
            }
            @Override
            protected String computeValue() {
                return String.valueOf(100 - currentShip.get().getDamage());
            }
        };
        damage.textProperty().bind(damageBinding);
        shipSelectionPane.add(damage, 2, 2);
        GridPane.setHalignment(damage, HPos.CENTER);

        Text sailorsOnShip2 = new Text();
        sailorsOnShip2.getStyleClass().add("dialogText");
        final SailorOnShipBinding sailorsOnShipBinding = new SailorOnShipBinding(currentShip);
        currentShip.addListener(new ChangeListener<IShip>() {
            @Override
            public void changed(ObservableValue<? extends IShip> observableValue,
                                IShip oldValue, IShip newValue) {
                sailorsOnShipBinding.delegateUnbind(oldValue.numberOfSailorsProperty());
                sailorsOnShipBinding.delegateBind(newValue.numberOfSailorsProperty());
            }
        });
        sailorsOnShip2.textProperty().bind(sailorsOnShipBinding);
        shipSelectionPane.add(sailorsOnShip2, 3, 2);
        GridPane.setHalignment(sailorsOnShip2, HPos.CENTER);
        return shipSelectionPane;
    }
    private class SailorOnShipBinding extends StringBinding {
        private final ObjectProperty<IShip> currentShip;
        SailorOnShipBinding(ObjectProperty<IShip> currentShip) {
            super.bind(currentShip, currentShip.get().numberOfSailorsProperty());
            this.currentShip = currentShip;
        }
        @Override
        protected String computeValue() {
            final IShip ship = currentShip.get();
            StringBuilder sb = new StringBuilder();
            sb.append(ship.getNumberOfSailors())
                    .append(" (")
                    .append(ship.getMinNumberOfSailors())
                    .append(" - ")
                    .append(ship.getMaxNumberOfSailors())
                    .append(")");
            return sb.toString();
        }
        public void delegateUnbind(Observable...values) {
            unbind(values);
        }
        public void delegateBind(Observable...values) {
            bind(values);
        }
    }

    /**
     * Create an element that will layout the <code>childNode</code>
     * horizontally centered within a Dialog.
     * @param childNode that should be centered
     * @return wrapping Pane
     */
    public Pane center(Control childNode) {
        final int width = IDialog.WRAPPING_WIDTH - 24;
        Pane g = new Pane(childNode);
        childNode.widthProperty().addListener((observable, oldValue, newValue) -> {
            double inset = (width - childNode.getWidth())/2;
            g.setLayoutX(inset);
        });
        return g;
    }
    /**
     * Create an element that will layout the <code>childNode</code>
     * horizontally centered within a Dialog.
     * @param childNode that should be centered
     * @return wrapping Pane
     */
    public Pane center(ImageView childNode) {
        final int width = IDialog.WRAPPING_WIDTH - 24;
        Pane g = new Pane(childNode);
        double in = (width - childNode.getImage().getWidth())/2;
        g.setLayoutX(in);
        childNode.getImage().widthProperty().addListener((observable, oldValue, newValue) -> {
            double inset = (width - childNode.getImage().getWidth())/2;
            g.setLayoutX(inset);
        });
        return g;
    }
    /**
     * Create an element that will layout the <code>childNode</code>
     * horizontally centered within a Dialog.
     * @param childNode that should be centered
     * @return wrapping Pane
     */
    public Group center(Pane childNode) {
        final int width = IDialog.WRAPPING_WIDTH - 24;
        Group g = new Group(childNode);
        childNode.widthProperty().addListener((observable, oldValue, newValue) -> {
            double inset = (width - childNode.getWidth())/2;
            g.setLayoutX(inset);
        });
        if (childNode instanceof FlowPane) {
            ((FlowPane) childNode).setAlignment(Pos.CENTER);
        }
        return g;
    }

    /**
     * Create an element that will layout the <code>childNode</code>
     * at the right dialog content border.
     * @param childNode that should be right aligned.
     * @return wrapping Pane
     */
    public Group right(Pane childNode) {
        final int width = IDialog.WRAPPING_WIDTH - 24;
        Group g = new Group(childNode);
        childNode.widthProperty().addListener((observable, oldValue, newValue) -> {
            double inset = (width - childNode.getWidth());
            g.setLayoutX(inset);
        });
        if (childNode instanceof FlowPane) {
            ((FlowPane) childNode).setAlignment(Pos.CENTER_RIGHT);
        }
        return g;
    }

    /**
     * Create an element that will layout the <code>childNode</code>
     * at the right dialog content border.
     * @param childNode that should be right aligned.
     * @return wrapping Pane
     */
    public Pane right(Control childNode) {
        final int width = IDialog.WRAPPING_WIDTH - 24;
        Pane g = new Pane(childNode);
        childNode.widthProperty().addListener((observable, oldValue, newValue) -> {
            double inset = (width - childNode.getWidth());
            g.setLayoutX(inset);
        });
        return g;
    }

    /**
     * Create a Node that forces vertical spacing.
     * @param heigth of the spacing node.
     * @return Node with specified <code>heigth</code>.
     */
    public Node createVerticalSpacer(int heigth) {
        return new PlaceHolder(1, heigth);
    }

    /**
     * Remove a node identified by <code>id</code> from the parent
     * and return the index.
     * @param parent containing the node
     * @param id of the id
     * @return index in the children of the removed node within id. If the node
     * cannot be found -1 will be returned.
     */
    public int removeById(Pane parent, String id) {
        Preconditions.checkNotNull(id, "The identifying id must not be null");
        ObservableList<Node> children = parent.getChildren();
        int index = -1;
        for (int i = 0; i < children.size(); i++) {
            Node node = children.get(i);
            if (id.equals(node.getId())) {
                index = i;
                break;
            }
        }
        if (index >= 0) {
            children.remove(index);
        }
        return index;
    }
}
