package ch.sahits.game.graphic.display.scene;

import ch.sahits.game.event.EGameStatusChange;
import ch.sahits.game.event.GameStateChange;
import ch.sahits.game.event.data.NewGame;
import ch.sahits.game.graphic.display.SceneChangeable;
import ch.sahits.game.graphic.display.util.GameOptionsService;
import ch.sahits.game.javafx.OpenPatricianScene;
import ch.sahits.game.javafx.control.OpenPatricianRadioButton;
import ch.sahits.game.javafx.control.OpenPatricianSlider;
import ch.sahits.game.javafx.control.OpenPatricianSpinner;
import ch.sahits.game.javafx.control.OpenPatricianStoneButton;
import ch.sahits.game.javafx.control.OpenPatricianWoodenTextInput;
import ch.sahits.game.javafx.model.OpenPatricianSpinnerOptionDataModel;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.data.map.Cities;
import ch.sahits.game.openpatrician.data.map.City;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.Difficulty;
import ch.sahits.game.openpatrician.model.EGameSpeed;
import ch.sahits.game.openpatrician.model.EObjective;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.GameFactory;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.CityFactory;
import ch.sahits.game.openpatrician.model.city.EKontorType;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.ShipFactory;
import ch.sahits.game.openpatrician.server.MapProviderService;
import ch.sahits.game.openpatrician.util.RandomNameLoader;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import ch.sahits.game.util.UIFactory;
import com.google.common.eventbus.AsyncEventBus;
import javafx.collections.FXCollections;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.geometry.Pos;
import javafx.scene.control.Label;
import javafx.scene.control.ToggleGroup;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.StackPane;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.MessageSource;
import org.springframework.oxm.Unmarshaller;

import javax.annotation.PostConstruct;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * Scene for starting a new game.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Oct 26, 2013
 *
 */
@Prototype
@ClassCategory(EClassCategory.PROTOTYPE_BEAN)
public class NewGameScene extends OpenPatricianScene {
	private final static Logger logger = LogManager.getLogger(NewGameScene.class);

	private RandomNameLoader firstNameLoader;
	private RandomNameLoader lastNameLoader;
	// Need to inject the font as for custom controls it is not stylable
	private OpenPatricianSpinner objective;
	private OpenPatricianSlider speed;
	private OpenPatricianSlider difficulty;
	private OpenPatricianSpinner startYear;
	private OpenPatricianSpinner hometown;
	private OpenPatricianSpinner map;
	private OpenPatricianRadioButton femaleRadioButton;
	private OpenPatricianRadioButton maleRadioButton;
	private OpenPatricianWoodenTextInput lastName;
	private OpenPatricianWoodenTextInput name;
	@Autowired
	private CityFactory cityFactory;
	@Autowired
	private GameFactory gameFactory;
	@Autowired
	private ShipFactory shipFactory;
	private SceneChangeable sceneChangeable;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired()
    @Qualifier("jaxb2MapMarshaller")
    private Unmarshaller unmarshaller;
	@Autowired
	private Locale locale;
	@Autowired
	private Date gameDate;
	@Autowired
	private UIFactory uiFactory;
	@Autowired
	private GameOptionsService gameOptions;
	@Autowired
	private MessageSource messageSource;
	@Autowired
	private MapProviderService mapProviderService;
	@Autowired
	private SceneChangeService sceneChanger;

	public NewGameScene(@Value("firstNameLoader") RandomNameLoader firstNameLoader, @Value("lastNameLoader") RandomNameLoader lastNameLoader) {
		super(new StackPane());
		this.firstNameLoader = firstNameLoader;
		this.lastNameLoader = lastNameLoader;
		
		getRoot().getStylesheets().add(this.getClass().getResource(getStyleSheetFilename()).toExternalForm());
    }
    @PostConstruct
	private void createControls() {
        StackPane root = (StackPane) getRoot();
		root.setId("NewGameStackPane");

		GridPane grid = new GridPane();
		grid.getStyleClass().add("grid");
		grid.setAlignment(Pos.CENTER);
		Label nameLbl = new Label(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.name", new Object[]{}, locale.getCurrentLocal()));
		grid.add(nameLbl, 0, 0);
		
		name = new OpenPatricianWoodenTextInput("");
		name.setText(firstNameLoader.getRandomName());
		name.setSize(25);
		grid.add(name, 1, 0);
		
		Label lastNameLbl = new Label(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.lastName", new Object[]{}, locale.getCurrentLocal()));
		grid.add(lastNameLbl, 3, 0);
		
		lastName = new OpenPatricianWoodenTextInput("");
		lastName.setText(lastNameLoader.getRandomName());
		lastName.setSize(25);
		grid.add(lastName, 4, 0);
		
		ToggleGroup toggleGroup = new ToggleGroup();
		Label sexLbl = new Label(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.sex", new Object[]{}, locale.getCurrentLocal()));
		grid.add(sexLbl, 0, 1);
		
		
		maleRadioButton = new OpenPatricianRadioButton(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.male", new Object[]{}, locale.getCurrentLocal()));
		maleRadioButton.setSelected(true);
		maleRadioButton.setToggleGroup(toggleGroup);
		maleRadioButton.setUnselectedLabel(messageSource.getMessage("no", new Object[]{}, locale.getCurrentLocal()));
		maleRadioButton.setSelectedLabel(messageSource.getMessage("yes", new Object[]{}, locale.getCurrentLocal()));
//		maleRadioButton.setAlignment(Pos.CENTER_LEFT);
		grid.add(maleRadioButton, 1, 1);
		GridPane.setHalignment(maleRadioButton, HPos.LEFT);

		femaleRadioButton = new OpenPatricianRadioButton(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.female", new Object[]{}, locale.getCurrentLocal()));
		femaleRadioButton.setSelected(false);
		femaleRadioButton.setToggleGroup(toggleGroup);
		femaleRadioButton.setUnselectedLabel(messageSource.getMessage("no", new Object[]{}, locale.getCurrentLocal()));
		femaleRadioButton.setSelectedLabel(messageSource.getMessage("yes", new Object[]{}, locale.getCurrentLocal()));
		grid.add(femaleRadioButton, 3, 1);
		GridPane.setHalignment(femaleRadioButton, HPos.LEFT);
		
		Label mapLbl = new Label(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.map", new Object[]{}, locale.getCurrentLocal()));
		grid.add(mapLbl, 0, 2);
		
		map = new OpenPatricianSpinner();
		map.setMaxWidth(200);
		map.setOptions(FXCollections.observableArrayList(getMapList()));
		map.selectedIndexProperty().set(0);
		map.selectedIndexProperty().addListener((observable, oldValue, newValue) -> {
			hometown.setOptions(FXCollections.observableArrayList(getHomeTowns4Map()));
		});
		grid.add(map, 1, 2);
		GridPane.setHalignment(map, HPos.LEFT);
		
		Label hometownLbl = new Label(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.hometown", new Object[]{}, locale.getCurrentLocal()));
		grid.add(hometownLbl, 3, 2);

		hometown = new OpenPatricianSpinner();
		hometown.setMaxWidth(200);
		hometown.setOptions(FXCollections.observableArrayList(getHomeTowns4Map()));
        hometown.selectedIndexProperty().setValue(0);
        map.selectedIndexProperty().addListener((observableValue, number, number2) -> {
            hometown.setOptions(FXCollections.observableArrayList(getHomeTowns4Map()));
            hometown.selectedIndexProperty().set(0);
        });
		grid.add(hometown, 4, 2);
		GridPane.setHalignment(hometown, HPos.LEFT);
		
		Label startYearLbl = new Label(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.startYear", new Object[]{}, locale.getCurrentLocal()));
		grid.add(startYearLbl, 0, 3);
		
		startYear = new OpenPatricianSpinner();
		startYear.setMaxWidth(200);
        final List<OpenPatricianSpinnerOptionDataModel> startYears = getStartYears();
        startYear.setOptions(FXCollections.observableArrayList(startYears));
        startYear.selectedIndexProperty().set(50);
		grid.add(startYear, 1, 3);
		GridPane.setHalignment(startYear, HPos.LEFT);
		
		Label difficultyLbl = new Label(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.difficulty", new Object[]{}, locale.getCurrentLocal()));
		grid.add(difficultyLbl, 0, 4);
		
		difficulty = new OpenPatricianSlider(200);
		difficulty.getStyleClass().addAll("defaultTextSize24");
		difficulty.setValues(getDificutlyList());
		difficulty.selectedIndexProperty().setValue(0);
		grid.add(difficulty, 1, 4);
		GridPane.setHalignment(difficulty, HPos.LEFT);
		
		Label gameSpeedLbl = new Label(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.gameSpeed", new Object[]{}, locale.getCurrentLocal()));
		grid.add(gameSpeedLbl, 3, 4);
		
		speed = new OpenPatricianSlider(200);
        speed.getStyleClass().addAll("defaultTextSize24");
		speed.setValues(gameOptions.getGameSpeedList());
		speed.selectedIndexProperty().setValue(1);
		grid.add(speed, 4, 4);
		GridPane.setHalignment(speed, HPos.LEFT);
		
		Label objectiveLbl = new Label(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.objective", new Object[]{}, locale.getCurrentLocal()));
		grid.add(objectiveLbl, 0, 5);
		
		objective = new OpenPatricianSpinner();
		objective.setMaxWidth(200);
		objective.setOptions(FXCollections.observableArrayList(getObjectiveList()));
        objective.selectedIndexProperty().setValue(0);
		grid.add(objective, 1, 5);
		GridPane.setHalignment(objective, HPos.LEFT);
		
		OpenPatricianStoneButton button = new OpenPatricianStoneButton(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.startGame", new Object[]{}, locale.getCurrentLocal()));
		button.setOnAction(new NewGameAction());
		grid.add(button, 1, 6);
		GridPane.setHalignment(button, HPos.LEFT);
		
		
		root.getChildren().add(grid);
	}
	public SceneChangeable getSceneChangeable() {
		return sceneChangeable;
	}

	public void setSceneChangeable(SceneChangeable sceneChangeable) {
		this.sceneChangeable = sceneChangeable;
	}

	
	/**
	 * Retrieve the list of the available maps
	 * @return
	 */
	private List<OpenPatricianSpinnerOptionDataModel> getMapList() {
		Map<String, String> maps = mapProviderService.getMaps();
		List<OpenPatricianSpinnerOptionDataModel> l = new ArrayList<>();
		for (Entry<String, String> entry : maps.entrySet()) {
			l.add(new OpenPatricianSpinnerOptionDataModel(messageSource.getMessage(entry.getKey(), new Object[]{}, locale.getCurrentLocal()), entry.getValue()));

		}
		return l;
	}
	/**
	 * Create a list of town names that are available on the selected map as home towns.
	 * @return
	 */
	private List<OpenPatricianSpinnerOptionDataModel> getHomeTowns4Map() {
		List<OpenPatricianSpinnerOptionDataModel>  l = new ArrayList<>();
		final String selectedValue = map.getSelectedValue();
		try {
            // CityFactory is not yet initialized as the map data can change
			logger.debug("Update home towns for map "+selectedValue);
            ch.sahits.game.openpatrician.data.map.Map mapData = (ch.sahits.game.openpatrician.data.map.Map) unmarshaller.unmarshal(getSourceFromFile(selectedValue));
            Cities cities = mapData.getCities();
            for (City city : cities.getCity()) {
				if (city.getKontorType().equals(EKontorType.KONTOR.name())) {
					String cityName = messageSource.getMessage(city.getName(), new Object[0], locale.getCurrentLocal());
					l.add(new OpenPatricianSpinnerOptionDataModel(cityName, city.getName()));
				}
            }
        } catch (IOException e) {
            logger.error("Could not read city data from the selected map: "+ selectedValue);
        }
		return l;
	}
    private Source getSourceFromFile(String fileName) {
        final InputStream resourceAsStream = getClass().getResourceAsStream(fileName);
        return new StreamSource(resourceAsStream);
    }
	/**
	 * Initialize the list of start years from 1300 to 1400
	 * @return
	 */
	private List<OpenPatricianSpinnerOptionDataModel> getStartYears() {
		List<OpenPatricianSpinnerOptionDataModel> l = new ArrayList<>();
		for (int year=1400;year>=1300;year--){
            final String y = String.valueOf(year);
			l.add(new OpenPatricianSpinnerOptionDataModel(y));
		}
		return l;
	}
	/**
	 * Compile a list of the difficulty levels
	 * @return
	 */
	private List<String> getDificutlyList() {
		ArrayList<String> l = new ArrayList<String>();
		l.add(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.CHANDLER", new Object[]{}, locale.getCurrentLocal()));
		l.add(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.MERCHANT", new Object[]{}, locale.getCurrentLocal()));
		l.add(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.TRADESMAN", new Object[]{}, locale.getCurrentLocal()));
		l.add(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.COUNCILMAN", new Object[]{}, locale.getCurrentLocal()));
		l.add(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.PATRICIAN", new Object[]{}, locale.getCurrentLocal()));
		return l;
	}

	/**
	 * Create a list of objectives to select from
	 * @return
	 */
	private List<OpenPatricianSpinnerOptionDataModel> getObjectiveList() {
		List<OpenPatricianSpinnerOptionDataModel> l = new ArrayList<>();
		l.add(new OpenPatricianSpinnerOptionDataModel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.objective1", new Object[]{}, locale.getCurrentLocal())));
		l.add(new OpenPatricianSpinnerOptionDataModel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.objective2", new Object[]{}, locale.getCurrentLocal())));
		l.add(new OpenPatricianSpinnerOptionDataModel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.objective3", new Object[]{}, locale.getCurrentLocal())));
		l.add(new OpenPatricianSpinnerOptionDataModel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.objective4", new Object[]{}, locale.getCurrentLocal())));
		l.add(new OpenPatricianSpinnerOptionDataModel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.objective5", new Object[]{}, locale.getCurrentLocal())));
		l.add(new OpenPatricianSpinnerOptionDataModel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.objective6", new Object[]{}, locale.getCurrentLocal())));
		l.add(new OpenPatricianSpinnerOptionDataModel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.objective7", new Object[]{}, locale.getCurrentLocal())));
		l.add(new OpenPatricianSpinnerOptionDataModel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.objective8", new Object[]{}, locale.getCurrentLocal())));
		l.add(new OpenPatricianSpinnerOptionDataModel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.objective9", new Object[]{}, locale.getCurrentLocal())));
		l.add(new OpenPatricianSpinnerOptionDataModel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.NewGameScene.objective10", new Object[]{}, locale.getCurrentLocal())));
		return l;
	}
	private String getStyleSheetFilename() {
		return "newGame.css";
	}
	
	private class NewGameAction implements EventHandler<MouseEvent>{
		/**
		 * check if the input is valid and one may proceed.
		 * @return
		 */
		private boolean validInput(){
			if (name.getText()==null || name.getText().trim().equals("")){
				logger.warn("Name may not be null");
				return false;
			}
			if (lastName.getText()==null || lastName.getText().trim().equals("")){
				logger.warn("Last name may not be null");
				return false;
			}
			return true;
		}

		@Override
		public void handle(MouseEvent mouseEvent) {
			if (validInput()){
                NewGame newGameDTO = NewGame.builder()
                        .difficulty(Difficulty.fromIndex(difficulty.getSelectedIndex()))
                        .firstName(name.getText())
                        .lastName(lastName.getText())
                        .hometown(hometown.getSelectedValue())
                        .objective(EObjective.values()[objective.getSelectedIndex()])
                        .speed(EGameSpeed.values()[speed.getSelectedIndex()])
                        .startYear(Integer.parseInt(startYear.getSelectedValue()))
                        .male(maleRadioButton.isSelected() && !femaleRadioButton.isSelected())
                        .mapName(map.getSelectedValue())
						.singleplayer(true)
						.environmentInitialisation(true)
                        .build();
                // notify server
                GameStateChange stateChange = new GameStateChange(EGameStatusChange.NEW_GAME);
                stateChange.setStateChangeData(newGameDTO);
				clientServerEventBus.post(stateChange);

                // Request the view change
				while (gameDate.getStartYear() == null) {
					try {
						Thread.sleep(1);
					} catch (InterruptedException e) {
						logger.warn("Interrupted while waiting for the startyear to be set.");
					}
				}
				// todo: andi 8/15/15: wait until all components are initialized
				final double width = getRoot().getWidth();
				final double height = getRoot().getHeight();
				sceneChanger.changeScene(sceneChangeable, uiFactory.getMainGameScene(width, height));

			} // end if valid input

		}
	}
}
