package ch.sahits.game.openpatrician.display.dialog;

import ch.sahits.game.openpatrician.display.javafx.MainGameView;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.openpatrician.javafx.control.PlaceHolder;
import ch.sahits.game.openpatrician.javafx.dialog.Dialog;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import javafx.application.Platform;
import javafx.beans.property.SimpleStringProperty;
import javafx.beans.property.StringProperty;
import javafx.scene.Node;
import javafx.scene.control.Label;
import javafx.scene.layout.StackPane;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Basic dialog with a close button and a title.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Nov 15, 2013
 *
 */
public abstract class CloseButtonDialog extends Dialog implements ICloseButtonDialog{
	private final Logger logger = LogManager.getLogger(getClass());
	@Autowired
    @Qualifier("clientEventBus")
    protected AsyncEventBus clientEventBus;
	@Autowired
	private ParentNodeProvider parentNodeProvider;

	private StringProperty title;
	private StringProperty closeLabel;
    private final OpenPatricianSmallWaxButton closeButton;

    public CloseButtonDialog() {
		super();
		title = new SimpleStringProperty(this,"title", "");
		closeLabel = new SimpleStringProperty(this,"closeLabel", "");
		Label titleLbl = new Label();
		titleLbl.getStyleClass().add("dialogTitle");
		titleLbl.textProperty().bind(title);
		StackPane titlePane = new StackPane();
		titlePane.setLayoutY(2*FRAME_BORDER);
		titlePane.getChildren().addAll(new PlaceHolder(WIDTH, 1), titleLbl);
		getContent().add(titlePane);
        closeButton = new OpenPatricianSmallWaxButton("X");
		closeButton.getStyleClass().add("closeButton");
		closeButton.setLayoutX(WIDTH - CLOSE_BUTTON_WIDTH - CLOSE_BUTTON_PADDING);
		closeButton.setLayoutY(CLOSE_BTN_Y_POS);
		closeButton.setOnAction(event -> {
			try {
				executeOnCloseButtonClicked();
			} catch (RuntimeException e) {
				logger.error("Failed to close dialog", e);
			}
		});
		closeButton.getStyleClass().add("buttonLabel");
		getContent().add(closeButton);
	}
	
	public StringProperty titleProperty() {
		return title;
	}
	public String getTitle() {
		return title.get();
	}
	public void setTitle(String title) {
		this.title.set(title);
	}
    protected OpenPatricianSmallWaxButton getCloseButton() {
        return closeButton;
    }

	public void removeDecoratedContent() {
        final List<DecoratedText> removables = new ArrayList<>();
		for (Iterator<Node> iterator = getContent().iterator(); iterator.hasNext(); ) {
			final Node child = iterator.next();
			if (child instanceof DecoratedText) {
                removables.add((DecoratedText) child);
			}
		}
        Platform.runLater(() -> {
            getContent().removeAll(removables);
        });
	}
	public StringProperty closeLabelProperty() {
		return closeLabel;
	}
	public String getCloseLabel() {
		return closeLabel.get();
	}
	public void setCloseLabel(String closeLabel) {
		this.closeLabel.set(closeLabel);
	}
	/**
	 * Add the additional buttons evenly spaced to the left of the close button.
	 * @param buttons
	 */
	public void addButtomControlButtons(OpenPatricianSmallWaxButton...buttons) {
		double availableSpace = WIDTH - CLOSE_BUTTON_WIDTH - 2*CLOSE_BUTTON_PADDING;
		double widthButtonAndGap = availableSpace/buttons.length;
		Preconditions.checkState(widthButtonAndGap >= CLOSE_BUTTON_WIDTH, "Cannot draw buttons as there is not enough space");
		double gap = widthButtonAndGap - CLOSE_BUTTON_WIDTH;
		double x = CLOSE_BUTTON_PADDING;
		for (OpenPatricianSmallWaxButton btn : buttons) {
			btn.setLayoutX(x);
			btn.setLayoutY(HEIGHT - CLOSE_BUTTON_HEIGHT - CLOSE_BUTTON_PADDING);
			x += CLOSE_BUTTON_WIDTH + gap;
			getContent().add(btn);
		}
		
	}

	public void executeOnCloseButtonClicked() {
		MainGameView mainGameView = (MainGameView) parentNodeProvider.getParent(this);
		if (mainGameView != null) { // In cases where the code is not properly cleaned up this may be null while the dialog is closed already.
			mainGameView.closeDialog();
		}
    }


}
