package ch.sahits.game.openpatrician.display.dialog;

import ch.sahits.game.openpatrician.display.dialog.service.DialogUtil;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import javafx.geometry.Pos;
import javafx.scene.Node;
import javafx.scene.control.Control;
import javafx.scene.layout.Pane;
import javafx.scene.layout.VBox;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import java.util.HashMap;

/**
 * Base dialog that has content layouted in a VBox style and can have
 * one action button.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Dec 05, 2016
 */
public abstract class DefaultSimpleDialog extends CloseButtonDialog {
    @Autowired
    private DialogUtil dialogHelper;
    @Autowired
    protected MessageSource messageSource;
    @Autowired
    protected Locale locale;
    @Autowired
    private DecoratedTextFactory textFactory;

    private final VBox box;

    public DefaultSimpleDialog() {
        this(false);
    }

    public DefaultSimpleDialog(boolean withoutTitle) {
        box = new VBox();
        box.setLayoutX(INSET);
        if (withoutTitle) {
            box.setLayoutY(CLOSE_BTN_Y_POS - 580);
        } else {
            box.setLayoutY(CLOSE_BTN_Y_POS - 500);
        }
        getContent().add(box);
    }
    public DefaultSimpleDialog(int spacing) {
        this(false);
        box.setSpacing(spacing);
    }

    /**
     * Add a decorated text.
     * @param textKey key of the text
     * @param id to be used on the decorated text node, if it is not null
     * @param textParameters text arguments.
     * @return decorated text element that was added.
     */
    public DecoratedText addDecoratedText(String textKey, String id, Object... textParameters) {
        String template = messageSource.getMessage(textKey,
                textParameters, locale.getCurrentLocal());
        DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
        addNode(text, id);
        return text;
    }

    /**
     * Add a generic node.
     * @param node
     * @param id to be used on the node, if it is not null
     */
    public void addNode(Node node, String id) {
        box.getChildren().add(node);
        if (id != null) {
            node.setId(id);
        } else {
            node.setId("");
        }
    }

    /**
     * Add a decorated text with alignement <code>CENTER_LEFT</code>, <code>CENTER</code> or
     * <code>CENTER_RIGHT</code>.
     * @param textKey key of the text
     * @param alignment of either <code>CENTER_LEFT</code>, <code>CENTER</code> or
     * <code>CENTER_RIGHT</code>.
     * @param id to be used on the decorated text node, if it is not null
     * @param textParameters text arguments.
     */
    public void addDecoratedText(String textKey, String id, Pos alignment, Object... textParameters) {
        String template = messageSource.getMessage(textKey,
                textParameters, locale.getCurrentLocal());
        DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
        addNode(text, id, alignment);
    }

    /**
     * Add a node with alignement <code>CENTER_LEFT</code>, <code>CENTER</code> or
     * <code>CENTER_RIGHT</code>.
     * @param node to be layouted
     * @param alignment of either <code>CENTER_LEFT</code>, <code>CENTER</code> or
     * <code>CENTER_RIGHT</code>.
     */
    public void addNode(Node node, String id, Pos alignment) {
        if (alignment == Pos.CENTER_LEFT) {
            addNode(node, id);
            return;
        }
        if (id != null) {
            node.setId(id);
        } else {
            node.setId("");
        }
        if (node instanceof Pane) {
            if (alignment == Pos.CENTER) {
                addNode(dialogHelper.center((Pane)node), null);
            } else if (alignment == Pos.CENTER_RIGHT){
                 addNode(dialogHelper.right((Pane)node), null);
            } else {
                throw new IllegalArgumentException("Alignment '"+alignment+"' is not supported");
            }
        } else if (node instanceof Control) {
            if (alignment == Pos.CENTER) {
                addNode(dialogHelper.center((Control)node), null);
            } else if (alignment == Pos.CENTER_RIGHT){
                addNode(dialogHelper.right((Control)node), null);
            } else {
                throw new IllegalArgumentException("Alignment '"+alignment+"' is not supported");
            }
        } else {
            throw new IllegalArgumentException("Can only add Panes and Controlls in a layouted form");
        }
    }

    /**
     * Add a vertical spacer.
     * @param heigth
     */
    public void addVerticalSpacer(int heigth) {
        box.getChildren().add(dialogHelper.createVerticalSpacer(heigth));
    }

    /**
     * Add the action button centered at the bottom.
     * @param textKey for the label on the button
     * @param  id for the button
     * @return button
     */
    public OpenPatricianLargeWaxButton addActionButton(String textKey, String id) {
        OpenPatricianLargeWaxButton button = new OpenPatricianLargeWaxButton(messageSource.getMessage(textKey,
                new Object[]{}, locale.getCurrentLocal()));
        button.setId(id);
        final int actionButtonX = (WIDTH - 124) / 2;
        button.setLayoutX(actionButtonX);
        button.setLayoutY(CLOSE_BTN_Y_POS - 24);
        getContent().add(button);
        return button;
    }

    /**
     * Remove the complete content that was added through this abstraction layer.
     */
    public void clearContent() {
        box.getChildren().clear();
    }

    /**
     * Remove a node from the content.
     * @param node
     */
    public void remove(Node node) {
        box.getChildren().remove(node);
    }
}
