package ch.sahits.game.openpatrician.display.dialog;

import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.openpatrician.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.model.ui.ButtonTemplate;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import javafx.css.CssMetaData;
import javafx.css.SimpleStyleableObjectProperty;
import javafx.css.StyleConverter;
import javafx.css.Styleable;
import javafx.css.StyleableObjectProperty;
import javafx.css.StyleableProperty;
import javafx.scene.control.Control;
import javafx.scene.layout.HBox;
import javafx.scene.text.Font;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Apr 06, 2015
 */
@Prototype
@ClassCategory({EClassCategory.JAVAFX, EClassCategory.PROTOTYPE_BEAN, EClassCategory.DIALOG})
public class GenericCloseButtonDialog extends CloseButtonDialog {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private DecoratedTextFactory textFactory;
    private final DialogTemplate template;

    public GenericCloseButtonDialog(DialogTemplate template) {
        super();
        this.template = template;
        getStylesheets().add(this.getClass().getResource("/styles/base.css").toExternalForm());
        getStyleClass().add("dialog");
    }

    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage(template.getTitleKey(), template.getMessageArgs(), locale.getCurrentLocal()));
        String s = messageSource.getMessage(template.getMessageKey(), template.getMessageArgs(), locale.getCurrentLocal());
        DecoratedText text = textFactory.createDecoratedText(s, new HashMap<>());
        getContent().add(text);
        if (template.getButtons()!=null && !template.getButtons().isEmpty()) {
            HBox box = new HBox(8);
            for (ButtonTemplate buttonTemplate : template.getButtons()) {
                s = messageSource.getMessage(buttonTemplate.getLabelKey(), template.getMessageArgs(), locale.getCurrentLocal());
                if (buttonTemplate.isLargeButton()) {
                    OpenPatricianLargeWaxButton btn = new OpenPatricianLargeWaxButton(s);
                    btn.getStyleClass().add("actionButton");
                    btn.setOnAction((evt) -> {
                        try {
                            buttonTemplate.getAction().run();
                        } catch (RuntimeException e) {
                            logger.error("Failed to execute action", e);
                        }
                    });
                    box.getChildren().add(btn);
                } else {
                    OpenPatricianSmallWaxButton btn = new OpenPatricianSmallWaxButton(s);
                    btn.getStyleClass().add("actionButton");
                    btn.setOnAction((evt) -> {
                        try {
                            buttonTemplate.getAction().run();
                        } catch (RuntimeException e) {
                            logger.error("Failed to execute action", e);
                        }
                    });
                    box.getChildren().add(btn);
                }
            }
            box.setLayoutY(CLOSE_BTN_Y_POS - 24);
            box.setLayoutX((WIDTH - box.getWidth())/2);
            getContent().add(box);
        }
        getCloseButton().setVisible(template.isClosable());

    }

    private StyleableObjectProperty<Font> font;

    public Font getFont() {
        return font == null ? Font.getDefault() : font.get();
    }

    public void setFont(Font font) {
        this.font.set(font);
    }

    public StyleableObjectProperty<Font> fontProperty() {
        if (font == null) {
            font = new SimpleStyleableObjectProperty<Font>(StyleableProperties.FONT, GenericCloseButtonDialog.this, "font", Font.getDefault());
        }
        return font;
    }

    private static class StyleableProperties {
        private static final CssMetaData<GenericCloseButtonDialog, Font> FONT =
                new CssMetaData<GenericCloseButtonDialog, Font>("-fx-font",
                        StyleConverter.getFontConverter(), Font.getDefault()) {
                    @Override
                    public boolean isSettable(GenericCloseButtonDialog control) {
                        return control.font == null || !control.font.isBound();
                    }

                    @Override
                    public StyleableProperty<Font> getStyleableProperty(GenericCloseButtonDialog control) {
                        return control.fontProperty();
                    }
                };
        private static final List<CssMetaData<? extends Styleable, ?>> STYLEABLES;

        static {
            final List<CssMetaData<? extends Styleable, ?>> styleables =
                    new ArrayList<CssMetaData<? extends Styleable, ?>>(Control.getClassCssMetaData());
            Collections.addAll(styleables,
                    FONT
            );
            STYLEABLES = Collections.unmodifiableList(styleables);
        }
    }

    public List<CssMetaData<? extends Styleable, ?>> getControlCssMetaData() {
        return getClassCssMetaData();
    }

    public static List<CssMetaData<? extends Styleable, ?>> getClassCssMetaData() {
        return StyleableProperties.STYLEABLES;
    }
}
