package ch.sahits.game.openpatrician.display.dialog;

import ch.sahits.game.openpatrician.event.NoticeBoardClose;
import ch.sahits.game.openpatrician.display.dialog.service.DialogUtil;
import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.PlaceHolder;
import ch.sahits.game.openpatrician.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.engine.land.city.ReputationCalculator;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.personal.EMilitantCareer;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.ICareer;
import ch.sahits.game.openpatrician.model.personal.IMilitantCareer;
import ch.sahits.game.openpatrician.model.personal.IPersonalData;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.personal.ISpouseData;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.scene.Node;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.Pane;
import javafx.scene.layout.VBox;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * Dialog presenting personal data.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Dec 7, 2013
 *
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class PersonalDialog extends CloseButtonDialog {
	/** Reference to the city view model */
	protected final ICityPlayerProxyJFX city;
	@Autowired
	private Date date;
	@Autowired
	@Qualifier("clientEventBus")
	private AsyncEventBus clientEventBus;
	@Autowired
	private Locale locale;

	@Autowired
	private MessageSource messageSource;
	@Autowired
	private ModelTranslations modelTranslator;
	@Autowired
	private DecoratedTextFactory textFactory;
	@Autowired
	private ReputationCalculator repCalculator;
	@Autowired
	private DialogUtil dialogUtil;
	@Autowired
	@Qualifier("xmlImageLoader")
	private IDataImageLoader imgLoader;

	public PersonalDialog(ICityPlayerProxyJFX city) {
		super();
		this.city = city;
		getStylesheets().add(this.getClass().getResource("/styles/base.css").toExternalForm());
		getStyleClass().add("dialog");
	}
	@PostConstruct
	private void initializeDialog() {
		setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.personal", new Object[]{}, locale.getCurrentLocal()));
		IPlayer player = city.getPlayer();
		IPersonalData personalData = player.getPersonalData();
		IReputation rep = city.getCity().getReputation(player);

		VBox box = new VBox();
		box.setLayoutX(2 * FRAME_BORDER);
		box.setLayoutY(80);

		createAndAddNewText(box, messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.name", new Object[]{
				personalData.getName(),
				personalData.getLastName(),
				modelTranslator.getLocalDisplayName(player.getCareerLevel()),
				modelTranslator.getLocalDisplayName(player.getRank())
		}, locale.getCurrentLocal()));

		box.getChildren().add(createEmptyNode());

		ICity birthPlace = personalData.getBirthPlace();
		LocalDateTime birthDate = personalData.getBirthDate();
		createAndAddNewText(box, messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.birthInfo", new Object[]{
				modelTranslator.toDisplayString(birthDate),
				birthPlace.getName(),
				personalData.getAge(date.getCurrentDate())
		}, locale.getCurrentLocal()));

		box.getChildren().add(createEmptyNode());

		createAndAddNewText(box, messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.hometown", new Object[]{player.getHometown().getName()}, locale.getCurrentLocal()));

		box.getChildren().add(createEmptyNode());

		createAndAddNewText(box, messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.socialStatus", new Object[]{modelTranslator.getLocalDisplayName(city.getPlayer().getRank())}, locale.getCurrentLocal()));
		createAndAddNewText(box, getReputationText(rep));

		List<String> careerAdvice = getCareerAdvice(player);
		if (careerAdvice.isEmpty()) {
			box.getChildren().add(createEmptyNode());
			box.getChildren().add(createEmptyNode());
		} else {
			createAndAddNewText(box, careerAdvice.get(0));
			createAndAddNewText(box, careerAdvice.get(1));
		}

		createAndAddNewText(box, getUnderworldReputation(player));

		box.getChildren().add(createEmptyNode());


		if (player.getSpouseData().isPresent()) {
			ISpouseData spouseData = player.getSpouseData().get();
			createAndAddNewText(box, messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.married", new Object[]{
					getPrefix(spouseData),
					spouseData.getName(),
					spouseData.getLastName()
			}, locale.getCurrentLocal()));


			BorderPane centered = new BorderPane();
			centered.setPrefWidth(WRAPPING_WIDTH);
			Pane pane = new Pane();
			centered.setCenter(pane);

			Image img = imgLoader.getImage("images/portraitBackground");
			ImageView background = new ImageView(img);
			background.setFitWidth(200);
			background.setPreserveRatio(true);
			String portraitResouce = spouseData.getPortraitResourceName();
			ImageView portrait = new ImageView(imgLoader.loadImage(portraitResouce));
			portrait.setFitWidth(200);
			portrait.setPreserveRatio(true);
			img = imgLoader.getImage("images/retro-gold-frame-oval");
			ImageView frame = new ImageView(img);
			frame.setFitWidth(200);
			frame.setPreserveRatio(true);
			pane.getChildren().addAll(background, portrait, frame);
			PlaceHolder ph = new PlaceHolder((WRAPPING_WIDTH - 200)/2, 1);
			centered.setLeft(ph);
			centered.setId("portrait");

			box.getChildren().add(centered);

			birthDate = spouseData.getBirthDate();

			createAndAddNewText(box, messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.birthInfo", new Object[]{
					modelTranslator.toDisplayString(birthDate),
					spouseData.getBirthPlace().getName(),
					spouseData.getAge(date.getCurrentDate())
			}, locale.getCurrentLocal()));

			box.getChildren().add(createEmptyNode());

			createAndAddNewText(box, messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.children", new Object[]{player.getChildren().size()}, locale.getCurrentLocal()));

			box.getChildren().add(createEmptyNode());

			List<String> spouseReputation = getSpouseReputation(player, spouseData);
			createAndAddNewText(box, spouseReputation.get(0));
			createAndAddNewText(box, spouseReputation.get(1));
		} // end spouse data


		getContent().add(box);
	}
	private void createAndAddNewText(VBox box, String s) {
		DecoratedText text = textFactory.createDecoratedText(s, new HashMap<>());
		box.getChildren().add(text);
	}

	private Node createEmptyNode() {
		DecoratedText text = textFactory.createDecoratedText("", new HashMap<>());

		return text;
	}

	private String getReputationText(IReputation reputation) {
		int repDiff = reputation.getReputationDifference();
		StringBuilder sb = new StringBuilder();
		if (repDiff<0){
			// reputation has sunk
			if (repDiff<=-10 && repDiff>-25){
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.sunkenRep1", new Object[]{}, locale.getCurrentLocal()));
			} else if (repDiff>-10){
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.sunkenRep2", new Object[]{}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.sunkenRep3", new Object[]{}, locale.getCurrentLocal()));
			}
		} else if (repDiff>0){
			// reputation has risen
			if (repDiff>=10 && repDiff<25){
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.risenRep1", new Object[]{}, locale.getCurrentLocal()));
			} else if (repDiff<10){
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.risenRep2", new Object[]{}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.risenRep3", new Object[]{}, locale.getCurrentLocal()));
			}
		} else {
			sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.unchangedRep", new Object[]{}, locale.getCurrentLocal()));
		}
		return sb.toString();
	}
	/**
	 * Get the two lines of text for career advice.
	 * @param player
	 * @return
	 */
	private List<String> getCareerAdvice(IPlayer player){
		ESocialRank rank = player.getRank();
		ArrayList<String> lines = new ArrayList<>();
		if (rank!=ESocialRank.ALDERMAN && rank!=ESocialRank.MAYOR){
			// draw string what is needed
			ESocialRank nextRank = rank.getNextRank();
			boolean needFortune = false;
			if (player.getCompany().getCompanyValue()<nextRank.getCompanyValue()){
				needFortune=true;
			}
			boolean needReputation=false;
			double remapedPopulaity = repCalculator.calculateWareReputation(player.getHometown(), player);
			if (remapedPopulaity<nextRank.getSocialRank()){
				needReputation=true;
			}
			StringBuilder sb = new StringBuilder();
			if (needFortune && needReputation){
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.careerAdvice1_1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
				lines.add(sb.toString());
				sb = new StringBuilder();
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.careerAdvice1_2", new Object[]{}, locale.getCurrentLocal()));
				lines.add(sb.toString());
			} else if (needFortune){
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.careerAdvice2_1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
				lines.add(sb.toString());
				sb = new StringBuilder();
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.careerAdvice1_2", new Object[]{}, locale.getCurrentLocal()));
				lines.add(sb.toString());
			} else if (needReputation){
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.careerAdvice3_1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
				lines.add(sb.toString());
				sb = new StringBuilder();
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.careerAdvice1_2", new Object[]{}, locale.getCurrentLocal()));
				lines.add(sb.toString());
			}
		}
		return lines;
	}

	public List<String> getSpouseReputation(IPlayer player, ISpouseData spouse) {
		ArrayList<String> lines = new ArrayList<>();
		StringBuilder sb = new StringBuilder();
		if (city.getCity()==spouse.getBirthPlace()){
			// TODO consider the spouses popularity and relations
			if (spouse.isMale()){
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.spouseRep1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.spouseRep2", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			}
			lines.add(sb.toString());
			sb = new StringBuilder();
			lines.add(sb.toString());
			sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.spouseRep3", new Object[]{}, locale.getCurrentLocal()));
		} else if (city.getCity()==player.getCompany().getHomeTown()){
			// TODO consider the spouses popularity and relations
			if (spouse.isMale()){
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.spouseRep1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.spouseRep2", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			}
			lines.add(sb.toString());
			sb = new StringBuilder();
			lines.add(sb.toString());
			sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.spouseRep3", new Object[]{}, locale.getCurrentLocal()));
		} else {
			if (spouse.isMale()){
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.spouseRep1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.spouseRep2", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			}
			lines.add(sb.toString());
			sb = new StringBuilder();
			lines.add(sb.toString());
			sb.append(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.spouseRep3", new Object[]{}, locale.getCurrentLocal()));
		}
		return lines;
	}

	private String getUnderworldReputation(IPlayer player) {
		ICareer careerLevel = player.getCareerLevel();
		String s;
		if (careerLevel instanceof IMilitantCareer){
			if (((EMilitantCareer)careerLevel)!=EMilitantCareer.BOISTEROUS &&
					((EMilitantCareer)careerLevel)!=EMilitantCareer.BOLD ){
				s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.underworldRep1", new Object[]{}, locale.getCurrentLocal());
			} else {
				s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.underworldRep2", new Object[]{}, locale.getCurrentLocal());
			}
		} else {
			s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.underworldRep2", new Object[]{}, locale.getCurrentLocal());
		}
		return s;
	}

	private String getPrefix(ISpouseData spouseData){
		if (spouseData.isMale()){
			return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.malePrefix", new Object[]{}, locale.getCurrentLocal());
		} else {
			return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.PersonalDialog.femalePrefix", new Object[]{}, locale.getCurrentLocal());
		}
	}

	@Override
	public void executeOnCloseButtonClicked() {
		clientEventBus.post(new NoticeBoardClose());
		super.executeOnCloseButtonClicked();
	}

}
