package ch.sahits.game.openpatrician.display.dialog;

import ch.sahits.game.openpatrician.event.NoticeBoardClose;
import ch.sahits.game.openpatrician.javafx.control.BarrelIconView;
import ch.sahits.game.openpatrician.javafx.model.EDialogType;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.StringBinding;
import javafx.scene.Group;
import javafx.scene.image.ImageView;
import javafx.scene.text.Text;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import java.util.Optional;

/**
 * Dialog for trading between the ship and the storage.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Nov 23, 2013
 *
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class StorageShipTradeDialog extends BaseTradeDialog {

    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
	@Autowired
	private Locale locale;

	@Autowired
	private MessageSource messageSource;

    public StorageShipTradeDialog(ICityPlayerProxyJFX city) {
		super(city, EDialogType.PORT_STORAGE_TO_SHIP);
	}


	/**
	 * Trade to the ship.
	 * @return
	 */
	@Override
	protected String getTradeToDestination() {
		return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.ship", new Object[]{}, locale.getCurrentLocal());
	}
	
	@Override
	protected String getTradeFromDestination() {
		return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.CityStorageTradeDialog.storage", new Object[]{}, locale.getCurrentLocal());
	}
	/**
	 * Text key for the table header of the second action button: to the ship.
	 * @return
	 */	protected String getSecondMoveActionHeaderTextKey() {
		return "ch.sahits.game.openpatrician.display.dialog.CityStorageTradeDialog.toShip";
	}

	/**
	 * Text key for the table header of the first action button: from the ship.
	 * @return
	 */
	protected String getFirstMoveActionHeaderTextKey() {
		return "ch.sahits.game.openpatrician.display.dialog.CityStorageTradeDialog.toStorage";
	}

	@Override
	protected Group createSubTitle(final ICityPlayerProxyJFX city) {
		final INavigableVessel ship = city.getActiveShip();

		String shipNameLabel = ship.getName();
		if (ship instanceof IConvoy) {
			shipNameLabel = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.convoyName", new Object[]{shipNameLabel}, locale.getCurrentLocal());
		}
		final String shipName = shipNameLabel;

		StringBinding shipLoad = new StringBinding() {
			
			{
				super.bind(ship.getLoadBinding());
			}
			
			@Override
			protected String computeValue() {
				return ship.getLoadBinding().asString().get()+" / "+(ship.getLoadableSpace())+" "+shipName;
			}
		};
		Text cityText = new Text();
		cityText.setWrappingWidth(160);

		cityText.textProperty().bind(shipLoad);
		cityText.getStyleClass().add("dialogText");
		Group g = new Group();
		g.setManaged(false);
		
		final Optional<ITradingOffice> optOffice = city.getPlayer().findTradingOffice(city.getCity());
		ImageView barrel = new BarrelIconView();
		barrel.setLayoutX(200 - 26);
		barrel.setLayoutY(-12);
		
		StringBinding storageLoad = new StringBinding() {
			
			{
				if (optOffice.isPresent()) {
					ITradingOffice office = optOffice.get();
					super.bind(office.storedAmountBinding(), office.capacityProperty());
				}
			}
			
			@Override
			protected String computeValue() {
				if (optOffice.isPresent()) {
					ITradingOffice office = optOffice.get();
					return office.storedAmountBinding().asString().get() + " / " + office.capacityProperty().get();
				} else {
					return "0 / 0";
				}
			}
		};
		
		Text shipSpace = new Text();
		shipSpace.textProperty().bind(storageLoad);
		shipSpace.getStyleClass().add("dialogText");
		shipSpace.setLayoutX(200);
		
		Text storage = new Text(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.CityStorageTradeDialog.storage", new Object[]{}, locale.getCurrentLocal()));
		storage.getStyleClass().add("dialogText");
		storage.setLayoutX(300);
		
		g.getChildren().addAll(cityText, barrel, shipSpace, storage);
		return g;
	}


	@Override
	public void executeOnCloseButtonClicked() {
        clientEventBus.post(new NoticeBoardClose());
		super.executeOnCloseButtonClicked();
	}
}
