package ch.sahits.game.openpatrician.display.dialog;

import ch.sahits.game.openpatrician.event.NoticeBoardClose;
import ch.sahits.game.openpatrician.javafx.control.BaleAmount;
import ch.sahits.game.openpatrician.javafx.control.BarrelAmount;
import ch.sahits.game.openpatrician.javafx.control.CoinPriceAlwaysVisible;
import ch.sahits.game.openpatrician.javafx.model.ControlTableCell;
import ch.sahits.game.openpatrician.javafx.model.StaticTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.Table;
import ch.sahits.game.openpatrician.javafx.model.TableHeader;
import ch.sahits.game.openpatrician.javafx.model.TableRow;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.ComputablePriceV2;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.geometry.HPos;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 13, 2013
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class WareInStockDialog extends TabelViewDialog {
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;

    private static final IntegerBinding ONE_INT_BINDING = new IntegerBinding() {
        @Override
        protected int computeValue() {
            return 1;
        }
    };
    /** Reference to the city view model */
    private final ICityPlayerProxyJFX city;
    @Autowired
    private Locale locale;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private ComputablePriceV2 computablePrice;

    public WareInStockDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
    }
    @Override
    public void executeOnCloseButtonClicked() {
        clientEventBus.post(new NoticeBoardClose());
        super.executeOnCloseButtonClicked();
    }
    @PostConstruct
    private void initializeModelAndDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.WareInStockDialog.title", new Object[]{}, locale.getCurrentLocal()));
        Table model = createModel();
        setModel(model);
    }

    private Table createModel() {
        Optional<ITradingOffice> optOffice = city.getPlayer().findTradingOffice(city.getCity());
        Preconditions.checkArgument(optOffice.isPresent(), "The trading office must be present.");
        Table model = new Table();
        TableHeader header = new TableHeader(4);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.ware", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.WareInStockDialog.stockInCity", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.WareInStockDialog.fromCity", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.WareInStockDialog.toCity", new Object[]{}, locale.getCurrentLocal())));
        header.setAligenment(0, HPos.RIGHT);
        header.setAligenment(1, HPos.CENTER);
        header.setAligenment(2, HPos.CENTER);
        header.setAligenment(3, HPos.CENTER);
        model.setHeader(header);
        model.setAligenment(0, HPos.RIGHT);
        model.setAligenment(1, HPos.RIGHT);
        model.setAligenment(2, HPos.RIGHT);
        model.setAligenment(3, HPos.RIGHT);
        model.setColumnWidth(100, 100, 100, 100);
        for (EWare ware : EWare.values()) {
            TableRow row = new TableRow();
            row.add(new StaticTextTableCell(translator.getLocalDisplayName(ware)));

            AmountablePrice<IWare> available = city.getCity().getWare(ware);
            ReadOnlyIntegerProperty storedAmountProperty = available.amountProperty();
            if (ware.isBarrelSizedWare()) {
                BarrelAmount barrelAmount = new BarrelAmount();
                barrelAmount.amountProperty().bind(storedAmountProperty.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmount baleAmount = new BaleAmount();
                baleAmount.amountProperty().bind(storedAmountProperty.asString());
                row.add(new ControlTableCell(baleAmount));
            }

            CoinPriceAlwaysVisible buyPrice = new CoinPriceAlwaysVisible();
            buyPrice.amountProperty().bind(buyPrice(ware, storedAmountProperty));
            row.add(new ControlTableCell(buyPrice));

            CoinPriceAlwaysVisible sellPrice = new CoinPriceAlwaysVisible();
            sellPrice.amountProperty().bind(sellPrice(ware, storedAmountProperty));
            row.add(new ControlTableCell(sellPrice));


            model.add(row);
        }

        return model;
    }
    /**
     * Create the binding for the buy price for the indicated ware.
     * @param ware
     * @param availableAmount
     * @return
     */
    private StringBinding buyPrice(final IWare ware, final ReadOnlyIntegerProperty availableAmount) {
        return new StringBinding() {
            {
                super.bind(availableAmount);
            }

            @Override
            protected String computeValue() {
                            return String.valueOf(computablePrice.buyPrice(ware, availableAmount, ONE_INT_BINDING));
            }
        };
    }
    /**
     * Create a binding for the sell price for the indicated ware.
     * @param ware
     * @param availableAmount
     * @return
     */
    private StringBinding  sellPrice(final IWare ware, final ReadOnlyIntegerProperty availableAmount) {
        return new StringBinding() {
            {
                super.bind(availableAmount);
            }
            @Override
            protected String computeValue() {
                  return String.valueOf(computablePrice.sellPrice(ware, availableAmount, ONE_INT_BINDING));
            }
        };
    }
}
