package ch.sahits.game.openpatrician.display.dialog;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.event.NoticeBoardClose;
import ch.sahits.game.openpatrician.javafx.control.BarrelIconView;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.openpatrician.javafx.model.ControlTableCell;
import ch.sahits.game.openpatrician.javafx.model.DynamicTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.ECellConstraint;
import ch.sahits.game.openpatrician.javafx.model.StaticTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.Table;
import ch.sahits.game.openpatrician.javafx.model.TableHeader;
import ch.sahits.game.openpatrician.javafx.model.TableRow;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.building.IWeaponStorage;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.IWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.PrimaryLargeWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.SecondaryLargeWeaponSlot;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.value.ObservableValue;
import javafx.geometry.HPos;
import javafx.scene.Group;
import javafx.scene.image.ImageView;
import javafx.scene.text.Text;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.Arrays;
import java.util.List;

/**
 * Dialog for transfering weapons between the trading office and a ship.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Aug 20, 2017
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN})
public class WeaponTransferDialog extends TabelViewDialog {
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    private Locale locale;

    @Autowired
    private MessageSource messageSource;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private ShipService shipService;

    private final ICityPlayerProxyJFX proxy;
    private IWeaponStorage weaponStorage;
    private ITradingOffice office;

    public WeaponTransferDialog(ICityPlayerProxyJFX proxy) {
        super();
        this.proxy = proxy;
    }

    @PostConstruct
    private void initializeDialog() {
        // Office must be present otherwise the dialog should not be accessible.
        office = proxy.getPlayer().findTradingOffice(proxy.getCity()).get();
        weaponStorage = office.getWeaponStorage();
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.WeaponTransferDialog.title", new Object[]{}, locale.getCurrentLocal()));

        Group subTitle = createSubTitle(proxy);
        subTitle.setLayoutX(2*FRAME_BORDER);
        subTitle.setLayoutY(80);
        getContent().add(subTitle);

        Table model = createModel();
        setModel(model);
    }

    private Group createSubTitle(final ICityPlayerProxyJFX city) {
        final INavigableVessel ship = city.getActiveShip();

        String shipNameLabel = ship.getName();
        final String shipName = shipNameLabel;

        StringBinding shipLoad = new StringBinding() {

            {
                super.bind(ship.getLoadBinding());
            }

            @Override
            protected String computeValue() {
                return ship.getLoadBinding().asString().get()+" / "+(ship.getLoadableSpace())+" "+shipName;
            }
        };
        Text cityText = new Text();
        cityText.setWrappingWidth(160);

        cityText.textProperty().bind(shipLoad);
        cityText.getStyleClass().add("dialogText");
        Group g = new Group();
        g.setManaged(false);

        ImageView barrel = new BarrelIconView();
        barrel.setLayoutX(200 - 26);
        barrel.setLayoutY(-12);

        StringBinding storageLoad = new StringBinding() {

            {
                super.bind(office.storedAmountBinding(), office.capacityProperty());
            }

            @Override
            protected String computeValue() {
                return office.storedAmountBinding().asString().get() + " / " + office.capacityProperty().get();
            }
        };

        Text shipSpace = new Text();
        shipSpace.textProperty().bind(storageLoad);
        shipSpace.getStyleClass().add("dialogText");
        shipSpace.setLayoutX(200);

        Text storage = new Text(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.CityStorageTradeDialog.storage", new Object[]{}, locale.getCurrentLocal()));
        storage.getStyleClass().add("dialogText");
        storage.setLayoutX(300);

        g.getChildren().addAll(cityText, barrel, shipSpace, storage);
        return g;
    }
    
    private Table createModel() {
        IShip ship = (IShip) proxy.getActiveShip();
        Table model = new Table();

        TableHeader header = new TableHeader(5);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.WeaponTransferDialog.header.weapon", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.WeaponTransferDialog.header.ship", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.WeaponTransferDialog.header.transfer", new Object[]{}, locale.getCurrentLocal())), ECellConstraint.COLSPAN2);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.WeaponTransferDialog.header.storage", new Object[]{}, locale.getCurrentLocal())));
        header.setAligenment(0, HPos.RIGHT);
        header.setAligenment(1, HPos.RIGHT);
        header.setAligenment(2, HPos.CENTER);
        header.setAligenment(4, HPos.RIGHT);
        model.setHeader(header);
        model.setColumnWidth(130, 60, 72, 72, 70);
        List<EWeapon> weapons = Arrays.asList(EWeapon.HAND_WEAPON, EWeapon.TREBUCHET_SMALL, EWeapon.BALLISTA_SMALL,
                EWeapon.TREBUCHET_BIG, EWeapon.BALLISTA_BIG, EWeapon.BOMBARD, EWeapon.CANNON);
        for (EWeapon weapon : weapons) {
            TableRow row = new TableRow();
            row.add(new StaticTextTableCell(translator.getLocalDisplayName(weapon)));
            DynamicTextTableCell shipAmount = new DynamicTextTableCell();
            IntegerBinding shipAmountBinding = shipAmount(weapon, ship);
            shipAmount.valueProperty().bind(shipAmountBinding.asString());
            row.add(shipAmount); // amount on ship
            BooleanBinding disableToShip = disableToShipBinding(ship, weapon);
            OpenPatricianSmallWaxButton toShip = new OpenPatricianSmallWaxButton("<");
            toShip.setId("toShip"+weapon);
            toShip.setOnAction(event -> {
                if (!toShip.isDisabled()) {
                    boolean placed = shipService.placeWeapon(weapon, ship);
                    if (placed) {
                        weaponStorage.update(weapon, -1);
                        disableToShip.invalidate();
                    }
                }
            });
            if (!EWeapon.HAND_WEAPON.equals(weapon)) {
                toShip.disableProperty().bind(disableToShip);
            } else {
                toShip.disableProperty().bind(weaponStorage.getWeaponProperty(EWeapon.HAND_WEAPON).lessThan(1));
            }
            row.add(new ControlTableCell(toShip)); // to ship
            OpenPatricianSmallWaxButton toOffice = new OpenPatricianSmallWaxButton(">");
            toOffice.setId("toOffice"+weapon);
            toOffice.setOnAction(event -> {
                if (!toOffice.isDisabled()) {
                    boolean removed = shipService.removeWeapon(weapon, ship);
                    if (removed) {
                        weaponStorage.update(weapon, 1);
                        shipAmountBinding.invalidate();
                    }
                }
            });
            toOffice.disableProperty().bind(shipAmountBinding.lessThan(1));
            row.add(new ControlTableCell(toOffice)); // to office
            DynamicTextTableCell officeAmount = new DynamicTextTableCell();
            officeAmount.valueProperty().bind(storageAmount(weapon).asString());
            row.add(officeAmount);
            model.add(row);
        }

        return model;
    }

    private BooleanBinding disableToShipBinding(IShip ship, EWeapon weapon) {
        return new BooleanBinding() {
            @Override
            protected boolean computeValue() {
                {
                    super.bind(weaponStorage.getWeaponProperty(weapon));
                }
                if (ship.getCapacity() > 0 && weaponStorage.getWeapon(weapon) > 0) {
                    List<IWeaponSlot> slots = ship.getWeaponSlots();
                     if (isLargeWeapon(weapon)) {
                         for (int i = 0; i < slots.size(); i++) {
                             IWeaponSlot slot = slots.get(i);
                             if (slot instanceof PrimaryLargeWeaponSlot) {
                                 SecondaryLargeWeaponSlot secondary = (SecondaryLargeWeaponSlot) slots.get(i + 1);
                                 if (!slot.getWeapon().isPresent() && !secondary.getWeapon().isPresent()) {
                                     return false;
                                 }
                             }
                         }
                     } else {
                         for (IWeaponSlot slot : slots) {
                             if (!slot.getWeapon().isPresent()) {
                                 return false;
                             }
                         }
                     }
                }
                return true;
            }
        };
    }

    private boolean isLargeWeapon(EWeapon weapon) {
        return weapon.equals(EWeapon.BALLISTA_BIG)
                || weapon.equals(EWeapon.BOMBARD)
                || weapon.equals(EWeapon.TREBUCHET_BIG);
    }

    /**
     * Unbound weapon amount on the ship.
     * @param weapon
     * @param ship
     * @return
     */
    private IntegerBinding shipAmount(EWeapon weapon, IShip ship) {
        return new IntegerBinding() {
            @Override
            protected int computeValue() {
                return ship.getWeaponAmount(weapon);
            }
        };
    }

    /**
     * Property that is bound to the weapon amount in storage.
     * @param weapon
     * @return
     */
    private ReadOnlyIntegerProperty storageAmount(EWeapon weapon) {
        return weaponStorage.getWeaponProperty(weapon);
    }
    @Override
    public void executeOnCloseButtonClicked() {
        clientEventBus.post(new NoticeBoardClose());
        super.executeOnCloseButtonClicked();
    }
}
