package ch.sahits.game.openpatrician.display.dialog.armory;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.dialog.TabelViewDialog;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.openpatrician.javafx.model.ControlTableCell;
import ch.sahits.game.openpatrician.javafx.model.DynamicTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.StaticTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.Table;
import ch.sahits.game.openpatrician.javafx.model.TableHeader;
import ch.sahits.game.openpatrician.javafx.model.TableRow;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.building.IWeaponStorage;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.javafx.bindings.StaticIntegerBinding;
import ch.sahits.game.openpatrician.model.product.ComputablePriceV2;
import ch.sahits.game.openpatrician.model.product.ETransferAmount;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.service.TransferUtil;
import ch.sahits.game.openpatrician.model.weapon.ArmoryRegistry;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.model.weapon.IArmory;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.geometry.HPos;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 *         Created on Jul 23, 2017
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ShipWeaponDialog extends TabelViewDialog {
    private static final int IMAGE_DIM = 60;
    /** Reference to the city view model */
    private final ICityPlayerProxyJFX city;
    private IArmory armory;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    private IDataImageLoader imageLoader;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private TransferUtil transferUtil;
    @Autowired
    private ComputablePriceV2 computablePrice;
    @Autowired
    private ArmoryRegistry registry;
    @Autowired
    private ClientViewState viewState;

    public ShipWeaponDialog(ICityPlayerProxyJFX city) {
        this.city = city;
    }

    @PostConstruct
    private void initializeDialog() {
        armory =  registry.getArmory(city.getCity());
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.armory.ShipWeaponDialog.title",
                new Object[]{}, locale.getCurrentLocal()));
        Table model = createModel(city);
        setModel(model);
    }

    private Table createModel(ICityPlayerProxyJFX cityProxy) {
        Table model = new Table();

        TableHeader header = new TableHeader(6);
        header.add(new StaticTextTableCell(""));
        header.add(new StaticTextTableCell(""));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.armory.ShipWeaponDialog.header.amrory", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.armory.ShipWeaponDialog.header.buy", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.armory.ShipWeaponDialog.header.sell", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.armory.ShipWeaponDialog.header.tradingOffice", new Object[]{}, locale.getCurrentLocal())));
        header.setAligenment(0, HPos.RIGHT);
        header.setAligenment(1, HPos.RIGHT);
        header.setAligenment(2, HPos.RIGHT);
        header.setAligenment(3, HPos.CENTER);
        header.setAligenment(4, HPos.CENTER);
        header.setAligenment(5, HPos.CENTER);
        model.setHeader(header);
        model.setAligenment(0, HPos.LEFT);
        model.setAligenment(1, HPos.LEFT);
        model.setAligenment(2, HPos.RIGHT);
        model.setAligenment(3, HPos.CENTER);
        model.setAligenment(4, HPos.CENTER);
        model.setAligenment(5, HPos.CENTER);
        model.setColumnWidth(130, 40, 72, 72, 60, 70);
        ICity city = cityProxy.getCity();
        Optional<ITradingOffice> optOffice = cityProxy.getPlayer().findTradingOffice(city);
        if (optOffice.isPresent()) {
            ITradingOffice office = optOffice.get();
            IWeaponStorage weaponStorage = office.getWeaponStorage();
            model.add(addWeaponRow(weaponStorage, EWeapon.HAND_WEAPON));
            model.add(addWeaponRow(weaponStorage, EWeapon.TREBUCHET_SMALL));
            model.add(addWeaponRow(weaponStorage, EWeapon.BALLISTA_SMALL));
            model.add(addWeaponRow(weaponStorage,  EWeapon.TREBUCHET_BIG));
            model.add(addWeaponRow(weaponStorage,  EWeapon.BALLISTA_BIG));
            model.add(addWeaponRow(weaponStorage,  EWeapon.BOMBARD));
            model.add(addWeaponRow(weaponStorage,  EWeapon.CANNON));
        } else {
            throw new IllegalStateException("This dialog should only be available if player has trading office in city.");
        }
        return model;
    }

    private TableRow addWeaponRow(IWeaponStorage weaponStorage, EWeapon weapon) {
        TableRow row = new TableRow();
        row.setHeigth(IMAGE_DIM);
        row.add(new StaticTextTableCell(translator.getLocalDisplayName(weapon)));
        Image img = imageLoader.getImage(getWeaponImageName(weapon), IMAGE_DIM, IMAGE_DIM);
        ImageView imgView = new ImageView(img);
        row.add(new ControlTableCell(imgView));
        DynamicTextTableCell weaponAmountInArmory = new DynamicTextTableCell();
        weaponAmountInArmory.valueProperty().bind(getAmountInArmoryAsStringBinding(weapon));
        row.add(weaponAmountInArmory);
        // Buy
        IntegerBinding weaponBuyPrice = buyPriceBinding(weapon);
        OpenPatricianSmallWaxButton weaponBuyBtn = new OpenPatricianSmallWaxButton("");
        weaponBuyBtn.disableProperty().bind(getAmountInArmory(weapon).greaterThan(0).not());
        weaponBuyBtn.setId("buyWeapon"+weapon);
        weaponBuyBtn.textProperty().bind(weaponBuyPrice.asString());
        row.add(new ControlTableCell(weaponBuyBtn));
        weaponBuyBtn.setOnAction(evt -> {
            if (!weaponBuyBtn.isDisable()) {
                IHumanPlayer player = city.getPlayer();
                ICompany company = player.getCompany();
                reduceStoredAmount(weapon);
                weaponStorage.update(weapon, 1);
                int price = weaponBuyPrice.get();
                company.updateCash(-price);
            }
        });
        // Sell
        IntegerBinding weaponSellPrice = sellPriceBinding(weapon);
        OpenPatricianSmallWaxButton weaponSellBtn = new OpenPatricianSmallWaxButton("");
        weaponSellBtn.disableProperty().bind(weaponStorage.getWeaponProperty(weapon).greaterThan(0).not());
        weaponSellBtn.setId("sellWeapon"+weapon);
        weaponSellBtn.textProperty().bind(weaponSellPrice.asString());
        row.add(new ControlTableCell(weaponSellBtn));
        weaponSellBtn.setOnAction(evt -> {
            if (!weaponSellBtn.isDisable()) {
                IHumanPlayer player = city.getPlayer();
                ICompany company = player.getCompany();
                increaseStoredAmount(weapon);
                weaponStorage.update(weapon,  - 1);
                int price = weaponSellPrice.get();
                company.updateCash(price);
            }
        });
        // Amount stored
        DynamicTextTableCell weaponsInStorage = new DynamicTextTableCell();
        weaponsInStorage.valueProperty().bind(weaponStorage.getWeaponProperty(weapon).asString());
        row.add(weaponsInStorage);
        return row;
    }

    private StringBinding getAmountInArmoryAsStringBinding(EWeapon weapon) {
        StringBinding binding = new StringBinding() {
            {
                super.bind(getAmountInArmory(weapon));
            }
            @Override
            protected String computeValue() {
                ReadOnlyIntegerProperty amountInArmory = getAmountInArmory(weapon);
                return String.valueOf(amountInArmory.getValue());
            }
        };
        return binding;
    }

    private void reduceStoredAmount(EWeapon weapon) {
        updateWeaponAmount(weapon, -1);

    }
    private void increaseStoredAmount(EWeapon weapon) {
        updateWeaponAmount(weapon, 1);
    }

    private void updateWeaponAmount(EWeapon weapon, int delta) {
        switch (weapon) {
            case HAND_WEAPON:
                armory.updateCutlassAmount(delta);
                break;
            case BALLISTA_BIG:
                armory.updateBallistaBigAmount(delta);
                break;
            case BALLISTA_SMALL:
                armory.updateBallistaSmallAmount(delta);
                break;
            case TREBUCHET_BIG:
                armory.updateTrebuchetBigAmount(delta);
                break;
            case TREBUCHET_SMALL:
                armory.updateTrebuchetSmallAmount(delta);
                break;
            case BOMBARD:
                armory.updateBombardAmount(delta);
                break;
            case CANNON:
                armory.updateCanonAmount(delta);
                break;
            default:
                throw new IllegalArgumentException("Unhandled weapon type: "+weapon);
        }
    }


    private String getWeaponImageName(EWeapon weapon) {
        switch (weapon) {
            case HAND_WEAPON:
                return "images/Cutlass";
            case BALLISTA_BIG:
                return "images/Ballista_big";
            case BALLISTA_SMALL:
                return "images/Ballista_small";
            case TREBUCHET_BIG:
                return "images/Trebuchet_big";
            case TREBUCHET_SMALL:
                return "images/Trebuchet_small";
            case BOMBARD:
                return "images/Bombard";
            case CANNON:
                return "images/Cannon";
            default:
                throw new IllegalArgumentException("Unhandled weapon type: "+weapon);
        }
    }


    private ReadOnlyIntegerProperty getAmountInArmory(EWeapon weapon) {
        switch (weapon) {
            case HAND_WEAPON:
                return armory.cutlassAmountProperty();
            case BALLISTA_BIG:
                return armory.ballistaBigAmountProperty();
            case BALLISTA_SMALL:
                return armory.ballistaSmallAmountProperty();
            case TREBUCHET_BIG:
                return armory.trebuchetBigAmountProperty();
            case TREBUCHET_SMALL:
                return armory.ballistaSmallAmountProperty();
            case BOMBARD:
                return armory.bombardAmountProperty();
            case CANNON:
                return armory.canonAmountProperty();
            default:
                throw new IllegalArgumentException("Unhandled weapon type: "+weapon);
        }
    }


    private IntegerBinding buyPriceBinding(final EWeapon weapon) {
        return new IntegerBinding() {
            {
                super.bind(getAmountInArmory(weapon));
            }
            @Override
            protected int computeValue() {
                int availableAmount = getAmountInArmory(weapon).get();
                int buyAmount = transferUtil.calculateAvailableAmount(ETransferAmount.ONE, availableAmount);
                if (buyAmount>0) {
                    return computablePrice.buyPrice(weapon, new SimpleIntegerProperty(availableAmount), new StaticIntegerBinding(buyAmount)) * 2;
                } else {
                    return 0;
                }
            }
        };
    }
    private IntegerBinding sellPriceBinding(final EWeapon weapon) {
        return new IntegerBinding() {
            {
                super.bind(getAmountInArmory(weapon));
            }
            @Override
            protected int computeValue() {
                int availableAmount = getAmountInArmory(weapon).get();
                int buyAmount = transferUtil.calculateAvailableAmount(ETransferAmount.ONE, availableAmount);
                if (buyAmount>0) {
                    return computablePrice.sellPrice(weapon, new SimpleIntegerProperty(availableAmount), new StaticIntegerBinding(buyAmount)) * 2;
                } else {
                    return 0;
                }
            }
        };
    }
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.MAIN_VIEW_ARMORY);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }
}
