package ch.sahits.game.openpatrician.display.dialog.guild;

import ch.sahits.game.openpatrician.event.data.AuctionBidLevel;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.display.dialog.service.DialogUtil;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.IItemNavigatorLabelWrapper;
import ch.sahits.game.openpatrician.javafx.control.ItemNavigator;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.javafx.service.ItemNavigatorLabelWrapperFactory;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.GuildService;
import ch.sahits.game.openpatrician.engine.land.city.internal.AuctionEngine;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DateService;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.ModelFactory;
import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.city.guild.EBidLevel;
import ch.sahits.game.openpatrician.model.city.guild.GuildList;
import ch.sahits.game.openpatrician.model.city.guild.IAuction;
import ch.sahits.game.openpatrician.model.city.guild.IBuildingAuction;
import ch.sahits.game.openpatrician.model.city.guild.IGuild;
import ch.sahits.game.openpatrician.model.city.guild.IShipAuction;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.Subscribe;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.EventHandler;
import javafx.scene.Group;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.Pane;
import javafx.scene.layout.VBox;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * Dialog to display next auction, as well as executing it.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Nov 26, 2016
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class AuctionGuildDialog extends CloseButtonDialog {
    private final Logger logger = LogManager.getLogger(getClass());
    /** Reference to the city view model */
    private final ICityPlayerProxyJFX city;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private GuildList guildList;
    @Autowired
    private GuildService guildService;
    @Autowired
    private Date date;
    @Autowired
    private ModelTranslations modelTranslations;
    @Autowired
    private ItemNavigatorLabelWrapperFactory itemWrapperFactory;
    @Autowired
    private DateService dateService;
    @Autowired
    private ModelFactory modelFactory;
    @Autowired
    private DialogUtil dialogHelper;
    @ObjectPropertyType(EBidLevel.class)
    private ObjectProperty<EBidLevel> currentBidLevel;
    private OpenPatricianLargeWaxButton auctionBtn;
    private VBox box;

    public AuctionGuildDialog(ICityPlayerProxyJFX city) {
        this.city = city;
    }
    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.title",
                new Object[]{}, locale.getCurrentLocal()));
        IGuild guild = guildList.findGuild(city.getCity()).get();
        List<IAuction> auctions = guild.getAuctions();

        box = new VBox();

        if (auctions.isEmpty()) {
            String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.noAuction",
                    new Object[]{}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
            text.setId("noAuction");
            box.getChildren().add(text);
        } else {
            LocalDateTime date = auctions.get(0).getAuctionDate();
            String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.nextAuction",
                    new Object[]{modelTranslations.toDisplayString(date)}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
            text.setId("nextAuction");
            box.getChildren().add(text);
        }

        box.getChildren().add(dialogHelper.createVerticalSpacer(20));

        // Offer own ship for auction

        addControlsForOwnShipAuction(guild, box);

        // Auction

        if (!auctions.isEmpty()) {
            IAuction nextAuction = auctions.get(0);
            if (dateService.isToday(nextAuction.getAuctionDate())) {
                // What is auctioned
                addAuctionDescription(box, nextAuction);
                // Current bid
                currentBidLevel = new SimpleObjectProperty<>(nextAuction.getBidLevel());

                addCurrentBidInformation(box, nextAuction, box.getChildren().size());

                // button to bid
                String s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.nextBid",
                        new Object[]{nextAuction.getCurrentBid() + AuctionEngine.BID_DELTA}, locale.getCurrentLocal());
                auctionBtn = new OpenPatricianLargeWaxButton(s);
                auctionBtn.getStyleClass().add("actionButton");
                auctionBtn.setId("auctionButton");
                Pane centeredAuction = dialogHelper.center(auctionBtn);
                auctionBtn.setOnAction(event -> {
                    try {
                        auctionBtn.setDisable(true);
                        EBidLevel current = currentBidLevel.get();
                        EBidLevel next = nextAuction.getBiddingPlayer() == null ? EBidLevel.INITIAL : calculateNextBidLevel(current);
                        clientEventBus.post(new AuctionBidLevel(current, next, city.getPlayer(), nextAuction));
                    } catch (RuntimeException e) {
                        logger.error("Failed to auction ship", e);
                    }
                });
                box.getChildren().add(centeredAuction);


                addBidLevelCountDown(box, nextAuction, box.getChildren().size());
                clientEventBus.register(this);
            } // End auction today
        }

        box.setLayoutX(50);
        box.setLayoutY(CLOSE_BTN_Y_POS - 500);
        getContent().addAll(box);
    }

    private void addBidLevelCountDown(VBox box, IAuction nextAuction, int index) {
        String template = getBidLevelTemplate(nextAuction);
        DecoratedText dt = textFactory.createDecoratedText(template, new HashMap<>());
        dt.setId("bidLevel-"+currentBidLevel.get());
        box.getChildren().add(index, dt);
    }

    private void addCurrentBidInformation(VBox box, IAuction nextAuction, int index) {
        if (nextAuction.getBiddingPlayer() == null) { // no bid
            String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.noBid",
                    new Object[]{}, locale.getCurrentLocal());
            DecoratedText dt = textFactory.createDecoratedText(template, new HashMap<>());
            dt.setId("noBid");
            box.getChildren().add(index, dt);
        } else { // current bid
            IPlayer bidder = nextAuction.getBiddingPlayer();
            String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.currentBid",
                    new Object[]{nextAuction.getCurrentBid(), bidder.getName(), bidder.getLastName()}, locale.getCurrentLocal());
            DecoratedText dt = textFactory.createDecoratedText(template, new HashMap<>());
            dt.setId("currentBid");
            box.getChildren().add(index, dt);
        }

        box.getChildren().add(dialogHelper.createVerticalSpacer(20));
    }

    private void addAuctionDescription(VBox box, IAuction nextAuction) {
        if (nextAuction instanceof IBuildingAuction) {
            IBuilding building = ((IBuildingAuction) nextAuction).getAuctionedBuilding();
            IPlayer owner = nextAuction.getOwner().get();
            String name = owner.getName()+" "+owner.getLastName();
            String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.buildingAuction",
                    new Object[]{modelTranslations.getLocalDisplayName(building), name}, locale.getCurrentLocal());
            DecoratedText dt = textFactory.createDecoratedText(template, new HashMap<>());
            dt.setId("buildingAuction");
            box.getChildren().add(dt);
        } else {
            IShip ship = ((IShipAuction)nextAuction).getAuctionedShip();
            String name = ship.getOwner().getName()+" "+ship.getOwner().getLastName();
            String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.shipAuction",
                    new Object[]{ship.getName(), name}, locale.getCurrentLocal());
            DecoratedText dt = textFactory.createDecoratedText(template, new HashMap<>());
            dt.setId("shipAuction");
            box.getChildren().add(dt);
        }
        box.getChildren().add(dialogHelper.createVerticalSpacer(20));
    }

    private void addControlsForOwnShipAuction(IGuild guild, VBox box) {
        String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.registerShip",
                new Object[]{}, locale.getCurrentLocal());
        final DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
        text.setId("registerShip");
        box.getChildren().add(text);
        if (city.getPlayersNavalVessels().isEmpty()) {
            template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.noShip",
                    new Object[]{}, locale.getCurrentLocal());
            DecoratedText dt = textFactory.createDecoratedText(template, new HashMap<>());
            dt.setId("noShipInPort");
            box.getChildren().add(dt);
        } else {
           List<INavigableVessel> vessels = city.getPlayersNavalVessels();
           List<IShip> ships = new ArrayList<>();
            for (INavigableVessel vessel : vessels) {
                if (vessel instanceof IShip) {
                    ships.add((IShip) vessel);
                }
            }
            List<IItemNavigatorLabelWrapper<IShip>> wrappedList = itemWrapperFactory.createListForVessels(ships);
            ItemNavigator<IShip> itemNavigator = new ItemNavigator<>(wrappedList);
            Group centerPane = dialogHelper.center(itemNavigator);

            box.getChildren().add(dialogHelper.createVerticalSpacer(10));

            box.getChildren().add(centerPane);

            box.getChildren().add(dialogHelper.createVerticalSpacer(10));

            String s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.auctionShip",
                    new Object[]{}, locale.getCurrentLocal());
            final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(s);
            action.getStyleClass().add("actionButton");
            action.setId("offerShip");
            Pane centeredButton = dialogHelper.center(action);
            box.getChildren().add(centeredButton);
            action.setOnAction(createAuctionShipAction(box, guild, text, itemNavigator, centerPane, centeredButton));

        }
        box.getChildren().add(dialogHelper.createVerticalSpacer(20));
    }


    private EventHandler<MouseEvent> createAuctionShipAction(final VBox container, IGuild guild, DecoratedText text, ItemNavigator<IShip> itemNavigator, Group centerPane, Pane action) {
        return event -> {
            try {
                // The elements are split up on two components, make sure they are gone from both.
                container.getChildren().removeAll(text, centerPane, action);
                getContent().removeAll(text, centerPane, action);
                LocalDateTime auctionDate = date.getCurrentDate().plusDays(10);
                auctionDate = guildService.cleanUpAuctionAndDetermineAuctionDate(auctionDate, guild);
                IShip ship = itemNavigator.getCurrent();
                IShipAuction auction = modelFactory.createShipAuction(auctionDate, ship.getValue(), city.getPlayer(), ship);
                city.getPlayer().removeShip(ship);
                guild.getAuctions().add(auction);
            } catch (RuntimeException e) {
                logger.error("Failed to set ship for auctioning", e);
            }
        };
    }

    private EBidLevel calculateNextBidLevel(EBidLevel current) {
        EBidLevel next;
        switch (current) {
            case INITIAL:
                next = EBidLevel.ONE;
                break;
            case ONE:
                next = EBidLevel.TWO;
                break;
            case TWO:
                next = EBidLevel.DONE;
                break;
            default:
                throw new IllegalStateException("Unhandled state: "+current);
        }
        return next;
    }

    private String getBidLevelTemplate(IAuction auction) {
        switch (currentBidLevel.get()) {
            case INITIAL:
                return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.noBid",
                        new Object[]{}, locale.getCurrentLocal());
            case ONE:
                return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.first",
                        new Object[]{}, locale.getCurrentLocal());
            case TWO:
                return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.second",
                        new Object[]{}, locale.getCurrentLocal());
            case DONE:
                IPlayer newOwner = auction.getBiddingPlayer();
                return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.sold",
                        new Object[]{newOwner.getName(), newOwner.getLastName()}, locale.getCurrentLocal());
            default:
                throw new IllegalStateException("Unhandled state: "+currentBidLevel.get());
        }
    }

    @Override
    public void executeOnCloseButtonClicked() {
        clientEventBus.unregister(this);
        super.executeOnCloseButtonClicked();
    }
    @Subscribe
    public void handleBidLevelChange(AuctionBidLevel change) {
        currentBidLevel.setValue(change.getNewLevel());
        if (!change.getBidder().equals(city.getPlayer())) {
            auctionBtn.setDisable(false);
        }
        // update bid labels
        int index = dialogHelper.removeById(box, "noBid");
        if (index < 0) {
            index = dialogHelper.removeById(box, "currentBid");
        }
        addCurrentBidInformation(box, change.getAuction(), index);
        index = dialogHelper.removeById(box, "bidLevel-"+change.getOldLevel());
        box.getChildren().remove(index);
        addBidLevelCountDown(box, change.getAuction(), index);
        if (change.getNewLevel() == EBidLevel.DONE) {
            auctionBtn.setVisible(false);
            clientEventBus.unregister(this);
        } else {
            String label = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.guild.AuctionGuildDialog.nextBid",
                    new Object[]{change.getAuction().getCurrentBid() + AuctionEngine.BID_DELTA}, locale.getCurrentLocal());
            auctionBtn.setText(label);
        }
    }

    // TODO: andi 12/3/16 handle case when another player bids => reset 
}
