package ch.sahits.game.openpatrician.display.dialog.service;

import ch.sahits.game.graphic.image.impl.SelectiveCachableXMLImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.data.xmlmodel.BasicOffset;
import ch.sahits.game.openpatrician.data.xmlmodel.BasicSlot;
import ch.sahits.game.openpatrician.data.xmlmodel.WeaponLocation;
import ch.sahits.game.openpatrician.data.xmlmodel.WeaponLocations;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.openpatrician.javafx.control.PlaceHolder;
import ch.sahits.game.openpatrician.javafx.dialog.IDialog;
import ch.sahits.game.openpatrician.javafx.model.Table;
import ch.sahits.game.openpatrician.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.javafx.service.JavaFXUtils;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.ESide;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.IWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.SecondaryLargeWeaponSlot;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.base.Preconditions;
import javafx.beans.Observable;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.collections.ObservableList;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.geometry.Point2D;
import javafx.geometry.Pos;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.control.Control;
import javafx.scene.control.Label;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.FlowPane;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.Pane;
import javafx.scene.layout.RowConstraints;
import javafx.scene.text.Text;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.oxm.Unmarshaller;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 20, 2013
 */
@Component
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class DialogUtil {
    private final Logger logger = LogManager.getLogger(getClass());

    @Autowired
    private JavaFXUtils fxUtils;
    @Autowired
    private ShipService shipService;
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    @Qualifier("xmlImageLoader")
    private SelectiveCachableXMLImageLoader imageLoader;
    @Autowired
    @Qualifier("jaxb2XmlModelMarshaller")
    private Unmarshaller unmarshaller;

    private WeaponLocations weaponLocations;

    @PostConstruct
    private void initilizeWeaponModel() {
        try {
            Source sourceFromFile = getSourceFromFile("/weaponLocation.xml");
            weaponLocations = (WeaponLocations) unmarshaller.unmarshal(sourceFromFile);
        } catch (IOException e) {
            logger.warn("Failed to initialize weapons location");
        }
    }
    private Source getSourceFromFile(String fileName) {
        final InputStream resourceAsStream = getClass().getResourceAsStream(fileName);
        return new StreamSource(resourceAsStream);
    }

    /**
     * Boolean binding for the navigation enabling of the ships cataloque.
     * Only the ships are considered (convoys are excluded)
     * @param ships to be observed.
     * @return binding.
     */
    public BooleanBinding enableShipCatalogueForShips(final ObservableList<IShip> ships) {
        return new BooleanBinding() {
            {
                super.bind(ships);
            }
            @Override
            protected boolean computeValue() {
                return ships.size() > 1;
            }
        };
    }

    /**
     * Navigate to the next ship in the catalogue.
     * Only the ships are considered (convoys are excluded)
     * @param ships in the selection
     * @param currentShip current ship
     * @return Event handler for the navigation.
     */
    public EventHandler<MouseEvent> createNextActionForShips(final ObservableList<IShip> ships,
                                                             final ObjectProperty<IShip> currentShip) {
        return mouseEvent -> {
            try {
                if (ships.size() > 1) {
                    if (ships.get(ships.size() - 1).equals(currentShip.get())) {
                        currentShip.set(ships.get(0)); // first one;
                    } else {
                        for (int i = 0; i < ships.size() - 1; i++) {
                            if (ships.get(i).equals(currentShip.get())) {
                                final IShip nextShip = ships.get(i + 1);
                                currentShip.set(nextShip);
                                break;
                            }
                        }
                    }
                }
            } catch (RuntimeException e) {
                logger.error("Failed to switch to next ship", e);
            }
        };
    }

    /**
     * Navigate to the previous ship in the catalogue.
     * Only the ships are considered (convoys are excluded)
     * @param ships in the selection
     * @param currentShip current ship
     * @return Event handler for the navigation.
     */

    public EventHandler<MouseEvent> createPreviousActionForShips(final ObservableList<IShip> ships,
                                                                 final ObjectProperty<IShip> currentShip) {
        return mouseEvent -> {
            try {
                if (ships.size() > 1) {
                    if (ships.get(0).equals(currentShip.get())) {
                        currentShip.set(ships.get(ships.size() - 1)); // last one;
                    } else {
                        for (int i = 1; i < ships.size(); i++) {
                            if (ships.get(i).equals(currentShip.get())) {
                                final IShip prevShip = ships.get(i - 1);
                                currentShip.set(prevShip);
                                break;
                            }
                        }
                    }
                }
            } catch (RuntimeException e) {
                logger.error("Failed to switch to previous ship", e);
            }
        };
    }

    /**
     * Create a grid pane based on a model.
     * @param model base model
     * @return grid pane
     */
    public GridPane createGridPaneFromModel(Table model) {
        return fxUtils.createGridPaneFromModel(model);
    }

    /**
     * Create the grid pane for the ship selection on three lines. The currently selected ship is the first in
     * <code>ships</code>.<br>
     * <pre>
     *     +----------+-------------+-------------+-------------+-----------+
     *     |                      ship type and name                        |
     *     |prevShip  | weapon icon | health icon | sailor icon |   nextShip|
     *     |          |weapon amount|   health    | no. sailors |           |
     *     +----------+-------------+-------------+-------------+-----------+
     * </pre>
     * @param ships observable list of ships through which is to be navigated.
     * @param  currentShip container to store the currently selected ship. The value will be set as par of the execution.
     * @return GridPane displaying ship details like weapon strength, health and crew compliment as well as navigation
     *         capabilities through the list of ships.
     */
    public GridPane createShipSelection3LinesForShips(ObservableList<IShip> ships, final ObjectProperty<IShip> currentShip) {
        // TODO: andi 9/10/17 handle cases where the ship list changes.
        Image weaponIcon = imageLoader.getImage("icons/64/bombard-icon", 32, 32);
        Image healthIcon = imageLoader.getImage("icons/64/health-icon", 32, 32);
        Image sailorIcon = imageLoader.getImage("icons/64/sailor-icon", 32, 32);
        currentShip.setValue(ships.get(0));
        GridPane shipSelectionPane = new GridPane();
        RowConstraints rowConstraint = new RowConstraints(24);
        shipSelectionPane.getRowConstraints().addAll(rowConstraint, rowConstraint, rowConstraint);
        ColumnConstraints colConstraint = new ColumnConstraints(64);
        shipSelectionPane.getColumnConstraints().addAll(colConstraint, colConstraint, colConstraint, colConstraint, colConstraint);
        // Row 0
        Text shipName = new Text();
        StringBinding shipNameBinding = new StringBinding() {
            {
                super.bind(currentShip);
            }
            @Override
            protected String computeValue() {
                final IShip ship = currentShip.get();
                return ship.getShipType().name()+" "+ ship.getName();
            }
        };
        shipName.getStyleClass().add("dialogText");
        shipName.setId("shipTypeName");
        shipName.textProperty().bind(shipNameBinding);
        shipSelectionPane.add(shipName,0, 0, 5, 1);
        GridPane.setHalignment(shipName, HPos.CENTER);

        // Row 1
        BooleanBinding enablePrevNext = enableShipCatalogueForShips(ships);

        final OpenPatricianSmallWaxButton prevShip = new OpenPatricianSmallWaxButton("<");
        prevShip.setId("prevShip");
        prevShip.getStyleClass().add("actionButton");
        prevShip.setOnAction(createPreviousActionForShips(ships, currentShip));
        prevShip.setDisable(!enablePrevNext.get());
        shipSelectionPane.add(prevShip, 0, 1);
        GridPane.setHalignment(prevShip, HPos.LEFT);

        ImageView weaponIconView = new ImageView(weaponIcon);
        weaponIconView.setId("weaponIcon");
        shipSelectionPane.add(weaponIconView, 1, 1);
        GridPane.setHalignment(weaponIconView, HPos.CENTER);

        ImageView healthIconView = new ImageView(healthIcon);
        healthIconView.setId("healthIcon");
        shipSelectionPane.add(healthIconView, 2, 1);
        GridPane.setHalignment(healthIconView, HPos.CENTER);

        ImageView sailorIconView = new ImageView(sailorIcon);
        sailorIconView.setId("sailorIcon");
        shipSelectionPane.add(sailorIconView, 3, 1);
        GridPane.setHalignment(sailorIconView, HPos.CENTER);

        final OpenPatricianSmallWaxButton nextShip = new OpenPatricianSmallWaxButton(">");
        nextShip.setId("nextShip");
        nextShip.getStyleClass().add("actionButton");
        nextShip.setOnAction(createNextActionForShips(ships, currentShip));
        nextShip.setDisable(!enablePrevNext.get());
        shipSelectionPane.add(nextShip, 4, 1);
        GridPane.setHalignment(nextShip, HPos.RIGHT);

        enablePrevNext.addListener(new ChangeListener<Boolean>() {
            @Override
            public void changed(ObservableValue<? extends Boolean> observableValue,
                                Boolean oldValue, Boolean newValue) {
                nextShip.setDisable(!newValue);
                prevShip.setDisable(!newValue);
            }
        });

        // Row 2
        final ObjectProperty<DecoratedText> text = createWeaponStrength(currentShip.get());
        currentShip.addListener((observable, oldValue, newValue) -> {
            DecoratedText newText = createWeaponStrength(newValue).get();
            shipSelectionPane.getChildren().remove(text.get());
            text.setValue(newText);
            shipSelectionPane.add(text.get(), 1, 2);
        });

        shipSelectionPane.add(text.get(), 1, 2);

        Text damage = new Text();
        damage.setId("health");
        damage.getStyleClass().add("dialogText");
        StringBinding damageBinding = new StringBinding() {
            {
                super.bind(currentShip);
            }
            @Override
            protected String computeValue() {
                return String.valueOf(100 - currentShip.get().getDamage());
            }
        };
        damage.textProperty().bind(damageBinding);
        shipSelectionPane.add(damage, 2, 2);
        GridPane.setHalignment(damage, HPos.CENTER);

        Text sailorsOnShip2 = new Text();
        sailorsOnShip2.setId("crew");
        sailorsOnShip2.getStyleClass().add("dialogText");
        final SailorOnShipBinding sailorsOnShipBinding = new SailorOnShipBinding(currentShip);
        currentShip.addListener(new ChangeListener<IShip>() {
            @Override
            public void changed(ObservableValue<? extends IShip> observableValue,
                                IShip oldValue, IShip newValue) {
                sailorsOnShipBinding.delegateUnbind(oldValue.numberOfSailorsProperty());
                sailorsOnShipBinding.delegateBind(newValue.numberOfSailorsProperty());
            }
        });
        sailorsOnShip2.textProperty().bind(sailorsOnShipBinding);
        shipSelectionPane.add(sailorsOnShip2, 3, 2);
        GridPane.setHalignment(sailorsOnShip2, HPos.CENTER);
        return shipSelectionPane;
    }

    /**
     * Create the grid pane for the ship information on three lines.<br>
     * <pre>
     *     +------------+-------------+-------------+-------------+
     *     | ship type and name                                   |
     *     | knot icon  | weapon icon | health icon | sailor icon |
     *     |            |weapon amount|   health    | no. sailors |
     *     +------------+-------------+-------------+-------------+
     * </pre>
     * @param  currentShip container to store the currently selected ship. The value will be set as par of the execution.
     * @return GridPane displaying ship details like speed, weapon strength, health and crew compliment.
     */
    public GridPane createShipInfoOnThreeLines(final ObjectProperty<IShip> currentShip) {
        Image weaponIcon = imageLoader.getImage("icons/64/bombard-icon", 32, 32);
        Image healthIcon = imageLoader.getImage("icons/64/health-icon", 32, 32);
        Image sailorIcon = imageLoader.getImage("icons/64/sailor-icon", 32, 32);
        Image knotIcon = imageLoader.getImage("icons/64/knot-icon", 32, 32); 
        Image compassIcon = imageLoader.getImage("icons/64/compass-icon", 24, 24);
        GridPane shipSelectionPane = new GridPane();
        RowConstraints rowConstraint = new RowConstraints(24);
        shipSelectionPane.getRowConstraints().addAll(rowConstraint, rowConstraint, rowConstraint);
        ColumnConstraints colConstraint = new ColumnConstraints(64);
        shipSelectionPane.getColumnConstraints().addAll(colConstraint, colConstraint, colConstraint, colConstraint, colConstraint);
        // Row 0
        Text shipName = new Text();
        StringBinding shipNameBinding = new StringBinding() {
            {
                super.bind(currentShip);
            }
            @Override
            protected String computeValue() {
                final IShip ship = currentShip.get();
                return ship.getShipType().name()+" "+ ship.getName();
            }
        };
        shipName.getStyleClass().add("dialogText");
        shipName.setId("shipTypeName");
        shipName.textProperty().bind(shipNameBinding);
        shipSelectionPane.add(shipName,0, 0, 4, 1);
        GridPane.setHalignment(shipName, HPos.LEFT);

        // Row 1

        ImageView knotIconView = new ImageView(knotIcon);
        knotIconView.setId("knotIcon");
        shipSelectionPane.add(knotIconView, 0, 1);
        GridPane.setHalignment(knotIconView, HPos.CENTER);

        ImageView weaponIconView = new ImageView(weaponIcon);
        weaponIconView.setId("weaponIcon");
        shipSelectionPane.add(weaponIconView, 1, 1);
        GridPane.setHalignment(weaponIconView, HPos.CENTER);

        ImageView healthIconView = new ImageView(healthIcon);
        healthIconView.setId("healthIcon");
        shipSelectionPane.add(healthIconView, 2, 1);
        GridPane.setHalignment(healthIconView, HPos.CENTER);

        ImageView sailorIconView = new ImageView(sailorIcon);
        sailorIconView.setId("sailorIcon");
        shipSelectionPane.add(sailorIconView, 3, 1);
        GridPane.setHalignment(sailorIconView, HPos.CENTER);



        // Row 2
        int speedInKnots = shipService.convertSpeedToKnots(currentShip.get());
        Text speed = new Text(String.valueOf(speedInKnots));
        speed.setId("speed");
        speed.getStyleClass().add("dialogText");
        shipSelectionPane.add(speed, 0, 2);
        GridPane.setHalignment(speed, HPos.CENTER);

        final ObjectProperty<DecoratedText> text = createWeaponStrength(currentShip.get());
        currentShip.addListener((observable, oldValue, newValue) -> {
            DecoratedText newText = createWeaponStrength(newValue).get();
            shipSelectionPane.getChildren().remove(text.get());
            text.setValue(newText);
            shipSelectionPane.add(text.get(), 1, 2);
        });

        shipSelectionPane.add(text.get(), 1, 2);

        Text damage = new Text();
        damage.setId("health");
        damage.getStyleClass().add("dialogText");
        StringBinding damageBinding = new StringBinding() {
            {
                super.bind(currentShip);
            }
            @Override
            protected String computeValue() {
                return String.valueOf(100 - currentShip.get().getDamage());
            }
        };
        damage.textProperty().bind(damageBinding);
        shipSelectionPane.add(damage, 2, 2);
        GridPane.setHalignment(damage, HPos.CENTER);

        Label sailorsOnShip2 = new Label();
        sailorsOnShip2.setId("crew");
        sailorsOnShip2.getStyleClass().add("dialogText");
        final SimpleSailorOnShipBinding sailorsOnShipBinding = new SimpleSailorOnShipBinding(currentShip);
        currentShip.addListener(new ChangeListener<IShip>() {
            @Override
            public void changed(ObservableValue<? extends IShip> observableValue,
                                IShip oldValue, IShip newValue) {
                sailorsOnShipBinding.delegateUnbind(oldValue.numberOfSailorsProperty());
                sailorsOnShipBinding.delegateBind(newValue.numberOfSailorsProperty());
            }
        });
        Group g = new Group();
        if (currentShip.get().getCaptian().isPresent()) {
            final ImageView imageView = new ImageView(compassIcon);
            imageView.setId("captainIcon");
            imageView.layoutXProperty().bind(sailorsOnShip2.widthProperty().add(3));
            g.getChildren().addAll(sailorsOnShip2, imageView);
        } else {
            g.getChildren().addAll(sailorsOnShip2);
        }
        sailorsOnShip2.textProperty().bind(sailorsOnShipBinding);
        shipSelectionPane.add(g, 3, 2);
        GridPane.setHalignment(g, HPos.CENTER);
        return shipSelectionPane;       
    }

    private ObjectProperty<DecoratedText> createWeaponStrength(IShip currentShip) {
        int strength = shipService.calculateShipsWeaponsStrength(currentShip);
        String key = null;
        if (strength > 0) {
            if (strength % 2 != 0) {
                if (strength == 1) {
                    key = "ch.sahits.game.openpatrician.model.ship.impl.Ship.weaponStrengthOne";
                } else {
                    key = "ch.sahits.game.openpatrician.model.ship.impl.Ship.weaponStrengthUneven";
                }
            } else {
                key = "ch.sahits.game.openpatrician.model.ship.impl.Ship.weaponStrengthEven";
            }
            strength /= 2;
            String template = messageSource.getMessage(key, new Object[]{strength}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
            text.setId("weaponStrength");
            return new SimpleObjectProperty<>(text);
        } else {
            return new SimpleObjectProperty<>(new DecoratedText());
        }
    }

    /**
     * Sailors on ship binding with min and max.
     */
    private class SailorOnShipBinding extends StringBinding {
        private final ObjectProperty<IShip> currentShip;
        SailorOnShipBinding(ObjectProperty<IShip> currentShip) {
            super.bind(currentShip, currentShip.get().numberOfSailorsProperty());
            this.currentShip = currentShip;
        }
        @Override
        protected String computeValue() {
            final IShip ship = currentShip.get();
            StringBuilder sb = new StringBuilder();
            sb.append(ship.getNumberOfSailors())
                    .append(" (")
                    .append(ship.getMinNumberOfSailors())
                    .append(" - ")
                    .append(ship.getMaxNumberOfSailors())
                    .append(")");
            return sb.toString();
        }

        public void delegateUnbind(Observable...values) {
            unbind(values);
        }
        public void delegateBind(Observable...values) {
            bind(values);
        }
    }

        /**
         * Sailors on ship binding.
         */
        private class SimpleSailorOnShipBinding extends StringBinding {
            private final ObjectProperty<IShip> currentShip;
            SimpleSailorOnShipBinding(ObjectProperty<IShip> currentShip) {
                super.bind(currentShip, currentShip.get().numberOfSailorsProperty());
                this.currentShip = currentShip;
            }
            @Override
            protected String computeValue() {
                final IShip ship = currentShip.get();
                return String.valueOf(ship.getNumberOfSailors());
            }
            public void delegateUnbind(Observable...values) {
                unbind(values);
            }
            public void delegateBind(Observable...values) {
                bind(values);
            }
        }

    /**
     * Create an element that will layout the <code>childNode</code>
     * horizontally centered within a Dialog.
     * @param childNode that should be centered
     * @return wrapping Pane
     */
    public Pane center(Control childNode) {
        final int width = IDialog.WRAPPING_WIDTH - 24;
        Pane g = new Pane(childNode);
        childNode.widthProperty().addListener((observable, oldValue, newValue) -> {
            double inset = (width - childNode.getWidth())/2;
            g.setLayoutX(inset);
        });
        return g;
    }
    /**
     * Create an element that will layout the <code>childNode</code>
     * horizontally centered within a Dialog.
     * @param childNode that should be centered
     * @return wrapping Pane
     */
    public Pane center(ImageView childNode) {
        final int width = IDialog.WRAPPING_WIDTH - 24;
        Pane g = new Pane(childNode);
        double in = (width - childNode.getImage().getWidth())/2;
        g.setLayoutX(in);
        childNode.getImage().widthProperty().addListener((observable, oldValue, newValue) -> {
            double inset = (width - childNode.getImage().getWidth())/2;
            g.setLayoutX(inset);
        });
        return g;
    }
    /**
     * Create an element that will layout the <code>childNode</code>
     * horizontally centered within a Dialog.
     * @param childNode that should be centered
     * @return wrapping Pane
     */
    public Group center(Pane childNode) {
        final int width = IDialog.WRAPPING_WIDTH - 24;
        Group g = new Group(childNode);
        childNode.widthProperty().addListener((observable, oldValue, newValue) -> {
            double inset = (width - childNode.getWidth())/2;
            g.setLayoutX(inset);
        });
        if (childNode instanceof FlowPane) {
            ((FlowPane) childNode).setAlignment(Pos.CENTER);
        }
        return g;
    }

    /**
     * Create an element that will layout the <code>childNode</code>
     * at the right dialog content border.
     * @param childNode that should be right aligned.
     * @return wrapping Pane
     */
    public Group right(Pane childNode) {
        final int width = IDialog.WRAPPING_WIDTH - 24;
        Group g = new Group(childNode);
        childNode.widthProperty().addListener((observable, oldValue, newValue) -> {
            double inset = (width - childNode.getWidth());
            g.setLayoutX(inset);
        });
        if (childNode instanceof FlowPane) {
            ((FlowPane) childNode).setAlignment(Pos.CENTER_RIGHT);
        }
        return g;
    }

    /**
     * Create an element that will layout the <code>childNode</code>
     * at the right dialog content border.
     * @param childNode that should be right aligned.
     * @return wrapping Pane
     */
    public Pane right(Control childNode) {
        final int width = IDialog.WRAPPING_WIDTH - 24;
        Pane g = new Pane(childNode);
        childNode.widthProperty().addListener((observable, oldValue, newValue) -> {
            double inset = (width - childNode.getWidth());
            g.setLayoutX(inset);
        });
        return g;
    }

    /**
     * Create a Node that forces vertical spacing.
     * @param heigth of the spacing node.
     * @return Node with specified <code>heigth</code>.
     */
    public Node createVerticalSpacer(int heigth) {
        return new PlaceHolder(1, heigth);
    }

    /**
     * Remove a node identified by <code>id</code> from the parent
     * and return the index.
     * @param parent containing the node
     * @param id of the id
     * @return index in the children of the removed node within id. If the node
     * cannot be found -1 will be returned.
     */
    public int removeById(Pane parent, String id) {
        Preconditions.checkNotNull(id, "The identifying id must not be null");
        ObservableList<Node> children = parent.getChildren();
        int index = -1;
        for (int i = 0; i < children.size(); i++) {
            Node node = children.get(i);
            if (id.equals(node.getId())) {
                index = i;
                break;
            }
        }
        if (index >= 0) {
            children.remove(index);
        }
        return index;
    }

    /**
     * Create the weapon display with all weapons and place holders.
     * @param ship
     * @return
     */
    public Pane createShipWeaponDisplay(IShip ship) {
        List<IWeaponSlot> weaponSlots = ship.getWeaponSlots();
        Pane pane = new Pane();
        pane.setId("weaponDisplay");
        ImageView deck = getShipDeck(ship.getShipType());
        pane.getChildren().add(deck);
        for (IWeaponSlot weaponSlot : weaponSlots) {
            if (weaponSlot.getWeapon().isPresent()) {
                // do nothing if big weapon and secondary slot
                EWeapon weapon = (EWeapon) weaponSlot.getWeapon().get();
                boolean big = shipService.isLargeWeapon(weapon);
                if (weaponSlot instanceof SecondaryLargeWeaponSlot && big) {
                    continue;
                }
                int id = weaponSlot.getId();
                BasicSlot slot = getSlotModel(id);
                ImageView imgView = getShipWeapon(weaponSlot);
                Point2D offset = new Point2D(0, 0);
                if (big) {
                    List<BasicOffset> offsets = weaponLocations.getPositioning().getBig();
                    for (BasicOffset basicOffset : offsets) {
                        if (basicOffset.getSide().equals(slot.getSide())) {
                            offset = new Point2D(basicOffset.getXOffset(), basicOffset.getYOffset());
                            break;
                        }
                    }
                } else {
                    List<BasicOffset> offsets = weaponLocations.getPositioning().getSmall();
                    for (BasicOffset basicOffset : offsets) {
                        if (basicOffset.getSide().equals(slot.getSide())) {
                            offset = new Point2D(basicOffset.getXOffset(), basicOffset.getYOffset());
                            break;
                        }
                    }
                }
                imgView.setLayoutX(slot.getX() + offset.getX());
                imgView.setLayoutY(slot.getY() + offset.getY());
                pane.getChildren().add(imgView);
            } else {
                ImageView placeHolder = getWeaponPlaceHolder(weaponSlot.getSide());
                int id = weaponSlot.getId();
                BasicSlot slot = getSlotModel(id);
                placeHolder.setLayoutX(slot.getX());
                placeHolder.setLayoutY(slot.getY());
                pane.getChildren().add(placeHolder);
            }
        }
        return pane;
    }

    private ImageView getShipDeck(EShipType shipType) {
        String imageName = null;
        switch (shipType) {
            case SNAIKKA:
                imageName = "images/deckSNAIKKA";
                break;
            case CRAYER:
                imageName = "images/deckCRAYER";
                break;
            case COG:
                imageName = "images/deckCOG";
                break;
            case HOLK:
                imageName = "images/deckHOLK";
                break;
        }
        Image img = imageLoader.getImage(imageName);
        return new ImageView(img);
    }

    private ImageView getWeaponPlaceHolder(ESide side) {
        String imageName = null;
        switch (side) {
            case LEFT:
                imageName = "images/WeaponPlaceHolderLeft";
                break;
            case RIGHT:
                imageName = "images/WeaponPlaceHolderRight";
                break;
        }
        Image img = imageLoader.getImage(imageName);
        return new ImageView(img);
    }

    private ImageView getShipWeapon(IWeaponSlot weaponSlot) {
        ESide side = weaponSlot.getSide();
        EWeapon weapon = (EWeapon) weaponSlot.getWeapon().get();
        String imageName = null;
        switch (weapon) {
            case CANNON:
                if (ESide.RIGHT.equals(side)) {
                      imageName = "images/canonTopRight";
                } else {
                    imageName = "images/canonTopLeft";
                }
                break;
            case BALLISTA_BIG:
                if (ESide.RIGHT.equals(side)) {
                    imageName = "images/ballistaBigTopRight";
                } else {
                    imageName = "images/ballistaBigTopLeft";
                }
                break;
            case BALLISTA_SMALL:
                if (ESide.RIGHT.equals(side)) {
                    imageName = "images/ballistaSmallTopRight";
                } else {
                    imageName = "images/ballistaSmallTopLeft";
                }
                break;
            case BOMBARD:
                if (ESide.RIGHT.equals(side)) {
                    imageName = "images/bombardTopRight";
                } else {
                    imageName = "images/bombardTopLeft";
                }
                break;
            case TREBUCHET_BIG:
                if (ESide.RIGHT.equals(side)) {
                    imageName = "images/trebuchetBigTopRight";
                } else {
                    imageName = "images/trebuchetBigTopLeft";
                }
                break;
            case TREBUCHET_SMALL:
                if (ESide.RIGHT.equals(side)) {
                    imageName = "images/trebuchetSmallTopRight";
                } else {
                    imageName = "images/trebuchetSmallTopLeft";
                }
                break;
        }
        Image img = imageLoader.getImage(imageName);
        return new ImageView(img);
    }

    private BasicSlot getSlotModel(int id) {
        for (WeaponLocation weaponLocation : weaponLocations.getWeaponLocation()) {
            for (BasicSlot basicSlot : weaponLocation.getSlot()) {
                if (id == basicSlot.getId()) {
                    return basicSlot;
                }
            }
        }
        return null;
    }
}
