package ch.sahits.game.openpatrician.display.dialog.shipyard;

import ch.sahits.game.openpatrician.event.NoticeBoardClose;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.city.IShipDueDate;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.EventHandler;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.Pane;
import javafx.scene.text.Text;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;

/**
 * Listing all the costruction ships and provide feature to cancel the order.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 17, 2013
 */
@Component("shipyardRepairListDialog")
@Scope(value = "prototype")
@ClassCategory({EClassCategory.DIALOG, EClassCategory.SINGLETON_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ShipyardRepairListDialogV2 extends CloseButtonDialog {
    private final Logger logger = LogManager.getLogger(getClass());

    /** Reference to the city view model */
    private final ICityPlayerProxyJFX city;
    @ObjectPropertyType(IShipDueDate.class)
    private final ObjectProperty<IShipDueDate> selectedShip;
    @Autowired
    private Date date;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    private Locale locale;
    @Autowired
    private ModelTranslations modelTranslations;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private DecoratedTextFactory textFactory;

    private GridPane grid;

    public ShipyardRepairListDialogV2(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
        selectedShip = new SimpleObjectProperty<>(this, "selectedShip", null);
    }
    @PostConstruct
    private void initializeComponents() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardRepairListDialogV2.title", new Object[]{}, locale.getCurrentLocal()));
        final IShipyard state = city.getCity().getCityState().getShipyardState();
        List<IShipDueDate> shipRepairList = state.getShipRepairList();
        if (shipRepairList.isEmpty()) {
            final String s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardRepairListDialogV2.noShipsRepairing", new Object[]{}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(s, new HashMap<>());
            text.setLayoutY(100);
            getContent().addAll(text);
        } else {
            Comparator<IShipDueDate> comparator = new Comparator<IShipDueDate>() {
                @Override
                public int compare(IShipDueDate dueDate1, IShipDueDate dueDate2) {
                    return dueDate1.getDueDate().compareTo(dueDate2.getDueDate());
                }
            };
            Collections.sort(shipRepairList, comparator);
            grid = new GridPane();
            grid.setLayoutX(2*FRAME_BORDER);
            grid.setLayoutY(100);
            String s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionListDialogV2.dueDate", new Object[]{}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(s, new HashMap<>());
            grid.add(text, 0, 0);
            grid.getColumnConstraints().add(new ColumnConstraints(100));
            s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardRenameDialogV2.shipName", new Object[]{}, locale.getCurrentLocal());
            text = textFactory.createDecoratedText(s, new HashMap<>());
            grid.add(text, 1, 0);
            grid.getColumnConstraints().add(new ColumnConstraints(250));
            s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionListDialogV2.shipType", new Object[]{}, locale.getCurrentLocal());
            text = textFactory.createDecoratedText(s, new HashMap<>());
            grid.add(text, 2, 0);
            grid.getColumnConstraints().add(new ColumnConstraints(100));

            for (int row = 1; row <= shipRepairList.size(); row++) {
                final int index = row - 1;
                IShipDueDate dueDate = shipRepairList.get(index);
                LocalDateTime time = dueDate.getDueDate();
                final IShip ship = dueDate.getShip();
                boolean isOwnedByPlayer = ship.getOwner().equals(city.getPlayer());
                Text text1 = new Text(modelTranslations.toDisplayString(time));
                text1.setId("date-"+index);
                text1.getStyleClass().add("dialogText");
                text1.setLayoutY(15);
                Pane col1 = new Pane();
                col1.getChildren().add(text1);
                grid.add(col1, 0, row);
                Text text2 = new Text(ship.getName());
                text2.setId("name-"+index);
                text2.getStyleClass().add("dialogText");
                text2.setLayoutY(15);
                Pane col2 = new Pane();
                col2.getChildren().add(text2);
                grid.add(col2, 1, row);
                Text text3 = new Text(modelTranslations.getLocalDisplayName(ship.getShipType()));
                text3.setId("shipType-"+index);
                text3.getStyleClass().add("dialogText");
                text3.setLayoutY(15);
                Pane col3 = new Pane();
                col3.getChildren().add(text3);
                grid.add(col3, 2, row);

                SelectLineEventHandler selectionLineEventHandler = new SelectLineEventHandler(dueDate, col1, col2, col3);

                if (isOwnedByPlayer) {
                    text1.setOnMouseReleased(selectionLineEventHandler);
                    text2.setOnMouseReleased(selectionLineEventHandler);
                    text3.setOnMouseReleased(selectionLineEventHandler);
                }
            } // end for

            final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardRepairListDialogV2.cancel", new Object[]{}, locale.getCurrentLocal()));
            action.getStyleClass().add("actionButton");
            action.setId("action");
            action.setOnAction(mouseEvent -> {
                try {
                    state.cancelRepair(selectedShip.get().getShip());
                    city.getPlayersShips().add(selectedShip.get().getShip());
                    executeOnCloseButtonClicked();
                } catch (RuntimeException e) {
                    logger.error("Failed to accept repair list action", e);
                }
            });
            final int actionButtonX = (WIDTH - 124) / 2;
            action.setLayoutX(actionButtonX);
            action.setLayoutY(CLOSE_BTN_Y_POS - 24);
            BooleanBinding actionEnabled = new BooleanBinding() {
                {
                    super.bind(selectedShip);
                }
                @Override
                protected boolean computeValue() {
                    return selectedShip.get() != null;
                }
            };
            action.setDisable(true);
            actionEnabled.addListener((observableValue, oldValue, newValue) -> {
                action.setDisable(!newValue);
            });

            getContent().addAll(grid, action);
        }
    }

    private class SelectLineEventHandler implements EventHandler<MouseEvent> {
        private final IShipDueDate ship;
        private final List<Pane> row;

        private SelectLineEventHandler(IShipDueDate ship, Pane...row) {
            this.ship = ship;
            this.row = Arrays.asList(row);
        }

        @Override
        public void handle(MouseEvent mouseEvent) {
            try {
                selectedShip.set(ship);
                for (Node node : grid.getChildren()) {
                    if (row.contains(node)) {
                        node.setStyle("-fx-background-color: grey");
                    } else {
                        node.setStyle("-fx-background-color: transparent");
                    }
                }
            } catch (RuntimeException e) {
                logger.error("Failed to select ship", e);
            }
        }
    }
    @Override
    public void executeOnCloseButtonClicked() {
        clientEventBus.post(new NoticeBoardClose());
        super.executeOnCloseButtonClicked();
    }

}
