package ch.sahits.game.openpatrician.display.dialog.tavern;

import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.event.ViewChangeEvent;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.display.dialog.IDialogCentralButton;
import ch.sahits.game.openpatrician.display.dialog.service.TavernDialogUtil;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.display.javafx.MainGameView;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.javafx.model.EDialogType;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.city.impl.TavernState;
import ch.sahits.game.openpatrician.model.people.IBuyer;
import ch.sahits.game.openpatrician.model.people.IConcurrent;
import ch.sahits.game.openpatrician.model.people.ICourier;
import ch.sahits.game.openpatrician.model.people.IEscorte;
import ch.sahits.game.openpatrician.model.people.IFugitive;
import ch.sahits.game.openpatrician.model.people.IPatrol;
import ch.sahits.game.openpatrician.model.people.IPirateHunter;
import ch.sahits.game.openpatrician.model.people.ISideRoomPerson;
import ch.sahits.game.openpatrician.model.people.ISmuggler;
import ch.sahits.game.openpatrician.model.people.IThieve;
import ch.sahits.game.openpatrician.model.people.ITransportTrader;
import ch.sahits.game.openpatrician.model.people.ITreasureMapOwner;
import ch.sahits.game.openpatrician.model.people.IWarehouseTenant;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import javafx.application.Platform;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.DoubleBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.EventHandler;
import javafx.scene.input.MouseEvent;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 22, 2013
 */
public abstract class TavernBaseSideRoomPersonDialog extends CloseButtonDialog implements IDialogCentralButton{
    protected final Logger logger = LogManager.getLogger(getClass());

    protected final ICityPlayerProxyJFX city;
    private final TavernState tavernState;
    protected final ObjectProperty<ISideRoomPerson> currentPerson;
    @Autowired
    protected Date date;
    @Autowired
    protected TavernDialogUtil tavernDialogUtil;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    protected Locale locale;

    @Autowired
    protected MessageSource messageSource;

    private OpenPatricianLargeWaxButton acceptBtn;


    public TavernBaseSideRoomPersonDialog(ICityPlayerProxyJFX city, ISideRoomPerson person) {
        super();
        this.city = city;
        tavernState = city.getCity().getCityState().getTavernState();
        currentPerson =  new SimpleObjectProperty<>(this, "currentPerson", person);
    }
    protected final void initializeButtons(){
        final BooleanBinding hasMorePersons = hasNextPersonBinding();
        acceptBtn = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernBaseSideRoomPersonDialog.accept", new Object[]{}, locale.getCurrentLocal()));
        acceptBtn.getStyleClass().add("actionButton");
        acceptBtn.setLayoutX(BUTTON_X);
        acceptBtn.setOnAction(createAcceptHandler());
        OpenPatricianLargeWaxButton nextBtn = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernBaseSideRoomPersonDialog.next", new Object[]{}, locale.getCurrentLocal()));
        nextBtn.getStyleClass().add("actionButton");
        nextBtn.setLayoutX(BUTTON_X);
        nextBtn.setLayoutY(LOWER_BUTTON_Y);
        nextBtn.setOnAction(createNextPersonHandler());
        nextBtn.visibleProperty().bind(hasMorePersons);
        acceptBtn.layoutYProperty().bind(new DoubleBinding() {
            {
                super.bind(hasMorePersons);
            }
            @Override
            protected double computeValue() {
                if (hasMorePersons.get()) {
                    return UPPER_BUTTON_Y;
                } else {
                    return LOWER_BUTTON_Y;
                }
            }
        });
        final BooleanBinding disableAcceptButton = disableAcceptButton();
        acceptBtn.setDisable(disableAcceptButton.get());
        disableAcceptButton.addListener((observableValue, oldValue, newValue) -> {
            acceptBtn.setDisable(newValue);
        });
        getContent().addAll(nextBtn, acceptBtn);
    }
    protected void removeAcceptButton() {
        Platform.runLater(() -> getContent().remove(acceptBtn));
    }

    /**
     * Define the action that is executed on hitting the accept button
     * @return
     */
    protected abstract EventHandler<MouseEvent> createAcceptHandler();

    /**
     * Create the event handler to display the next person
      * @return
     */
    private EventHandler<MouseEvent> createNextPersonHandler() {
        return mouseEvent -> {
            try {
                tavernState.talkingToProperty().get().remove(currentPerson.get());
                List<ISideRoomPerson> persons = tavernState.presentPersonsInSideRoomBinding().get();
                for (int i = 0; i < persons.size(); i++) {
                    ISideRoomPerson person = persons.get(i);
                    if (person.equals(currentPerson.get())) {
                        if (i == persons.size() - 1) {
                            currentPerson.set(persons.get(0));
                        } else {
                            currentPerson.set(persons.get(i + 1));
                        }
                        break;
                    }
                } // end for
                tavernState.talkingToProperty().get().put(currentPerson.get(), city.getPlayer());
                EDialogType nextPerson = map(currentPerson.get());
                clientEventBus.post(new ViewChangeEvent(MainGameView.class, nextPerson));
            } catch (RuntimeException e) {
                logger.error("Failed to switch to next person", e);
            }
        };
    }

    /**
     * Binding indicating if there is another person in the side room.
     * @return
     */
    private BooleanBinding hasNextPersonBinding() {
        return new BooleanBinding() {
            {
                super.bind(tavernState.presentPersonsInSideRoomBinding());
            }
            @Override
            protected boolean computeValue() {
                return tavernState.presentPersonsInSideRoomBinding().get().size() > 1;
            }
        };
    }

    private EDialogType map(ISideRoomPerson person) {
        if (person instanceof ITransportTrader) {
            return EDialogType.TAVERN_TRANSPORT_TRADER;
        }
        if (person instanceof IBuyer) {
           return EDialogType.TAVERN_BUYER;
        }
        if (person instanceof IEscorte) {
            return EDialogType.TAVERN_ESCORTE;
        }
        if (person instanceof IFugitive) {
           return EDialogType.TAVERN_FUGATIVE;
        }
        if (person instanceof ICourier) {
            return EDialogType.TAVERN_COURIER;
        }
        if (person instanceof IPatrol) {
            return EDialogType.TAVERN_PATROL;
        }
        if (person instanceof IThieve) {
            return EDialogType.TAVERN_BURGLAR;
        }
        if (person instanceof IConcurrent) {
            return EDialogType.TAVERN_CONCURRENT;
        }
        if (person instanceof IPirateHunter) {
            return EDialogType.TAVERN_PIRATE_HUNTER;
        }
        if (person instanceof IWarehouseTenant) {
            return EDialogType.TAVERN_WAREHOUSE_TENANT;
        }
        if (person instanceof ISmuggler) {
           return EDialogType.TAVERN_SMUGGLER;
        }
        if (person instanceof ITreasureMapOwner) {
            return EDialogType.TAVERN_TREASUREMAP_OWNER;
        }
        throw new RuntimeException("Unhandled side room person: "+person.getClass().getName());
    }


    /**
     * Binding defining the disabling of the accept button.
     * Binding evaluates to true when the accept button should be disabled.
     * @return
     */
    protected abstract BooleanBinding disableAcceptButton();

    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_TAVERN);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }
    @Override
    public void close() {
        super.close();
        tavernDialogUtil.stopTalkingToPerson(tavernState, city.getPlayer());
    }

}
