package ch.sahits.game.openpatrician.display.dialog.tavern;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.display.dialog.service.DialogUtil;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.openpatrician.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.model.people.ISailorState;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.beans.value.WeakChangeListener;
import javafx.geometry.HPos;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.RowConstraints;
import javafx.scene.layout.VBox;
import javafx.scene.text.Text;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.HashMap;

/**
 * Dialog for hiring sailors.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 20, 2013
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class TavernSailorDialog extends CloseButtonDialog {
    private final Logger logger = LogManager.getLogger(getClass());


    private final ICityPlayerProxyJFX city;
    private final ISailorState sailorState;
    @ObjectPropertyType(IShip.class)
    private final ObjectProperty<IShip> currentShip;
    @Autowired
    private DialogUtil dialogUtil;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    private Locale locale;

    @Autowired
    private MessageSource messageSource;
    @Autowired
    private DecoratedTextFactory textFactory;
    private ChangeListener<Number> sailorNumberListener;

    public TavernSailorDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
        sailorState = city.getCity().getCityState().getTavernState().getSailors();
        this.currentShip = new SimpleObjectProperty<>(this, "currentShip", null);
        if (city.getActiveShip() instanceof IShip) {
            IShip ship = (IShip) city.getActiveShip();
            currentShip.setValue(ship);
        }
    }
    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernSailorDialog.title", new Object[]{}, locale.getCurrentLocal()));
        VBox topText = new VBox();
        topText.setLayoutX(2 * FRAME_BORDER);
        topText.setLayoutY(100);

        Text firstLine = new Text();
        firstLine.getStyleClass().add("dialogText");
        StringBinding firstLineBinding = new StringBinding() {
            {
                super.bind(sailorState.numberOfSailorsProperty());
            }
            @Override
            protected String computeValue() {
                if (sailorState.getNumberOfSailors() > 1) {
                    return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernSailorDialog.dialog1", new Object[]{ sailorState.getNumberOfSailors()}, locale.getCurrentLocal());
                } else if (sailorState.getNumberOfSailors() == 1) {
                    return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernSailorDialog.dialog2", new Object[]{}, locale.getCurrentLocal());
                } else {
                    return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernSailorDialog.dialog3", new Object[]{}, locale.getCurrentLocal());
                }
            }
        };
        firstLine.textProperty().bind(firstLineBinding);
        Text secondLine = new Text();
        secondLine.getStyleClass().add("dialogText");
        secondLine.setText(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernSailorDialog.dialog4", new Object[]{}, locale.getCurrentLocal()));
        topText.getChildren().addAll(firstLine, secondLine);
        getContent().add(topText);

        final IntegerProperty sailorsAdditionalOnShip = new SimpleIntegerProperty(0);
        final IntegerProperty sailorsInTavern = new SimpleIntegerProperty(sailorState.getNumberOfSailors());
        BooleanBinding enableAction = new BooleanBinding() {
            {
                super.bind(currentShip,sailorsAdditionalOnShip);
            }
            @Override
            protected boolean computeValue() {
                return currentShip.get() != null
                        && sailorsAdditionalOnShip.get() > 0;
            }
        };
        currentShip.addListener((observableValue, oldValue, newValue) -> {
            sailorsAdditionalOnShip.set(0);
            sailorsInTavern.set(sailorState.getNumberOfSailors()); // reset
        });
        sailorNumberListener = (observableValue, oldValue, newValue) -> {
                int inTavern = newValue.intValue() - sailorsAdditionalOnShip.get();
                if (inTavern < 0) {
                    sailorsAdditionalOnShip.set(Math.max(sailorsAdditionalOnShip.get() + inTavern, 0));
                    inTavern = 0;
                }
                sailorsInTavern.set(inTavern);
        };
        sailorState.numberOfSailorsProperty().addListener(new WeakChangeListener<>(sailorNumberListener));


        GridPane sailorsPane = createSailorMoveControls(sailorsAdditionalOnShip, sailorsInTavern, enableAction);
        getContent().add(sailorsPane);

        String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernSailorDialog.dialog5", new Object[]{}, locale.getCurrentLocal());
        DecoratedText selectedShip = decoratedTextFactory.createDecoratedText(template, new HashMap<>());
        selectedShip.setLayoutX(2 * FRAME_BORDER);
        selectedShip.setLayoutY(250);
        getContent().add(selectedShip);

        if (currentShip.get() == null) {
            template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernCaptainDialog.noShip", new Object[]{}, locale.getCurrentLocal());
            DecoratedText noShip = decoratedTextFactory.createDecoratedText(template, new HashMap<>());
            noShip.setLayoutX(2*FRAME_BORDER);
            noShip.setLayoutY(300);
            getContent().add(noShip);
        } else {
            GridPane shipSelectionPane = dialogUtil.createShipSelection3LinesForShips(city.getPlayersShips(), currentShip);
            shipSelectionPane.setLayoutX(2*FRAME_BORDER);
            shipSelectionPane.setLayoutY(300);
            getContent().add(shipSelectionPane);
        }

        final int actionButtonX = (WIDTH - 124) / 2;
        final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernSailorDialog.hire", new Object[]{}, locale.getCurrentLocal()));
        action.getStyleClass().add("actionButton");
        action.setOnAction(mouseEvent -> {
            try {
                int hireNbSailors = sailorsAdditionalOnShip.get();
                currentShip.get().setNumberOfSailors(hireNbSailors + currentShip.get().getNumberOfSailors());
                sailorsAdditionalOnShip.setValue(0);
                sailorState.setNumberOfSailors(sailorsInTavern.get());
                city.getCity().getCityState().getPopUpdateStatistic().hireSailors(hireNbSailors);
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                logger.error("Failed to accept sailor hireing", e);
            }

        });
        action.setLayoutX(actionButtonX);
        action.setLayoutY(CLOSE_BTN_Y_POS - 24);

        action.setDisable(!enableAction.get());
        enableAction.addListener((observableValue, oldValue, newValue) -> {
             action.setDisable(!newValue);
        });
        getContent().add(action);


    }


    /**
     * Create the grid pane for moving the sailors between the tavern and the ship.
     * The sailors are only updated, when they are hired.
     * @param sailorsOnShip property indicating the current sailors assigned to the ship (might
     *                      differ from the number actually on the ship)
     * @param sailorsInTavern property indicating the current sailors in the tavern.
     * @param enableAction binding indicating if the action is enabled.
     * @return Grid pane.
     */
    private GridPane createSailorMoveControls(final IntegerProperty sailorsOnShip, final IntegerProperty sailorsInTavern, BooleanBinding enableAction) {
        GridPane sailorsPane = new GridPane();
        sailorsPane.setLayoutX(100);
        sailorsPane.setLayoutY(150);
        RowConstraints rowConstraint = new RowConstraints(24);
        sailorsPane.getRowConstraints().addAll(rowConstraint, rowConstraint);
        ColumnConstraints colConstraint = new ColumnConstraints(64);
        sailorsPane.getColumnConstraints().addAll(colConstraint, colConstraint, colConstraint, colConstraint);
        String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernSailorDialog.tavern", new Object[]{}, locale.getCurrentLocal());
        DecoratedText tavernTxt = decoratedTextFactory.createDecoratedText(template, new HashMap<>());
        sailorsPane.add(tavernTxt, 0, 0);
        GridPane.setHalignment(tavernTxt, HPos.CENTER);
        StringBinding shipBinding = new StringBinding() {
            {
                super.bind(currentShip);
            }
            @Override
            protected String computeValue() {
                if (currentShip.get() != null) {
                    int nbSailors = currentShip.get().getNumberOfSailors();
                    return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.tavern.TavernSailorDialog.ship", new Object[]{nbSailors}, locale.getCurrentLocal());
                } else {
                    return "";
                }

            }
        };
        Text shipTxt = new Text();
        shipTxt.textProperty().bind(shipBinding);
        shipTxt.getStyleClass().add("dialogText");
        sailorsPane.add(shipTxt, 3, 0);
        GridPane.setHalignment(shipTxt, HPos.CENTER);

        Text nbInTavern = new Text();
        nbInTavern.getStyleClass().add("dialogText");
        nbInTavern.textProperty().bind(sailorsInTavern.asString());
        sailorsPane.add(nbInTavern, 0, 1);
        GridPane.setHalignment(nbInTavern, HPos.CENTER);
        final OpenPatricianSmallWaxButton toTavernBtn = new OpenPatricianSmallWaxButton("<");
        toTavernBtn.setOnAction(mouseEvent -> {
            try {
                if (sailorsOnShip.get() > 0) {
                    sailorsOnShip.set(sailorsOnShip.get() - 1);
                    sailorsInTavern.set(sailorsInTavern.get() + 1);
                }
            } catch (RuntimeException e) {
                logger.error("Failed to add sailors", e);
            }
        });
        if (sailorsOnShip.get() < 1 || currentShip.get() == null) {
            toTavernBtn.setDisable(true);
        }
        sailorsPane.add(toTavernBtn, 1, 1);
        GridPane.setHalignment(toTavernBtn, HPos.CENTER);
        final OpenPatricianSmallWaxButton toShipBtn = new OpenPatricianSmallWaxButton(">");
        toShipBtn.setOnAction(mouseEvent -> {
            try {
                if (sailorsInTavern.get() > 0) {
                    sailorsOnShip.set(sailorsOnShip.get() + 1);
                    sailorsInTavern.set(sailorsInTavern.get() - 1);
                    enableAction.invalidate();
                }
            } catch (RuntimeException e) {
                logger.error("Failed to remove sailors", e);
            }
        });
        sailorsOnShip.addListener(new ChangeListener<Number>() {
            @Override
            public void changed(ObservableValue<? extends Number> observableValue,
                                Number oldValue, Number newValue) {
                if (newValue.intValue() <= 0) {
                    toTavernBtn.setDisable(true);
                } else {
                    toTavernBtn.setDisable(false);
                }
                if (sailorsInTavern.get() <= 0 || newValue.intValue() + currentShip.get().getNumberOfSailors() >= currentShip.get().getMaxNumberOfSailors()) {
                    toShipBtn.setDisable(true);
                } else {
                    toShipBtn.setDisable(false);
                }
            }
        });
        sailorsInTavern.addListener(new ChangeListener<Number>() {
            @Override
            public void changed(ObservableValue<? extends Number> observableValue,
                                Number oldValue, Number newValue) {
                if (newValue.intValue() <= 0 || sailorsOnShip.get() + currentShip.get().getNumberOfSailors() >= currentShip.get().getMaxNumberOfSailors()) {
                    toShipBtn.setDisable(true);
                } else {
                    toShipBtn.setDisable(false);
                }

            }
        });
        if (sailorsInTavern.get() < 1 || currentShip.get() == null || currentShip.get().getNumberOfSailors() == currentShip.get().getMaxNumberOfSailors()) {
            toShipBtn.setDisable(true);
        }
        sailorsPane.add(toShipBtn, 2, 1);
        GridPane.setHalignment(toShipBtn, HPos.CENTER);

        Text nbInShip = new Text();
        nbInShip.getStyleClass().add("dialogText");
        nbInShip.textProperty().bind(sailorsOnShip.asString());
        sailorsPane.add(nbInShip, 3, 1);
        GridPane.setHalignment(nbInShip, HPos.CENTER);
        return sailorsPane;
    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_TAVERN);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }

}
