package ch.sahits.game.openpatrician.display.dialog.tavern;

import ch.sahits.game.openpatrician.display.event.handler.impl.EventHandlerFactory;
import ch.sahits.game.openpatrician.display.event.handler.impl.ShipArrivalForLoading;
import ch.sahits.game.openpatrician.display.event.task.ClientTaskFactory;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.people.ISmuggler;
import ch.sahits.game.openpatrician.model.ship.IShip;
import javafx.beans.binding.BooleanBinding;
import javafx.event.EventHandler;
import javafx.scene.input.MouseEvent;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Random;

import static com.google.common.collect.Lists.newArrayList;

/**
 * Dialog for the smuggler. Upon accepting a random ship with enough capacity is chosen in the port.
 * If no such ship is available there is a waiting period for a ship. The ware is loaded onto the chosen
 * ship and occupies some space. Once the ship with the loaded ware reaches the destination the ware is
 * unloaded. There is a chance the ship owner is caught in the act.
 *
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 27, 2013
 * @see ShipArrivalForLoading
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class TavernSmugglerDialog extends TavernBaseSideRoomPersonDialog {
    @Autowired
    private EventHandlerFactory eventHandlerFactory;
    @Autowired
    private Random rnd;
    @Autowired
    private TimedUpdatableTaskList timedTaskListener;
    @Autowired
    private ClientTaskFactory taskFactory;


    public TavernSmugglerDialog(ICityPlayerProxyJFX city) {
        super(city, city.getCity().getCityState().getTavernState().getSmuggler());
    }
    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernSmugglerDialog.title", new Object[]{}, locale.getCurrentLocal()));
        initializeButtons();
        ISmuggler smuggler = (ISmuggler) currentPerson.get();

        String loadedText = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernSmugglerDialog.dialog", new Object[0], locale.getCurrentLocal());


        HashMap<String, Object> parameters = new HashMap<>();
        parameters.put("amount", smuggler.getAmount());
        DecoratedText dt = decoratedTextFactory.createDecoratedText(loadedText, parameters);


        getContent().addAll(dt);
    }

    @Override
    protected EventHandler<MouseEvent> createAcceptHandler() {
        return mouseEvent -> {
            try {
                List<IShip> ships = newArrayList(city.getPlayersShips()); // we need a copy
                Collections.shuffle(ships);
                ISmuggler smuggler = (ISmuggler) currentPerson.get();
                boolean shipFound = false;
                for (IShip ship : ships) {
                    if (ship.getCapacity() > smuggler.getAmount()) {
                        ship.setOccupiedSpace(smuggler.getAmount());
                        shipFound = true;
                        break;
                    }
                }
                if (!shipFound) {
                    LocalDateTime deadline = date.getCurrentDate().plusDays(10);
                    eventHandlerFactory.getShipArrivalForLoadingHandler(city.getCity(), city.getPlayer(), deadline, smuggler.getAmount(), smuggler.getPremium());
                }
                int probability = rnd.nextInt(3);
                if (probability < 1) {
                    timedTaskListener.add((TimedTask) taskFactory.getDelayedCaughtAction(city.getPlayer(), city.getCity()));
                }
                city.getPlayer().updateCrimialDrive(1);
                smuggler.leave();
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                logger.error("Failed to accept smugglar offer", e);
            }
        };
    }

    @Override
    protected BooleanBinding disableAcceptButton() {
        return new BooleanBinding() {
            {
                super.bind(city.getPlayersShips());
                // todo: andi 23/12/13: should not consider ships that are automated in some way (these are the ships which's capacity may change)
            }
            @Override
            protected boolean computeValue() {
                ISmuggler smuggler = (ISmuggler) currentPerson.get();
                return tavernDialogUtil.noShipWithEnoughCapacityAvailable(city, smuggler.getAmount()/10);
            }
        };
    }



}
