package ch.sahits.game.openpatrician.display.dialog.tavern;

import ch.sahits.game.openpatrician.display.event.task.ClientTaskFactory;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.people.IWarehouseTenant;
import javafx.beans.binding.BooleanBinding;
import javafx.event.EventHandler;
import javafx.scene.input.MouseEvent;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Optional;
import java.util.Random;

/**
 *
 * The dialog for a warehouse tenant. Upon accepting the agreed upon amount is stored in the warehouse for
 * an arbitrary amount of time. Each week the rent is deposited. At the end the ware is removed.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 26, 2013
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class TavernWarehouseTenantDialog extends TavernBaseSideRoomPersonDialog {
    @Autowired
    private Random rnd;

    @Autowired
    private TimedUpdatableTaskList list;
    @Autowired
    private ClientTaskFactory taskFactory;


    public TavernWarehouseTenantDialog(ICityPlayerProxyJFX city) {
        super(city, city.getCity().getCityState().getTavernState().getWarehouseTenant());
    }
    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernWarehouseTenantDialog.title", new Object[]{}, locale.getCurrentLocal()));
        initializeButtons();
        IWarehouseTenant tenant = (IWarehouseTenant) currentPerson.get();

        String loadedText = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernWarehouseTenantDialog.dialog",
                new Object[]{tenant.getName()}, locale.getCurrentLocal());


        HashMap<String, Object> parameters = new HashMap<>();
        parameters.put("amount", tenant.getAmount());
        parameters.put("price", tenant.getRentPerWeek());
        DecoratedText dt = decoratedTextFactory.createDecoratedText(loadedText, parameters);


        getContent().addAll(dt);
    }

    @Override
    protected EventHandler<MouseEvent> createAcceptHandler() {
        return mouseEvent -> {
            try {
                final IPlayer player = city.getPlayer();
                final IWarehouseTenant tenant = (IWarehouseTenant) currentPerson.get();
                final Optional<ITradingOffice> optOffice = city.getPlayer().findTradingOffice(city.getCity());
                if (optOffice.isPresent()) {
                    ITradingOffice office = optOffice.get();
                    office.setOccupiedSpace(office.getOccupiedSpace() + tenant.getAmount());
                    int weeks = rnd.nextInt(15) + 5;
                    LocalDateTime currentDate = date.getCurrentDate();
                    for (int i = 0; i < weeks; i++) {
                        currentDate = currentDate.plusWeeks(1);
                        TimedTask task = taskFactory.getWarehouseTenantPayment(player, tenant, currentDate);
                        list.add(task);
                    }
                    TimedTask task = taskFactory.getWarehouseTenantLeaving(office, tenant, date.getCurrentDate().plusWeeks(weeks));
                    list.add(task);
                    tenant.leave();
                }
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                logger.error("Failed to accept warehouse tentants offer offer", e);
            }
        };
    }

    @Override
    protected BooleanBinding disableAcceptButton() {
        return new BooleanBinding() {
            @Override
            protected boolean computeValue() {
                return !city.getPlayer().findTradingOffice(city.getCity()).isPresent();
            }
        };
    }
}
