package ch.sahits.game.openpatrician.display.event.task;

import ch.sahits.game.openpatrician.event.data.ShipAttackEvent;
import ch.sahits.game.openpatrician.event.handler.TimedEventListener;
import ch.sahits.game.openpatrician.display.event.handler.impl.EventHandlerFactory;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;

/**
 * Order of the opponent to attack the original client. The offer is postponed 10 days and will
 * be canceled once the opponents ship is attacked.
 * This task adds itself the the timed event list.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Feb 24, 2013
 *
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class OpponentsOrder extends TimedTask  {
    private final IPlayer opponent;
    private final IPlayer client;
    private final IPlayer player;
    private final int premium;

    @Autowired
    private TimedUpdatableTaskList list;
    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("clientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientEventBus;
    @Autowired
    @XStreamOmitField
    private EventHandlerFactory eventHandlerFactory;
    @Autowired
    private Date date;
    @Autowired
    @XStreamOmitField
    private Locale locale;

    @Autowired
    @XStreamOmitField
    @Qualifier("messageSource")
    private MessageSource messageSource;

    private TimedEventListener order;

    public OpponentsOrder(IPlayer opponent,
                          IPlayer client, IPlayer player, int premium) {
        this.opponent = opponent;
        this.client = client;
        this.player = player;
        this.premium = premium;
    }
    @PostConstruct
    private void initializeEventBus() {
        setExecutionTime(date.getCurrentDate().plusDays(10));
        clientServerEventBus.register(this);
        list.add(this);

    }

    @Override
    public void run() {

        clientEventBus.post(new DisplayMessage("ch.sahits.game.graphic.event.task.OpponentsOrder.message1", new Object[]{client.getName(), client.getLastName(),premium}));
        order =  eventHandlerFactory.getSinkConcurrentsShipOrder(client, opponent, 10, premium, null);
        clientServerEventBus.unregister(order);
    }

    /**
     * Handling ships attacks.
     * @param event
     */
    @Subscribe
    public void handleShipAttack(ShipAttackEvent event) {
        IShipOwner attackedOwner = event.getAttackedShip().getOwner();
        IShipOwner attackingOwner = event.getAttackingShip().getOwner();
        if (attackedOwner != null && attackedOwner.equals(opponent)
                && attackingOwner != null && attackingOwner.equals(player)) {
            clientServerEventBus.unregister(this);
            if (order != null) {
                clientServerEventBus.unregister(order);
            }
        }

    }}
