package ch.sahits.game.openpatrician.display.javafx.control;

import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NavigationStateChange;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.event.ViewChangeEvent;
import ch.sahits.game.openpatrician.event.data.MapUpdateCityAdd;
import ch.sahits.game.openpatrician.event.data.NewGameClient;
import ch.sahits.game.openpatrician.event.data.ShipEntersPortEvent;
import ch.sahits.game.openpatrician.event.data.ShipLeavingPort;
import ch.sahits.game.openpatrician.event.data.SwitchCity;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.EViewState;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.graphic.image.IImageUtilities;
import ch.sahits.game.openpatrician.display.javafx.IDialogContoller;
import ch.sahits.game.openpatrician.display.javafx.MainGameView;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ui.MapState;
import ch.sahits.game.openpatrician.utilities.IRebinabable;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.application.Platform;
import javafx.geometry.Dimension2D;
import javafx.geometry.Rectangle2D;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.Pane;
import javafx.scene.paint.Color;
import javafx.scene.shape.Circle;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

/**
 * Mini map control. This view shows the map for the sea and
 * the one of the town. However only one is visible. With the lever
 * at the side can be switched between the two.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Nov 1, 2013
 *
 */
@LazySingleton
@ClassCategory({EClassCategory.JAVAFX, EClassCategory.SINGLETON_BEAN})
public class MiniMap extends Group implements IRebinabable {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private IMap map;
    @Autowired
    private IImageUtilities imageUtilities;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    private MapState mapState;

    private ImageView imgView;
    private boolean navigationDisabled = false;
    private IDialogContoller dialogContoller; // this instance exists not at startup
    private Pane mapPane;
    private double scale;

    @PostConstruct
    private void initializeControl() {
        clientServerEventBus.register(this);
        clientEventBus.register(this);
        setManaged(false);
        mapPane = new Pane();
        imgView = new ImageView();
        imgView.setFitWidth(236);
        imgView.setFitHeight(192);
        imgView.setPreserveRatio(false);
        // scale the map to the correct size add white dots on cities that have ship in them
        // clicking on the city changes into the other city (if there is a ship or kontor
        // right clicking on a point on the map sends the active ship to that location
        mapPane.getChildren().addAll(imgView);
        getChildren().add(mapPane);

        addEventHandlers();
    }
    public void setDialogContoller(IDialogContoller dialogContoller) {
        this.dialogContoller = dialogContoller;
    }
    private void addEventHandlers() {
        setOnMouseReleased(event -> {
            if (viewState.getState() == EViewState.MAP) {
                logger.info("Switch back to port scene");
                mapState.setShowingLargeSeaMap(false);
                ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.MAIN_VIEW_PORT);
                clientEventBus.post(new ViewChangeEvent(MainGameView.class, proxy));
            } else {
                if (mapState.isMiniMapShowingSea()) {
                    logger.info("Switch to big sea map");
                    mapState.setShowingLargeSeaMap(true);
                    dialogContoller.closeDialog();
                    ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.MAIN_VIEW_SEA_MAP);
                    clientEventBus.post(new ViewChangeEvent(MainGameView.class, proxy));
                    proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_HIDE);
                    clientEventBus.post(new NoticeBoardUpdate(proxy));
                } else {
                    logger.warn("Cannot switch from town view to sea map");
                }
            }
        });
    }

    @PreDestroy
    private void destroy() {
        clientServerEventBus.unregister(this);
        clientEventBus.unregister(this);
    }

    /**
     * Initialize the player that goes with this client. The player does not change later on,
     * @param newGameDTO
     */
    @Subscribe
    public void initializeState(NewGameClient newGameDTO) {
        final IPlayer player = newGameDTO.getPlayer();
        initializeMapImage(player);
     }

    public void initializeMapImage(IPlayer player) {
        Image mapImage = imageUtilities.createMapWithCities(map, player);
        Dimension2D dim = map.getDimension();
        scale = 192 / dim.getHeight();
        double widthOrigImg = 236 / scale;
        double xlocation = player.getHometown().getCoordinates().getX();
        double x = Math.max(0, xlocation - widthOrigImg/2);
        Rectangle2D viewport = new Rectangle2D(x, 0, widthOrigImg, dim.getHeight());
        final Image transferableMap = mapImage;

        Platform.runLater(() -> {
            imgView.setViewport(viewport);
            imgView.setImage(transferableMap);
            for (ICity city : map.getCities()) {
                if (city.getCoordinates().getX() >= x && city.getCoordinates().getX() <= x + widthOrigImg) {
                    List<INavigableVessel> ships = player.findShips(city);
                    if (!ships.isEmpty()) {
                        drawShipPresenceInCity(city, x);
                    }
                }
            }
        });
    }

    private void drawShipPresenceInCity(ICity city, double offset) {
        int radius = 8;
        int cityX = (int) Math.rint(city.getCoordinates().getX());
        int cityY = (int) Math.rint(city.getCoordinates().getY());
        Circle c = new Circle((cityX - offset) * scale, cityY * scale, radius * scale, Color.WHITE);
        mapPane.getChildren().add(c);
    }


    @Subscribe
    public void handleMapChange(MapUpdateCityAdd event) {
        initializeMapImage(event.getPlayer());
    }
    @Subscribe
    public void handleShipLeavesCity(ShipLeavingPort event) {
        ICity city = event.getCity();
        final IShipOwner owner = event.getShip().getOwner();
        if (owner instanceof IHumanPlayer && owner.equals(viewState.getPlayer())) {
            List<INavigableVessel> ships = ((IHumanPlayer)owner).findShips(city);
            if (ships.isEmpty()) {
                double cityX = (int) Math.rint(city.getCoordinates().getX()) * scale;
                double cityY = (int) Math.rint(city.getCoordinates().getY()) * scale;
                double viewportOffset = imgView.getViewport().getMinX() * scale;
                for (Iterator<Node> iterator = mapPane.getChildren().iterator(); iterator.hasNext(); ) {
                    Node node = iterator.next();
                    if (node instanceof Circle) {
                        Circle c = (Circle) node;
                        if (c.getCenterX() + viewportOffset == cityX && c.getCenterY() == cityY) {
                            Platform.runLater(() -> {
                                iterator.remove();
                            });
                            break;
                        }
                    }
                }
            }
        }
    }
    @Subscribe
    public void handleShipEntersPort(ShipEntersPortEvent event) {
        ICity city = event.getCity();
        final IShipOwner owner = event.getShip().getOwner();
        if (owner instanceof IHumanPlayer && owner.equals(viewState.getPlayer())) {
            List<INavigableVessel> ships = ((IHumanPlayer)owner).findShips(city);
            if (ships.size() == 1) {
                Dimension2D dim = map.getDimension();
                scale = 192 / dim.getHeight();
                double viewportOffset = imgView.getViewport().getMinX();
                Platform.runLater(() -> drawShipPresenceInCity(city, viewportOffset));
                if (shouldSwitchToCity(city)) {
                    clientEventBus.post(new SwitchCity(city));
                }
            }
        }
    }
    private boolean shouldSwitchToCity(ICity toCity) {
        if (viewState.getCurrentCityProxy().isPresent()) {
            ICity currentCity = viewState.getCurrentCityProxy().get().getCity();
            IHumanPlayer player = viewState.getPlayer();
            if (!currentCity.findBuilding(ITradingOffice.class, Optional.of(player)).isEmpty()) {
                return false; // if there is a trading office in the current city do not switch
            }
            if (!player.findShips(currentCity).isEmpty()) {
                return true;
            } else {
                return false;
            }

        } else {
            return true;
        }
    }


    @Subscribe
    public void handleNavigationChange(NavigationStateChange event) {
        switch (event.getChange()) {
            case DISABLE_NAVIGATION:
                navigationDisabled = true;
                break;
            case ENABLE_NAVIGATION:
                navigationDisabled = false;
                break;
            default:
                logger.info("Not interested in navigation change event: "+event.getChange());
        }
    }

    @Override
    public void rebind() {
        for (Iterator<Node> iterator = mapPane.getChildren().iterator(); iterator.hasNext(); ) {
            Node node = iterator.next();
            if (node instanceof Circle) {
                iterator.remove();
            }
        }
    }
}
