package ch.sahits.game.openpatrician.display.model;

import ch.sahits.game.openpatrician.event.data.ShipBecomesUnavailableEvent;
import ch.sahits.game.openpatrician.event.data.ShipEntersPortEvent;
import ch.sahits.game.openpatrician.event.data.ShipLeavingPort;
import ch.sahits.game.openpatrician.display.javafx.control.GameStatus;
import ch.sahits.game.openpatrician.display.javafx.control.MiniMap;
import ch.sahits.game.openpatrician.display.javafx.control.ViewStatus;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.utilities.annotation.UniquePrototype;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.spring.DependentAnnotationConfigApplicationContext;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.ListBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import lombok.Getter;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.List;

/**
 * State information of the player, the city and the ships.
 * This instance is tied to the ClientViewState.
 */
@ClassCategory({EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
@UniquePrototype
public class CityPlayerProxyJFX implements ICityPlayerProxyJFX {
    @Getter
    @Setter
	private ICity city;
    @Getter
	private IHumanPlayer player;
	/** List holding all ships owned by the player currently present in the city */
	@ListType(INavigableVessel.class)
	private ObservableList<INavigableVessel> playersVessels = FXCollections.observableArrayList();
	@ListType(IShip.class)
	@XStreamOmitField
	private ListBinding<IShip> playersShips;

    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    /** Reference the active ship of the player */
    @ObjectPropertyType(INavigableVessel.class)
	private ObjectProperty<INavigableVessel> activeShip = new SimpleObjectProperty<>(null);
	@Autowired
	@XStreamOmitField
    private DependentAnnotationConfigApplicationContext context;
	/**
	/**
	 * Constructor initializing the city and the player with one ship in port.
	 * @param city
	 * @param player
	 * @param activeShip
	 */
	public CityPlayerProxyJFX(ICity city, IHumanPlayer player, INavigableVessel activeShip) {
        this.city = city;
        this.player = player;
        if (activeShip != null) {
            this.activeShip.setValue(activeShip);
            playersVessels.add(activeShip);
        }
	}

    @PostConstruct
    private void register() {
		clientServerEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
    }

	/**
	 * Activate a ship. If the ship is not one of the players or not in port or not available
	 * an {@link IllegalArgumentException} will be thrown
	 * @param ship
	 * @throws IllegalArgumentException if the ship cannot be activated due to its non existence
	 */
	@Override
	public void activateShip(INavigableVessel ship){
		activeShip.setValue(ship);
	}

	@Override
	public ObservableList<INavigableVessel> getPlayersNavalVessels(){
		return playersVessels;
	}
	/**
	 * A ship arrives in the city
	 * @param ship
	 */
	public void arrive(INavigableVessel ship){
		if (player.getFleet().contains(ship)){
			playersVessels.add(ship);
			if (playersVessels.size()==1){
				activeShip.setValue(ship);
			}
		}
		if (ship instanceof IConvoy && ((IConvoy)ship).getPlayers().contains(player)) {
			playersVessels.add(ship);
		}
	}
	/**
	 * Ship leave the city
	 * @param ship
	 */
	public void leave(INavigableVessel ship){
        playersVessels.remove(ship);
		if (ship.equals(activeShip.get())){
			activeShip.setValue(null);
            if (!playersVessels.isEmpty()) {
                activeShip.setValue(playersVessels.get(0));
            }
		}
	}

	@Override
	public ObservableList<IShip> getPlayersShips() {
		if (playersShips == null) {
			playersShips = new ListBinding<IShip>() {
				{
					super.bind(playersVessels);
				}
				@Override
				protected ObservableList<IShip> computeValue() {
					ObservableList<IShip> ships = FXCollections.observableArrayList();
					for (INavigableVessel vessel : playersVessels) {
						 if (vessel instanceof IShip) {
							 ships.add((IShip) vessel);
						 }
					}
					return ships;
				}
			};
		}
		return playersShips;
	}
    @Subscribe
    public void handleShipLeavesPort(ShipLeavingPort event) {
        if (getCity().equals(event.getCity())) {
            leave(event.getShip());
        }
    }
    @Subscribe
    public void handleShipEntersPort(ShipEntersPortEvent event) {
        if (getCity().equals(event.getCity()) ) {
            arrive(event.getShip());
        }
    }
	@Subscribe
	public void handleShipBecomesUnavailable(ShipBecomesUnavailableEvent event) {
		if (city.equals(event.getCity())) {
            leave(event.getShip());
        }
	}

    /**
     * Execute after savegame was loaded triggered from the ClientViewState.
     */
	public void postLoad() {
		List<INavigableVessel> ships = player.findShips(this.city);
		this.playersVessels.clear();
		this.playersVessels.addAll(ships);
		if (!ships.isEmpty()) {
			this.activeShip.setValue(playersVessels.get(0));
		} else {
			this.activeShip.setValue(null);
		}
        GameStatus gameStatus = context.getBean(GameStatus.class);
        gameStatus.bindToPlayer(player);
        ViewStatus viewStatus = context.getBean(ViewStatus.class);
        viewStatus.setCity(city.getName());
        MiniMap miniMap = context.getBean(MiniMap.class);
        miniMap.initializeMapImage(player);
	}

	@Override
	public INavigableVessel getActiveShip() {
		return activeShip.get();
	}
   	@Override
	public ObjectProperty<INavigableVessel> activeShipProperty() {
		return activeShip;
	}
}
