package ch.sahits.game.openpatrician.display.scene;

import ch.sahits.game.openpatrician.event.data.NewGameClient;
import ch.sahits.game.graphic.image.IImageLoader;
import ch.sahits.game.openpatrician.display.service.IJavaFXControlLoader;
import ch.sahits.game.openpatrician.display.javafx.MainGameView;
import ch.sahits.game.openpatrician.javafx.OpenPatricianScene;
import ch.sahits.game.openpatrician.display.javafx.control.GameStatus;
import ch.sahits.game.openpatrician.display.javafx.control.MainMenu;
import ch.sahits.game.openpatrician.display.javafx.control.MiniMap;
import ch.sahits.game.openpatrician.javafx.control.NoticeBoard;
import ch.sahits.game.openpatrician.display.javafx.control.SubMenu;
import ch.sahits.game.openpatrician.display.javafx.control.ViewStatus;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.application.Platform;
import javafx.scene.Group;
import javafx.scene.layout.StackPane;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

@Prototype
@ClassCategory(EClassCategory.PROTOTYPE_BEAN)
public class MainGameScene extends OpenPatricianScene {
	/** Height of the map overview. */
	private static final int MAP_OVERVIEW_HEIGHT = 207;
	/** Width of the left panels (without borders). */
	private static final int LEFT_STATUS_PANEL_WIDTH = 250;
	/** Height of the top status bar. */
	static final int TOP_STATUS_HEIGHT = 70;
	/** Width/height of the border. */
	private static final int BORDER_SPACING = 10;
	/** Height of the horizontal fries. */
	private static final int HORIZONTAL_DECO_HEIGHT = 20;
	/** Height of the menu bar. */
	static final int MENU_HEIGHT = 71;
	/** Height of the notice view. */
	static final int NOTICE_HEIGHT = 250;
	/** Width og the left panels inclusive borders. */
	static final int LEFT_PANEL_WIDTH = 310;
	/** The minimal height the display must have */
	static final int MINMIMAL_DISPLAY_HEIGHT = TOP_STATUS_HEIGHT+MAP_OVERVIEW_HEIGHT+MENU_HEIGHT+TOP_STATUS_HEIGHT+NOTICE_HEIGHT-2;
	
	private MainGameSceneBackground background;
	private GameStatus gameStatus;
	private ViewStatus viewStatus;
	private MainGameView mainGameView;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;

	private IJavaFXControlLoader controlLoader;
	private Group foreground;
	private final MiniMap miniMap;

	public MainGameScene(double width, double height, IImageLoader loader, IJavaFXControlLoader controlLoader) {
		super(new StackPane());
		this.controlLoader = controlLoader;
		// This can be improved by rendering the background as an image each time the size changes and use the image instead
		background = new MainGameSceneBackground(width, height, loader);
		
		foreground = new Group();
		foreground.setManaged(false);
		
		gameStatus = controlLoader.getGameStatus();
		gameStatus.setLayoutY(HORIZONTAL_DECO_HEIGHT);
		gameStatus.setFont(controlLoader.getFontLoader().createDefaultFont(18));
		
		viewStatus = controlLoader.getViewStatus();
		viewStatus.setWidth(width-(LEFT_PANEL_WIDTH+BORDER_SPACING));
		viewStatus.setLayoutX(LEFT_PANEL_WIDTH);
		viewStatus.setLayoutY(HORIZONTAL_DECO_HEIGHT + 10);

		miniMap = controlLoader.getMiniMap();
		miniMap.setLayoutX(5);
		miniMap.setLayoutY(TOP_STATUS_HEIGHT+8);
		
		MainMenu mainMenu = controlLoader.getMainMenu();
		mainMenu.setLayoutX(15);
		mainMenu.setLayoutY(TOP_STATUS_HEIGHT+MAP_OVERVIEW_HEIGHT+10);
		
		SubMenu subMenu = controlLoader.getSubMenu();
		subMenu.setLayoutX(BORDER_SPACING);
		subMenu.setLayoutY(TOP_STATUS_HEIGHT+MAP_OVERVIEW_HEIGHT+MENU_HEIGHT+HORIZONTAL_DECO_HEIGHT);
		
		NoticeBoard noticeBoard = controlLoader.getNoticeBoard();
		noticeBoard.setLayoutX(5);
		noticeBoard.setLayoutY(TOP_STATUS_HEIGHT+MAP_OVERVIEW_HEIGHT+MENU_HEIGHT+TOP_STATUS_HEIGHT+6);
		
		
		foreground.getChildren().addAll(gameStatus, viewStatus, miniMap, mainMenu, subMenu, noticeBoard);
		
		StackPane root = (StackPane) getRoot();
		root.getChildren().addAll(background, foreground);
	}
    @PostConstruct
    private void initializeEventSystem() {
		clientServerEventBus.register(this);
    }

	@PreDestroy
	private void unregister() {
		clientServerEventBus.unregister(this);
	}

    /**
     * Event handler for initializing the new game when the server is done.
     * @param newGameDTO data transfer object.
     */
    @Subscribe
    public void initializeGameView(NewGameClient newGameDTO) {
        double width = calculateMainGameViewWidth(background.getSceneWidth());
        double height = calculateMainGameViewHeight(background.getSceneHeight());
        mainGameView = controlLoader.getMainGameView(width, height);
        mainGameView.setLayoutX(LEFT_PANEL_WIDTH);
        mainGameView.setLayoutY(MENU_HEIGHT - 1);
        // Adding the child has to happen in the FX thread and not the event bus thread
        Platform.runLater(() -> {
			if (foreground.getChildren().contains(mainGameView)) {
				foreground.getChildren().remove(mainGameView);
			}
			foreground.getChildren().add(mainGameView);
		});
		miniMap.setDialogContoller(mainGameView);
    }

	private double calculateMainGameViewHeight(double height) {
		return height - BORDER_SPACING - MENU_HEIGHT + 1;
	}

	private double calculateMainGameViewWidth(double width) {
		return width - LEFT_PANEL_WIDTH - BORDER_SPACING;
	}
	@Override
	public void widthChange(double oldWidth, double newWidth) {
		background.widthChange(oldWidth, newWidth);
		viewStatus.setWidth(newWidth-(LEFT_PANEL_WIDTH+BORDER_SPACING));
		if (mainGameView != null) {
			mainGameView.widthChange(calculateMainGameViewWidth(oldWidth), calculateMainGameViewWidth(newWidth));
		}
	}
	

	@Override
	public void heightChange(double oldHeight, double newHeigth) {
		background.heightChange(oldHeight, newHeigth);
		if (mainGameView != null) {
			mainGameView.heightChange(calculateMainGameViewHeight(oldHeight), calculateMainGameViewHeight(newHeigth));
		}
	}

}
