package ch.sahits.game.openpatrician.display.scene;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import javafx.geometry.Rectangle2D;
import javafx.scene.Group;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.HBox;
import javafx.scene.layout.VBox;
import ch.sahits.game.graphic.image.IImageLoader;
import lombok.AccessLevel;
import lombok.Getter;

/**
 * Background component of the main game scene.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Nov 1, 2013
 *
 */
@ClassCategory(EClassCategory.JAVAFX)
public class MainGameSceneBackground extends Group {
	/** Dimensions of the mortar between plates. */
	static final int MORTAR_SQUARE_DIM = 12; // TODO aho Nov 1, 2013: might not need it this class
	/** Dimensions of the mortar for the corners. */
	static final int MORTAR_CORNER_DIM = 10; // TODO aho Nov 1, 2013: might not need it this class
	
	private IImageLoader loader;
	@Getter(AccessLevel.PACKAGE)
	private double sceneWidth;
	@Getter(AccessLevel.PACKAGE)
	private double sceneHeight;
	private double topLeftWidth;

	private HBox topFringRepeated;
	private VBox rightBorder;
	private HBox bottomBorder;
	private ImageView ornamentView;

	public MainGameSceneBackground(double width, double height, IImageLoader loader) {
		setManaged(false);
		this.loader = loader;
		sceneWidth = width;
		sceneHeight = height;

		Image topLeftImage = loader.getImage("images/FrameLayout");
		ImageView topLeft = new ImageView(topLeftImage);
		
		topFringRepeated = new HBox();
		setupTopFringe(width);
		
		rightBorder = new VBox();
		setupRightBorder(width, height);
		
		bottomBorder = new HBox();
		setupBottomBorder(width, height);
		
		// render deco under left side panel
		ornamentView = new ImageView();
		setupOrnament(height);
		
		
		
		
		getChildren().addAll(topLeft, topFringRepeated, rightBorder, bottomBorder, ornamentView);
		
	}
	
	/**
	 * Set up the ornament image.
	 * @param height total heigth of the scene
	 */
	private void setupOrnament(double height) {
		ornamentView.setLayoutY(MainGameScene.MINMIMAL_DISPLAY_HEIGHT);
		int decoHeight = (int) Math.floor(height-MainGameScene.MINMIMAL_DISPLAY_HEIGHT);
		String ornamentName;
		// add the two consecuting heigth and devide by two to get the separation limits
		if (decoHeight<125){
			ornamentName="images/Ornament310x94";
		} else if (decoHeight<158){
			ornamentName="images/Ornament310x155";
		} else if (decoHeight<173){
			ornamentName="images/Ornament310x162";
		} else if (decoHeight<248){
			ornamentName="images/Ornament310x185";
		} else if (decoHeight<387) {
			ornamentName = "images/Ornament310x310";
		} else if (decoHeight<492) {
			ornamentName = "images/Ornament310x464";
		} else if (decoHeight<569) {
			ornamentName = "images/Ornament310x519";
		} else if (decoHeight<633) {
			ornamentName = "images/Ornament310x619";
		} else if (decoHeight<839) {
			ornamentName = "images/Ornament310x646";
		} else {
			ornamentName = "images/Ornament310x1032";
		}

		Image ornament = loader.getImage(ornamentName, MainGameScene.LEFT_PANEL_WIDTH, decoHeight);
		ornamentView.setImage(ornament);
	}
	/**
	 * Set up the bottom border
	 * @param width total width of the scene
	 * @param height total height of the scene 
	 */
	private void setupBottomBorder(double width, double height) {
		bottomBorder.getChildren().clear();
		bottomBorder.setLayoutX(MainGameScene.LEFT_PANEL_WIDTH);
		bottomBorder.setLayoutY(height - MORTAR_CORNER_DIM);
		
		final int defaultPlateding = 200;
		double horizontalSpaceToFill = width-MainGameScene.LEFT_PANEL_WIDTH-1.5*MORTAR_CORNER_DIM; // half corner on left full at the right
		int onePlateHeigth = MORTAR_SQUARE_DIM+defaultPlateding;
		int countComplete = (int)Math.round(horizontalSpaceToFill/onePlateHeigth);
		// add one more and scale vertically so the align properly
		int bottomBorderMiddleOrigWidth = 374;
		int unscaledWidth = (countComplete+1)*(bottomBorderMiddleOrigWidth);
		double horizontalFillSpaceWithPlating = horizontalSpaceToFill-countComplete*MORTAR_SQUARE_DIM; // one mortar less than plate
		double scale = horizontalFillSpaceWithPlating/unscaledWidth;
		double bottomBorderMiddleDestWidth = bottomBorderMiddleOrigWidth*scale;
		
		Image bottomBoderMiddle = loader.getImage("images/bottomBorderMiddle", bottomBorderMiddleDestWidth, 10);
		Image leftbottomCornerMortar = loader.getImage("images/leftBottomBorderCorner");
		ImageView bottomRightMortarCornerView = new ImageView(leftbottomCornerMortar);
		Rectangle2D leftBottomMortarCornerViewport = new Rectangle2D(MORTAR_CORNER_DIM/2, 0, MORTAR_CORNER_DIM/2, MORTAR_CORNER_DIM);
		bottomRightMortarCornerView.setViewport(leftBottomMortarCornerViewport);
		bottomBorder.getChildren().add(bottomRightMortarCornerView);
		Image bottomMortar = loader.getImage("images/bottomBorderMortar");
		int coveredWidth = (int) Math.floor(MainGameScene.LEFT_PANEL_WIDTH+1.5*MORTAR_CORNER_DIM+(countComplete+1)*bottomBoderMiddle.getWidth()+countComplete*bottomMortar.getWidth());
		int diff = (int) Math.ceil(width-coveredWidth); // rounding error
		for (int w=0;w<=countComplete;w++){
			if (w < countComplete) {
				ImageView imgView = new ImageView(bottomBoderMiddle);
				bottomBorder.getChildren().add(imgView);
				ImageView bottomMortarImgView = new ImageView(bottomMortar);
				bottomBorder.getChildren().add(bottomMortarImgView);
			} else {
				Image bottomBoderMiddleWider = loader.getImage("images/bottomBorderMiddle", bottomBorderMiddleDestWidth + diff, 10);
				ImageView imgView = new ImageView(bottomBoderMiddleWider);
				bottomBorder.getChildren().add(imgView);
			}
		}
	}
	/**
	 * Setup the right border
	 * @param width total width of the scene
	 * @param height total height of the scene 
	 */
	private void setupRightBorder(double width, double height) {
		rightBorder.getChildren().clear();
		final int defaultPlateding = 200;
		double verticalSpaceToFill = height-MainGameScene.TOP_STATUS_HEIGHT-1.5*MORTAR_CORNER_DIM; // half corner on top full at the bottom
		int onePlateHeigth = MORTAR_SQUARE_DIM+defaultPlateding;
		int countComplete = (int)Math.round(verticalSpaceToFill/onePlateHeigth);
		// add one more and scale vertically so the align properly
		int reightBorderMiddleOrigHeight = 374;
		int unscaledHeigth = (countComplete+1)*(reightBorderMiddleOrigHeight);
		double horizontalFillSpaceWithPlaging = verticalSpaceToFill-countComplete*MORTAR_SQUARE_DIM; // one mortar less than plate
		double scale = horizontalFillSpaceWithPlaging/unscaledHeigth;
		double reightBorderMiddleDestHeight = reightBorderMiddleOrigHeight*scale;
		Image reightBoderMiddle = loader.getImage("images/reightBorderMiddle", 10, reightBorderMiddleDestHeight);
		// draw top right corner
		Image topRightMortar = loader.getImage("images/rightTopBorderCorner");
		Rectangle2D topRightMortarViewport = new Rectangle2D(0, MORTAR_CORNER_DIM/2, MORTAR_CORNER_DIM, MORTAR_CORNER_DIM/2); // bottom half of image
		ImageView topRightMortartImgView = new ImageView(topRightMortar);
		topRightMortartImgView.setViewport(topRightMortarViewport);
		rightBorder.getChildren().add(topRightMortartImgView);
		Image rightMortar = loader.getImage("images/reightBorderMortar");
		int coveredHeigth = (int) Math.floor(MainGameScene.TOP_STATUS_HEIGHT+1.5*MORTAR_CORNER_DIM+(countComplete+1)*reightBoderMiddle.getHeight()+countComplete*rightMortar.getHeight());
		int diff = (int) Math.ceil(height-coveredHeigth);
		for (int h=0;h<=countComplete;h++){
			if (h < countComplete) {
				ImageView imgView = new ImageView(reightBoderMiddle);
				rightBorder.getChildren().add(imgView);
				ImageView rightMortartImgView = new ImageView(rightMortar);
				rightBorder.getChildren().add(rightMortartImgView);
			} else {
				Image reightBoderMiddleHeiger = loader.getImage("images/reightBorderMiddle", 10, reightBorderMiddleDestHeight + diff);
				ImageView imgView = new ImageView(reightBoderMiddleHeiger);
				rightBorder.getChildren().add(imgView); // rounding error
			}
		}
		Image bottomRightMortar = loader.getImage("images/rightBottomBorderCorner");
		ImageView bottomRightMortartImgView = new ImageView(bottomRightMortar);
		rightBorder.getChildren().add(bottomRightMortartImgView);
		rightBorder.setLayoutY(MainGameScene.TOP_STATUS_HEIGHT);
		rightBorder.setLayoutX(width - MORTAR_CORNER_DIM);
	}
	/**
	 * Set up the top fringe.
	 * @param width total with of the scene
	 */
	private void setupTopFringe(double width) {
		double remWidth = width-topLeftWidth;
		topFringRepeated.getChildren().clear();
		Image topFringe = loader.getImage("images/topDeco");
		topLeftWidth = topFringe.getWidth();
		double decoWidth = topFringe.getWidth();
		int countComplete = (int) Math.floor(remWidth/decoWidth); // number the deco can be drawn complete
		double partialWidth = remWidth-(countComplete*decoWidth);
		
		for (int i = 0; i < countComplete; i++) {
			ImageView imgView = new ImageView(topFringe);
			topFringRepeated.getChildren().add(imgView);
		}
		Rectangle2D viewportTopRight = new Rectangle2D(0, 0, partialWidth, topFringe.getHeight());
		ImageView topRight = new ImageView(topFringe);
		topRight.setViewport(viewportTopRight);
		topFringRepeated.getChildren().add(topRight);
		topFringRepeated.layoutXProperty().set(topLeftWidth);
	}
	public void widthChange(double oldWidth, double newWidth) {
		setupTopFringe(newWidth);
		setupRightBorder(newWidth, this.sceneHeight);
		setupBottomBorder(newWidth, this.sceneHeight);
		this.sceneWidth = newWidth;
	}
	

	public void heightChange(double oldHeight, double newHeigth) {
		if (newHeigth > 600) {
			setupRightBorder(this.sceneWidth, newHeigth);
			setupBottomBorder(this.sceneWidth, newHeigth);
			setupOrnament(newHeigth);
		}
		this.sceneHeight = newHeigth;
	}

}
