package ch.sahits.game.openpatrician.display.dialog.armory;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.dialog.TabelViewDialog;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.openpatrician.javafx.model.ControlTableCell;
import ch.sahits.game.openpatrician.javafx.model.DynamicTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.StaticTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.Table;
import ch.sahits.game.openpatrician.javafx.model.TableHeader;
import ch.sahits.game.openpatrician.javafx.model.TableRow;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.building.IWeaponStorage;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.javafx.bindings.StaticIntegerBinding;
import ch.sahits.game.openpatrician.model.product.ComputablePriceV2;
import ch.sahits.game.openpatrician.model.product.ETransferAmount;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.service.TransferUtil;
import ch.sahits.game.openpatrician.model.weapon.ArmoryRegistry;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.model.weapon.IArmory;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.geometry.HPos;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 *         Created on Jul 23, 2017
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class HandWeaponDialog extends TabelViewDialog {
    private static final int IMAGE_DIM = 60;
    /** Reference to the city view model */
    private final ICityPlayerProxyJFX city;
    private IArmory armory;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    private IDataImageLoader imageLoader;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private TransferUtil transferUtil;
    @Autowired
    private ComputablePriceV2 computablePrice;
    @Autowired
    private ArmoryRegistry registry;
    @Autowired
    private ClientViewState viewState;

    public HandWeaponDialog(ICityPlayerProxyJFX city) {
        this.city = city;
    }

    @PostConstruct
    private void initializeDialog() {
        armory =  registry.getArmory(city.getCity());
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.armory.HandWeaponDialog.title",
                new Object[]{}, locale.getCurrentLocal()));
        Table model = createModel(city);
        setModel(model);
    }

    private Table createModel(ICityPlayerProxyJFX cityProxy) {
        Table model = new Table();

        TableHeader header = new TableHeader(6);
        header.add(new StaticTextTableCell(""));
        header.add(new StaticTextTableCell(""));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.armory.HandWeaponDialog.header.amrory", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.armory.HandWeaponDialog.header.buy", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.armory.HandWeaponDialog.header.prio", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.armory.HandWeaponDialog.header.tradingOffice", new Object[]{}, locale.getCurrentLocal())));
        header.setAligenment(0, HPos.RIGHT);
        header.setAligenment(1, HPos.RIGHT);
        header.setAligenment(2, HPos.RIGHT);
        header.setAligenment(3, HPos.CENTER);
        header.setAligenment(4, HPos.CENTER);
        header.setAligenment(5, HPos.CENTER);
        model.setHeader(header);
        model.setAligenment(0, HPos.LEFT);
        model.setAligenment(1, HPos.LEFT);
        model.setAligenment(2, HPos.RIGHT);
        model.setAligenment(3, HPos.CENTER);
        model.setAligenment(4, HPos.CENTER);
        model.setAligenment(5, HPos.CENTER);
        model.setColumnWidth(110, 60, 72, 72, 60, 70);
        ICity city = cityProxy.getCity();
        Optional<ITradingOffice> optOffice = cityProxy.getPlayer().findTradingOffice(city);
        if (optOffice.isPresent()) {
            ITradingOffice office = optOffice.get();
            IWeaponStorage weaponStorage = office.getWeaponStorage();
            final Image unchecked = imageLoader.getImage("images/waxseal_24x24");
            final Image checked = imageLoader.getImage("icons/waxseal_checked_icon");
            model.add(addWeaponRow(weaponStorage, unchecked, checked, EWeapon.SWORD));
            model.add(addWeaponRow(weaponStorage, unchecked, checked, EWeapon.BOW));
            model.add(addWeaponRow(weaponStorage, unchecked, checked, EWeapon.CROSSBOW));
            model.add(addWeaponRow(weaponStorage, unchecked, checked, EWeapon.MUSKET));
        } else {
            throw new IllegalStateException("This dialog should only be available if player has trading office in city.");
        }
        return model;
    }

    private TableRow addWeaponRow(IWeaponStorage weaponStorage, Image unchecked, Image checked, EWeapon weapon) {
        TableRow row = new TableRow();
        row.setHeigth(IMAGE_DIM);
        row.add(new StaticTextTableCell(translator.getLocalDisplayName(weapon)));
        Image img = imageLoader.getImage(getWeaponImageName(weapon), IMAGE_DIM, IMAGE_DIM);
        ImageView imgView = new ImageView(img);
        row.add(new ControlTableCell(imgView));
        DynamicTextTableCell weaponAmountInArmory = new DynamicTextTableCell();
        weaponAmountInArmory.valueProperty().bind(getAmountInArmoryAsStringBinding(weapon));
        row.add(weaponAmountInArmory);
        IntegerBinding weaponPrice = buyPriceBinding(weapon);
        OpenPatricianSmallWaxButton weaponBuyBtn = new OpenPatricianSmallWaxButton("");
        weaponBuyBtn.disableProperty().bind(getAmountInArmory(weapon).greaterThan(0).not());
        weaponBuyBtn.setId("buyWeapon"+weapon);
        weaponBuyBtn.textProperty().bind(weaponPrice.asString());
        row.add(new ControlTableCell(weaponBuyBtn));
        weaponBuyBtn.setOnAction(evt -> {
            if (!weaponBuyBtn.isDisable()) {
                IHumanPlayer player = city.getPlayer();
                ICompany company = player.getCompany();
                reduceStoredAmount(weapon);
                weaponStorage.update(weapon,  1);
                int price = weaponPrice.get();
                company.updateCash(-price);
            }
        });
        // Prio check box
        ImageView prioView = new ImageView();
        prioView.setId("priority"+weapon);
        setPriorityImage(unchecked, checked, weapon, prioView);
        prioView.setOnMouseReleased(evt -> {
            togglePrio(weapon);
            setPriorityImage(unchecked, checked, weapon, prioView);

        });
        row.add(new ControlTableCell(prioView));
        // Amount stored
        DynamicTextTableCell weaponsInStorage = new DynamicTextTableCell();
        weaponsInStorage.valueProperty().bind(weaponStorage.getWeaponProperty(weapon).asString());
        row.add(weaponsInStorage);
        return row;
    }

    private StringBinding getAmountInArmoryAsStringBinding(EWeapon weapon) {
        return new StringBinding() {
            {
                super.bind(getAmountInArmory(weapon));
            }
            @Override
            protected String computeValue() {
                ReadOnlyIntegerProperty amountInArmory = getAmountInArmory(weapon);
                return String.valueOf(amountInArmory.getValue());
            }
        };
    }

    private void reduceStoredAmount(EWeapon weapon) {
        switch (weapon) {
            case SWORD:
                armory.updateSwordAmount(-1);
                break;
            case BOW:
                armory.updateBowAmount(-1);
                break;
            case CROSSBOW:
                armory.updateCrossbowAmount(-1);
                break;
            case MUSKET:
                armory.updateMusketAmount(-1);
                break;
            default:
                throw new IllegalArgumentException("Unhandled weapon type: "+weapon);
        }    }

    private void togglePrio(EWeapon weapon) {
        switch (weapon) {
            case SWORD:
                armory.toggleSwordPriority();
                break;
            case BOW:
                armory.toggleBowPriority();
                break;
            case CROSSBOW:
                armory.toggleCrossbowPriority();
                break;
            case MUSKET:
                armory.toggleMusketPriority();
                break;
            default:
                throw new IllegalArgumentException("Unhandled weapon type: "+weapon);
        }
    }

    private void setPriorityImage(Image unchecked, Image checked, EWeapon weapon, ImageView prioView) {
        if (isPriority(weapon)) {
            prioView.setImage(checked);
        } else{
            prioView.setImage(unchecked);
        }
    }

    private String getWeaponImageName(EWeapon weapon) {
        switch (weapon) {
            case SWORD:
                return "images/Sword";
            case BOW:
                return "images/Bow";
            case CROSSBOW:
                return "images/Crossbow";
            case MUSKET:
                return "images/Musket";
            default:
                throw new IllegalArgumentException("Unhandled weapon type: "+weapon);
        }
    }

    private boolean isPriority(EWeapon weapon) {
        switch (weapon) {
            case SWORD:
                return armory.isSwordPriority();
            case BOW:
                return armory.isBowPriority();
            case CROSSBOW:
                return armory.isCrossbowPriority();
            case MUSKET:
                return armory.isMusketPriority();
            default:
                throw new IllegalArgumentException("Unhandled weapon type: "+weapon);
        }
    }

    private ReadOnlyIntegerProperty getAmountInArmory(EWeapon weapon) {
        switch (weapon) {
            case SWORD:
                return armory.swordAmountProperty();
            case BOW:
                return armory.bowAmountProperty();
            case CROSSBOW:
                return armory.crossbowAmountProperty();
            case MUSKET:
                return armory.musketAmountProperty();
            default:
                throw new IllegalArgumentException("Unhandled weapon type: "+weapon);
        }
    }


    private IntegerBinding buyPriceBinding(final EWeapon weapon) {
        return new IntegerBinding() {
            {
                super.bind(getAmountInArmory(weapon));
            }
            @Override
            protected int computeValue() {
                int availableAmount = getAmountInArmory(weapon).get();
                int buyAmount = transferUtil.calculateAvailableAmount(ETransferAmount.ONE, availableAmount);
                if (buyAmount>0) {
                    return computablePrice.buyPrice(weapon, new SimpleIntegerProperty(availableAmount), new StaticIntegerBinding(buyAmount)) * 2;
                } else {
                    return 0;
                }
            }
        };
    }
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.MAIN_VIEW_ARMORY);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }
}
