package ch.sahits.game.openpatrician.display.dialog.church;

import ch.sahits.game.graphic.image.IFontLoader;
import ch.sahits.game.graphic.image.impl.SelectiveCachableXMLImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.display.event.task.ClientTaskFactory;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.openpatrician.javafx.control.BaleAmountAlwaysVisible;
import ch.sahits.game.openpatrician.javafx.control.BarrelAmountAlwaysVisible;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.openpatrician.javafx.model.ControlTableCell;
import ch.sahits.game.openpatrician.javafx.model.ITableCell;
import ch.sahits.game.openpatrician.javafx.model.StaticTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.Table;
import ch.sahits.game.openpatrician.javafx.model.TableHeader;
import ch.sahits.game.openpatrician.javafx.model.TableRow;
import ch.sahits.game.openpatrician.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.javafx.service.JavaFXUtils;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.annotations.VisibleForTesting;
import javafx.beans.binding.Bindings;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.collections.FXCollections;
import javafx.collections.MapChangeListener;
import javafx.collections.ObservableMap;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.RowConstraints;
import javafx.scene.layout.VBox;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.Random;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Jul 10, 2015
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ChurchFeedingDialog extends CloseButtonDialog {
    private final Logger logger = LogManager.getLogger(getClass());
    /** Reference to the city view model */
    private final ICityPlayerProxyJFX city;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    @Qualifier("xmlImageLoader")
    private SelectiveCachableXMLImageLoader imageLoader;
    @Autowired
    private JavaFXUtils fxUtils;
    @Autowired
    private IFontLoader fontLoader;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private Random rnd;
    @Autowired
    private Date date;
    @Autowired
    private TimedUpdatableTaskList timedTaskListener;
    @Autowired
    private ClientTaskFactory taskFactory;

    private int numberOfColumns;
    private int mainTableYPosition;

    private final ITradingOffice office;

    private static final EWare[] MATERIALS = new EWare[]{EWare.GRAIN,
            EWare.MEAT,
            EWare.FISH,
            EWare.BEER,
            EWare.WINE};
    @MapType(key = IWare.class, value = IntegerProperty.class)
    private ObservableMap<IWare, IntegerProperty> availableMap = FXCollections.observableMap(new HashMap<>());

    public ChurchFeedingDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
        List<ITradingOffice> offices = city.getCity().findBuilding(ITradingOffice.class, Optional.of(city.getPlayer()));
        if (offices.isEmpty()) {
            office = null;
        } else {
            office = offices.get(0);
        }
        mainTableYPosition = 250 + 36;
        clearMap();
    }

    private void clearMap() {
        for (EWare ware : MATERIALS) {
            availableMap.put(ware, new SimpleIntegerProperty(0));
        }
    }

    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.church.ChurchFeedingDialog.title", new Object[]{}, locale.getCurrentLocal()));
        String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.church.ChurchFeedingDialog.introText", new Object[]{}, locale.getCurrentLocal());
        DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
        VBox box = new VBox(text);

        Table mainTable = createMainTable();
        GridPane mainTablePane = new GridPane();
        mainTablePane.setLayoutX(2 * FRAME_BORDER);
        mainTablePane.setLayoutY(mainTableYPosition);
        RowConstraints rowConstraint = getRowConstraints();
        mainTablePane.getRowConstraints().add(rowConstraint);
        // Header
        TableHeader header = mainTable.getHeader();
        numberOfColumns = mainTable.getNumberOfColumns();
        int currentCol = 0;
        for (int i = 0; i < numberOfColumns && header.size() > 0; i++) {
            int colWidth = mainTable.getColumnWidth(i);
            ColumnConstraints colConstraint = new ColumnConstraints(colWidth);
            colConstraint.setHalignment(header.getAligenment(i));
            mainTablePane.getColumnConstraints().add(colConstraint);
            ITableCell cell = header.get(i);
            final String textStyleClass = "tableHeader";
            fxUtils.addCellToGridPane(mainTablePane, -1, null, currentCol++, cell, header.getAligenment(i), textStyleClass);

        }
        // Table
        for (int rowNum = 0; rowNum < mainTable.getNumberOfRows(); rowNum++) {
            TableRow row = mainTable.get(rowNum);
            mainTablePane.getRowConstraints().add(rowConstraint);
            final int iterationLimit = Math.min(mainTable.getNumberOfColumns(), row.size());
            for (int col = 0; col < iterationLimit; col++) {

                ITableCell cell = row.get(col);
                final HPos hAlignment = mainTable.getAligenment(col);
                final String textStyleClass = "tableCell";
                fxUtils.addCellToGridPane(mainTablePane, rowNum, row, col, cell, hAlignment, textStyleClass);
            }
        }
        mainTablePane.getStyleClass().add("tableFont");

        box.setLayoutX(50);
        box.setLayoutY(CLOSE_BTN_Y_POS - 500);
        box.getChildren().addAll(mainTablePane);


        final int actionButtonX = (WIDTH - 124) / 2;

        String s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.church.ChurchFeedingDialog.donate", new Object[]{}, locale.getCurrentLocal());
        final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(s);
        action.getStyleClass().add("actionButton");
        action.setOnAction(getAction());
        action.setLayoutX(actionButtonX);
        action.setLayoutY(CLOSE_BTN_Y_POS - 24);
        BooleanBinding actionEnabled = actionEnabledBinding();
        action.setDisable(!actionEnabled.get());
        actionEnabled.addListener((observableValue, oldValue, newValue) -> action.setDisable(!newValue));


        getContent().addAll(box, action);

    }

    private BooleanBinding actionEnabledBinding() {
        return new BooleanBinding() {
            {
                super.bind(availableMap);
            }
            @Override
            protected boolean computeValue() {
                if (availableMap.isEmpty()) {
                    return false;
                }
                for (IntegerProperty amount : availableMap.values()) {
                    if (amount.get() > 0) {
                        return true;
                    }
                }
                return false;
            }
        };
    }

    private EventHandler<MouseEvent> getAction() {
        return evt -> {
            try {
                int amountBeggers = rnd.nextInt(city.getCity().getPopulationBinding().get() / 70);
                int amountWares = 0;
                for (IWare ware : availableMap.keySet()) {
                    amountWares += availableMap.get(ware).get() * ware.getSizeAsBarrels();
                }
                boolean sufficient = amountBeggers * 5 <= amountWares;
                LocalDateTime execution = date.getCurrentDate().plusDays(rnd.nextInt(30));
                Object[] args = new Object[]{amountBeggers};
                if (sufficient) {
                    DialogTemplate template = DialogTemplate.builder()
                            .closable(true)
                            .titleKey("ch.sahits.game.openpatrician.display.dialog.church.ChurchFeedingDialog.successfulFeedingTitle")
                            .messageKey("ch.sahits.game.openpatrician.display.dialog.church.ChurchFeedingDialog.successfulFeeding")
                            .messageArgs(args)
                            .build();
                    TimedTask task = taskFactory.getPostponedDisplayDialogMessage(execution, template);
                    timedTaskListener.add(task);
                    task = taskFactory.getReputationUpdateTaskChurchFeeding(city.getCity(), city.getPlayer(), amountBeggers, execution);
                    timedTaskListener.add(task);
                } else {
                    DialogTemplate template = DialogTemplate.builder()
                            .closable(true)
                            .titleKey("ch.sahits.game.openpatrician.display.dialog.church.ChurchFeedingDialog.unsuccessfulFeedingTitle")
                            .messageKey("ch.sahits.game.openpatrician.display.dialog.church.ChurchFeedingDialog.unsuccessfulFeeding")
                            .messageArgs(args)
                            .build();
                    TimedTask task = taskFactory.getPostponedDisplayDialogMessage(execution, template);
                    timedTaskListener.add(task);
                }
                availableMap.clear();
                city.getCity().getCityState().getPopUpdateStatistic().feedingThePoor();
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                logger.error("Failed to order feeding of the poor", e);
            }
        };
    }

    private Table createMainTable() {
        Table model = new Table();
        TableHeader header = new TableHeader(5);
        String s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.ware", new Object[]{}, locale.getCurrentLocal());

        header.add(new StaticTextTableCell(s));
        s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.CityStorageTradeDialog.storage", new Object[]{}, locale.getCurrentLocal());
        header.add(new StaticTextTableCell(s));
        header.add(new StaticTextTableCell(""));
        header.add(new StaticTextTableCell(""));
        s = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.church.ChurchFeedingDialog.donation", new Object[]{}, locale.getCurrentLocal());
        header.add(new StaticTextTableCell(s));
        header.setAligenment(0, HPos.LEFT);
        header.setAligenment(1, HPos.CENTER);
        header.setAligenment(4, HPos.LEFT);
        model.setHeader(header);
        model.setAligenment(0, HPos.LEFT);
        model.setAligenment(1, HPos.CENTER);
        model.setAligenment(3, HPos.RIGHT);
        model.setAligenment(4, HPos.CENTER);
        model.setColumnWidth(70, 70, 70, 70, 70);
        for (final EWare ware : MATERIALS) {
            final TableRow row = new TableRow();
            row.add(new StaticTextTableCell(translator.getLocalDisplayName(ware)));

            final ReadOnlyIntegerProperty stored = office != null ? office.getWare(ware).amountProperty() : new SimpleIntegerProperty(0);
            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = new BarrelAmountAlwaysVisible();
                barrelAmount.amountProperty().bind(stored.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = new BaleAmountAlwaysVisible();
                baleAmount.amountProperty().bind(stored.asString());
                row.add(new ControlTableCell(baleAmount));
            }



            BooleanBinding removeEnabled = new BooleanBinding() {
                {
                    super.bind(availableMap);
                }
                @Override
                protected boolean computeValue() {
                    return !(availableMap.get(ware) != null && availableMap.get(ware).get() > 0);
                }
            };
//                    Bindings.lessThan(1, donated).not();
            BooleanBinding donateEnabled = Bindings.lessThan(1, stored).not();
            final OpenPatricianSmallWaxButton previous = new OpenPatricianSmallWaxButton("<");
            previous.getStyleClass().add("actionButton");
            previous.setOnAction(createPreviousAction(ware));
            previous.disableProperty().bind(removeEnabled);
            row.add(new ControlTableCell(previous));
            final OpenPatricianSmallWaxButton next = new OpenPatricianSmallWaxButton(">");
            next.getStyleClass().add("actionButton");
            next.setOnAction(createNextAction(ware));
            next.disableProperty().bind(donateEnabled);
            row.add(new ControlTableCell(next));


            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = new BarrelAmountAlwaysVisible();
                barrelAmount.setAmount(0);
                availableMap.addListener((MapChangeListener<IWare, IntegerProperty>) change -> {
                    final IntegerProperty valueAdded = change.getValueAdded();
                    if (ware.equals(change.getKey()) && valueAdded != null) {
                        barrelAmount.amountProperty().setValue(String.valueOf(valueAdded.get()));
                    }
                });
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = new BaleAmountAlwaysVisible();
                baleAmount.setAmount(0);
                availableMap.addListener((MapChangeListener<IWare, IntegerProperty>) change -> {
                    final IntegerProperty valueAdded = change.getValueAdded();
                    if (ware.equals(change.getKey()) && valueAdded != null) {
                        baleAmount.amountProperty().setValue(String.valueOf(valueAdded.get()));
                    }
                });
                row.add(new ControlTableCell(baleAmount));
            }

            model.add(row);
        }
        return model;
    }

    private EventHandler<MouseEvent> createNextAction(EWare ware) {
        return evt -> {
            try {
                office.move(ware, -1);
                IntegerProperty donated = availableMap.get(ware);
                availableMap.put(ware, new SimpleIntegerProperty(donated.add(1).intValue()));
            } catch (RuntimeException e) {
                logger.error("Failed to move wares", e);
            }
        };
    }

    private EventHandler<MouseEvent> createPreviousAction(EWare ware) {
        return evt -> {
            try {
                office.move(ware, 1);
                IntegerProperty donated = availableMap.get(ware);
                availableMap.put(ware, new SimpleIntegerProperty(donated.subtract(1).intValue()));
            } catch (RuntimeException e) {
                logger.error("Failed to move wares", e);
            }
        };
    }

    private RowConstraints getRowConstraints() {
        RowConstraints rowConstraint;
        rowConstraint = new RowConstraints(24);
        return rowConstraint;
    }

    @Override
    public void executeOnCloseButtonClicked() {
        for (IWare ware : availableMap.keySet()) {
            IntegerProperty amount = availableMap.get(ware);
            office.move(ware, amount.intValue(), 0);
        }
        availableMap.clear();
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.MAIN_VIEW_CHURCH);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }
    @VisibleForTesting
    int getAmountDonated(IWare ware) {
        return availableMap.get(ware).get();
    }
}
