package ch.sahits.game.openpatrician.display.dialog.cityhall;

import ch.sahits.game.event.data.ClockTickDayChange;
import ch.sahits.game.graphic.image.impl.SelectiveCachableXMLImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ModelStateAccessor;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.openpatrician.event.data.ElectionWinnerNotification;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSmallToggableWaxButton;
import ch.sahits.game.openpatrician.javafx.control.PlaceHolder;
import ch.sahits.game.openpatrician.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DateService;
import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ECityWall;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.cityhall.EElectionType;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityHall;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.CityWallPetition;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.HeadTaxPetition;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.MilitiaPetition;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.SpecialTaxPetition;
import ch.sahits.game.openpatrician.model.personal.EEconomicCareer;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.application.Platform;
import javafx.event.EventHandler;
import javafx.geometry.Pos;
import javafx.scene.control.Label;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.HBox;
import javafx.scene.layout.StackPane;
import javafx.scene.layout.VBox;
import javafx.scene.text.Text;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Mar 23, 2015
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class TownInfoDialog extends CloseButtonDialog {
    private final Logger logger = LogManager.getLogger(getClass());

    @Autowired
    private ClientViewState viewState;
    @Autowired
    private ModelStateAccessor cityHallAccessor;
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private ModelTranslations modelTranslator;
    @Autowired
    private Date date;
    @Autowired
    private MapService cityService;
    @Autowired
    @Qualifier("xmlImageLoader")
    private SelectiveCachableXMLImageLoader imageLoader;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    @Autowired
    private DateService dateService;

    private final ICityPlayerProxyJFX city;

    private HBox selectionBox = null;
    private VBox box;
    private boolean displayAtElectionDay = false;


    public TownInfoDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
        getStylesheets().add(this.getClass().getResource("/styles/base.css").toExternalForm());
        getStyleClass().add("dialog");
    }

    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.title", new Object[]{}, locale.getCurrentLocal()));
        ICity town = city.getCity();
        IPlayer player = city.getPlayer();
        resetContent(town, player);

        final int actionButtonX = (WIDTH - 124) / 2;
        final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.petition", new Object[]{}, locale.getCurrentLocal()));
        action.getStyleClass().add("actionButton");
        action.setOnAction(getAction(box, action));
        action.setLayoutX(actionButtonX);
        action.setLayoutY(CLOSE_BTN_Y_POS - 24);
        action.setDisable(!canPosePetition(town, player)); // should be disabled when the petition cannot be posed

        getContent().addAll(action);
        clientServerEventBus.register(this);
        timerEventBus.register(this);
    }

    private void unregister() {
        clientEventBus.unregister(this);
        timerEventBus.unregister(this);
    }

    private void resetContent(ICity town, IPlayer player) {
        getContent().remove(box);
        box = textFactory.createMultiParagraphContainer();
        ICityHall cityHall = cityHallAccessor.getCityHall(town);
        ICitizen mayor = cityHall.getMayor();
        Object[] args = {town.getName(), modelTranslator.getLocalDisplayDistinctArticle(player.getPersonalData().isMale(), false), mayor.getName()+" "+mayor.getLastName()};

        String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.paragraph1", args, locale.getCurrentLocal());
        DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
        box.getChildren().add(text);
        final LocalDateTime electionDate = cityHall.getElectionDate();
        if (dateService.isSameDay(electionDate, date.getCurrentDate())) {
            template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.electionDay", new Object[]{}, locale.getCurrentLocal());
            displayAtElectionDay = true;
        } else {
            template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.paragraph2", new Object[]{modelTranslator.toDisplayString(electionDate)}, locale.getCurrentLocal());
            displayAtElectionDay = false;
        }
        text = textFactory.createDecoratedText(template, new HashMap<>());
        box.getChildren().add(text);
        List<ICitizen> candidates = cityHall.getCandidates();
        for (ICitizen candidate : candidates) {
            text = getCandidateRow(candidate);
            box.getChildren().add(text);

        }
        box.getChildren().addAll(new Label(" "), new Label(" "));

        HBox row = new HBox(10);
        row.setId("candidateship");
        row.setVisible(canBeCandidate(player, town));

        template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.acceptNomination", new Object[]{}, locale.getCurrentLocal());
        Text question = new Text(template);
        question.getStyleClass().add("dialog");
        OpenPatricianSmallToggableWaxButton yesButton = new OpenPatricianSmallToggableWaxButton(messageSource.getMessage("yes", new Object[]{}, locale.getCurrentLocal()));
        OpenPatricianSmallToggableWaxButton noButton = new OpenPatricianSmallToggableWaxButton(messageSource.getMessage("no", new Object[]{}, locale.getCurrentLocal()));
        yesButton.getStylesheets().add(this.getClass().getResource("/styles/base.css").toExternalForm());
        noButton.getStylesheets().add(this.getClass().getResource("/styles/base.css").toExternalForm());
        if (candidates.contains(player)) {
            yesButton.getStyleClass().add("active");
            noButton.getStyleClass().add("inactive");
        } else {
            yesButton.getStyleClass().add("inactive");
            noButton.getStyleClass().add("inactive");
        }
        yesButton.setOnAction((mouseEvent) -> {
            try {
                if (!candidates.contains(player)) {
                    yesButton.getStyleClass().remove("inactive");
                    yesButton.getStyleClass().add("active");
                    noButton.getStyleClass().add("inactive");
                    candidates.add(player);
                    DecoratedText dt = getCandidateRow(player);
                    box.getChildren().add(2, dt);
                    noButton.toggledProperty().setValue(false);
                }
            } catch (RuntimeException e) {
                logger.error("Failed to accept info", e);
            }
        });
        noButton.setOnAction((mouseEvent) -> {
            try {
                if (candidates.contains(player)) {
                    yesButton.getStyleClass().remove("active");
                    yesButton.getStyleClass().add("inactive");
                    noButton.getStyleClass().add("inactive");
                    candidates.remove(player);
                    box.getChildren().remove(2);
                    yesButton.toggledProperty().setValue(false);
                }
            } catch (RuntimeException e) {
                logger.error("Failed to say no to info", e);
            }
        });
        PlaceHolder ph = new PlaceHolder(30, 1);
        row.getChildren().addAll(question, ph, yesButton, noButton);

        box.getChildren().add(row);


        getContent().addAll(box);
    }

    private boolean canBeCandidate(IPlayer player, ICity town) {
        ESocialRank rank = player.getRank();
        if (rank == ESocialRank.COUNCILMAN || rank == ESocialRank.PATRICIAN || rank == ESocialRank.MAYOR || rank == ESocialRank.ALDERMAN) {
            final ICity hometown = player.getHometown();
            return town.equals(hometown);
        }
        return false;
    }

    private EventHandler<MouseEvent> getAction(VBox box, OpenPatricianLargeWaxButton action) {
        return (mouseEvent) -> {
            try {
                box.getChildren().clear();
                action.setVisible(false);
                String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.petitionIntro", new Object[]{}, locale.getCurrentLocal());
                DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());
                box.getChildren().add(text);

                VBox innerBox = new VBox();
                innerBox.setAlignment(Pos.TOP_CENTER);
                StackPane pane = new StackPane(innerBox);
                pane.prefHeightProperty().bind(text.wrappingWidthProperty());

                template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.petitionCityHall", new Object[]{}, locale.getCurrentLocal());
                Label label = textFactory.createLabel(template);
                label.setAlignment(Pos.CENTER);
                if (canEnlargeCityWalls()) {
                    template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.confirm", new Object[]{}, locale.getCurrentLocal());
                    action.setText(template);

                    label.setOnMouseReleased((evt) -> {
                        if (selectionBox != null) {
                            innerBox.getChildren().remove(selectionBox);
                            selectionBox = null;
                        }
                        action.setOnAction((event) -> {
                            try {
                                // Add petetion
                                CityWallPetition petition = new CityWallPetition();
                                cityHallAccessor.getCityHall(city.getCity()).setPetition(Optional.of(petition));
                                executeOnCloseButtonClicked();
                            } catch (RuntimeException e) {
                                logger.error("Failed to accept info message", e);
                            }

                        });
                        action.setVisible(true);
                    });
                } else {
                    label.getStyleClass().add("inactive");
                }

                innerBox.getChildren().add(label);

                template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.petitionGuard", new Object[]{}, locale.getCurrentLocal());
                label = textFactory.createLabel(template);
                label.setAlignment(Pos.CENTER);
                if (canHaveMoreGuards()) {
                    template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.confirm", new Object[]{}, locale.getCurrentLocal());
                    action.setText(template);

                    label.setOnMouseReleased((evt) -> {
                        if (selectionBox != null) {
                            innerBox.getChildren().remove(selectionBox);
                            selectionBox = null;
                        }
                        action.setOnAction((event) -> {
                            try {
                                // Add petetion
                                MilitiaPetition petition = new MilitiaPetition();
                                cityHallAccessor.getCityHall(city.getCity()).setPetition(Optional.of(petition));
                                executeOnCloseButtonClicked();
                            } catch (RuntimeException e) {
                                logger.error("Failed to dismiss info message", e);
                            }

                        });
                        action.setVisible(true);
                    });
                } else {
                    label.getStyleClass().add("inactive");
                }

                innerBox.getChildren().add(label);


                template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.petitionHeadTax", new Object[]{}, locale.getCurrentLocal());
                label = textFactory.createLabel(template);
                label.setAlignment(Pos.CENTER);
                label.setOnMouseReleased((evt) -> {
                    if (selectionBox != null) {
                        innerBox.getChildren().remove(selectionBox);
                        selectionBox = null;
                    }
                    action.setVisible(true);
                    Label lbl = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.headTax", new Object[]{}, locale.getCurrentLocal()));
                    Image img = imageLoader.getImage("icons/minusIcon");
                    ImageView imgView = new ImageView(img);
                    imgView.setId("minusIcon");
                    final Label input = new Label();
                    input.setId("headTaxLabel");
                    input.setText(String.valueOf(cityHallAccessor.getCityHall(city.getCity()).getTreasury().getCurrentHeadTaxValue()));
                    imgView.setOnMouseReleased((e) -> {
                        double value = Double.parseDouble(input.getText());
                        input.setText(String.valueOf(Math.max(value - 0.5, 0)));
                    });
                    selectionBox = new HBox(5);
                    selectionBox.setAlignment(Pos.TOP_CENTER);
                    selectionBox.getChildren().addAll(lbl, imgView, input);
                    lbl = new Label("%");
                    img = imageLoader.getImage("icons/plusIcon");
                    imgView = new ImageView(img);
                    imgView.setId("plusIcon");
                    imgView.setOnMouseReleased((e) -> {
                        double value = Double.parseDouble(input.getText());
                        input.setText(String.valueOf(value + 0.5));
                    });
                    selectionBox.getChildren().addAll(lbl, imgView);
                    innerBox.getChildren().add(selectionBox);
                    action.setOnAction((e) -> {
                        HeadTaxPetition petition = new HeadTaxPetition(Double.parseDouble(input.getText()));
                        cityHallAccessor.getCityHall(city.getCity()).setPetition(Optional.of(petition));
                        executeOnCloseButtonClicked();
                    });

                });
                innerBox.getChildren().add(label);

                template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.petitionSpecialTax", new Object[]{}, locale.getCurrentLocal());
                label = textFactory.createLabel(template);
                label.setAlignment(Pos.CENTER);
                label.setOnMouseReleased((evt) -> {
                    if (selectionBox != null) {
                        innerBox.getChildren().remove(selectionBox);
                        selectionBox = null;
                    }
                    action.setVisible(true);
                    Label lbl = new Label(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.specialTax", new Object[]{}, locale.getCurrentLocal()));
                    Image img = imageLoader.getImage("icons/minusIcon");
                    ImageView imgView = new ImageView(img);
                    imgView.setId("minusIcon");
                    final Label input = new Label();
                    input.setId("specialTaxLabel");
                    input.setText("0");
                    imgView.setOnMouseReleased((e) -> {
                        int value = Integer.parseInt(input.getText());
                        input.setText(String.valueOf(Math.max(value - 5000, 0)));
                    });
                    selectionBox = new HBox(5);
                    selectionBox.setAlignment(Pos.TOP_CENTER);
                    selectionBox.getChildren().addAll(lbl, imgView, input);
                    img = imageLoader.getImage("icons/coin_icon", 32, 32);
                    imgView = new ImageView(img);
                    selectionBox.getChildren().add(imgView);
                    img = imageLoader.getImage("icons/plusIcon");
                    imgView = new ImageView(img);
                    imgView.setId("plusIcon");
                    imgView.setOnMouseReleased((e) -> {
                        int value = Integer.parseInt(input.getText());
                        input.setText(String.valueOf(value + 5000));
                    });
                    selectionBox.getChildren().addAll(imgView);
                    innerBox.getChildren().add(selectionBox);
                    action.setOnAction((e) -> {
                        SpecialTaxPetition petition = new SpecialTaxPetition(Integer.parseInt(input.getText()));
                        cityHallAccessor.getCityHall(city.getCity()).setPetition(Optional.of(petition));
                        executeOnCloseButtonClicked();
                    });

                });
                innerBox.getChildren().add(label);
                box.getChildren().add(pane);
            } catch (RuntimeException e) {
                logger.error("Failed to switch outrigger", e);
            }
        };

    }

    private boolean canEnlargeCityWalls() {
        return city.getCity().getCityState().getCityWall().getExtension() != ECityWall.EXTENDED_TWICE;
    }
    private boolean canHaveMoreGuards() {
        int maxPossible = cityService.getMaxNumberOfGuards(city.getCity().getPopulationBinding().get());
        return cityHallAccessor.getCityHall(city.getCity()).getCityGuard().size() < maxPossible;
    }

    /**
     * Petition can be requested in the hometown if the player is counsilman, patrician or mayor, or in any city if the player is alerman.
     * @param town for which the petition capability is checked
     * @param player for whom to check the petition capability
     * @return true if the player can pose a petition
     */
    private boolean canPosePetition(ICity town, IPlayer player) {
        ESocialRank rank = player.getRank();
        if (cityHallAccessor.getCityHall(town).getPetition().isPresent()) {
            return false; // Only allow one petition at a time
        }
        if (rank == ESocialRank.COUNCILMAN || rank == ESocialRank.PATRICIAN || rank == ESocialRank.MAYOR) {
            final ICity hometown = player.getHometown();
            if (town.equals(hometown)) {
                return true;
            }
        }
        return rank == ESocialRank.ALDERMAN;
    }

    private DecoratedText getCandidateRow(ICitizen candidate) {
        Object[] args;
        String template;
        DecoratedText text;
        if (candidate instanceof IHumanPlayer) {
            IPlayer p = (IPlayer) candidate;
            args = new Object[]{modelTranslator.getLocalDisplayDistinctArticle(p.getPersonalData().isMale(), true), modelTranslator.getLocalDisplayName(p.getCareerLevel()), modelTranslator.getLocalDisplayName(p.getRank()), p.getName()+" "+p.getLastName()};
        } else {
            args = new Object[]{modelTranslator.getLocalDisplayDistinctArticle(true, true), modelTranslator.getLocalDisplayName(EEconomicCareer.SMART), modelTranslator.getLocalDisplayName(candidate.getRank()), candidate.getName()+" "+candidate.getLastName()};
        }
        template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.cityhall.TownInfoNotice.candidate", args, locale.getCurrentLocal());
        text = textFactory.createDecoratedText(template, new HashMap<>());
        return text;
    }

    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_CITY_HALL_TREASURY);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
        clientServerEventBus.unregister(this);
    }
    @Subscribe
    public void checkDayChangeUpdates(ClockTickDayChange dayChange) {
        ICity town = city.getCity();
        IPlayer player = city.getPlayer();
        ICityHall cityHall = cityHallAccessor.getCityHall(town);
        final LocalDateTime electionDate = cityHall.getElectionDate();
        if (dateService.isSameDay(electionDate, date.getCurrentDate())) {
            Platform.runLater(() -> resetContent(town, player));
        }
    }
    @Subscribe
    public void checkPostDayChangeUpdates(ElectionWinnerNotification electionResult) {
        if (displayAtElectionDay && electionResult.getElectionType().equals(EElectionType.MAYORAL) && electionResult.getCity().equals(city.getCity())) {
            ICity town = city.getCity();
            IPlayer player = city.getPlayer();
            ICityHall cityHall = cityHallAccessor.getCityHall(town);
            final LocalDateTime electionDate = cityHall.getElectionDate();
            if (!dateService.isSameDay(electionDate, date.getCurrentDate())) {
                Platform.runLater(() -> resetContent(town, player));
            }
        }
    }

}
