package ch.sahits.game.openpatrician.display.dialog.shipyard;

import ch.sahits.game.graphic.image.IFontLoader;
import ch.sahits.game.graphic.image.impl.SelectiveCachableXMLImageLoader;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.event.NoticeBoardClose;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.openpatrician.javafx.control.PaginationV2;
import ch.sahits.game.openpatrician.javafx.control.PaginationV2Builder;
import ch.sahits.game.openpatrician.javafx.control.PlaceHolder;
import ch.sahits.game.openpatrician.javafx.model.ECellConstraint;
import ch.sahits.game.openpatrician.javafx.model.ITableCell;
import ch.sahits.game.openpatrician.javafx.model.Table;
import ch.sahits.game.openpatrician.javafx.model.TableHeader;
import ch.sahits.game.openpatrician.javafx.model.TableRow;
import ch.sahits.game.openpatrician.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.javafx.service.JavaFXUtils;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.product.ComputablePriceV2;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import ch.sahits.game.openpatrician.utilities.model.Text;
import ch.sahits.game.openpatrician.utilities.service.TextParser;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.ObjectBinding;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.geometry.Insets;
import javafx.geometry.VPos;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.control.Label;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.RowConstraints;
import javafx.scene.layout.StackPane;
import javafx.scene.text.Font;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 14, 2013
 */
public abstract class BaseShipyardDialog extends CloseButtonDialog {
    private final Logger logger = LogManager.getLogger(getClass());
    /** Reference to the city view model */
    protected final ICityPlayerProxyJFX city;
    private IntegerProperty currentShipTypeIndex;
    protected final EShipType[] shipTypes;
    @Autowired
    protected MessageSource messageSource;
    @Autowired
    protected Locale locale;
    @Autowired
    @Qualifier("resourceReference")
    private MessageSource resources;
    @Autowired
    private TextParser textParser;
    @Autowired
    @Qualifier("xmlImageLoader")
    private SelectiveCachableXMLImageLoader imageLoader;
    @Autowired
    private JavaFXUtils fxUtils;
    @Autowired
    private IFontLoader fontLoader;
    @Autowired
    private ModelTranslations modelTranslations;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    protected ComputablePriceV2 computablePrice;

    private ECellConstraint cellConstraint;
    private int numberOfColumns;
    protected BooleanBinding enablePreviousNext;
    protected int mainTableYPosition;
    private Label titleLbl;

    public BaseShipyardDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
        getStylesheets().add(this.getClass().getResource("/styles/base.css").toExternalForm());
        getStyleClass().add("dialog");
        currentShipTypeIndex = new SimpleIntegerProperty(this,"currentShipTypeIndex", 0);
        IShipyard shipyard = city.getCity().getCityState().getShipyardState();
        shipTypes = shipyard.getBuildableShipTypes();
        enablePreviousNext = new BooleanBinding() {
            @Override
            protected boolean computeValue() {
                return true;
            }
        };
        mainTableYPosition = 250 + 36;
    }
    @PostConstruct
    private void initializeComponents() {
        if (hasShips()) {
            initializeRequirements();
            titleLbl = new Label();
            titleLbl.setId("title");
            titleLbl.getStyleClass().add("dialogText");
            String shipType = getTitleText();
            titleLbl.setText(shipType);
            currentShipTypeIndex.addListener((observable, oldValue1, newValue1) -> selectionChanged());
            StackPane titlePane = new StackPane();
            titlePane.setLayoutY(60);
            titlePane.getChildren().addAll(new PlaceHolder(WIDTH, 1), titleLbl);
            getContent().add(titlePane);

            ImageView sideView = new ImageView();
            sideView.imageProperty().bind(shipSideImageBinding());
            sideView.setLayoutY(70);
            sideView.setLayoutX(30);
            final EventHandler<? super MouseEvent> historyDisplayHandler = createHistoryDisplayHandler();
            sideView.setOnMouseReleased(historyDisplayHandler);
            ImageView frontView = new ImageView();
            frontView.imageProperty().bind(shipFrontImageBinding());
            frontView.setLayoutY(70);
            frontView.setLayoutX(289);
            frontView.setOnMouseReleased(historyDisplayHandler);
            getContent().addAll(sideView, frontView);
            Table topTable = createTopTable();
            GridPane topTablePane = new GridPane();
            topTablePane.setLayoutX(50);
            topTablePane.setLayoutY(250);
            RowConstraints rowConstraint = new RowConstraints(48);
            rowConstraint.setValignment(VPos.TOP);
            topTablePane.getRowConstraints().add(rowConstraint);
            for (TableRow row : topTable) {
                for (int i = 0; i < topTable.getNumberOfColumns(); i++) {
                    int colWidth = topTable.getColumnWidth(i);
                    ColumnConstraints colConstraint = new ColumnConstraints(colWidth);
                    colConstraint.setHalignment(topTable.getAligenment(i));
                    topTablePane.getColumnConstraints().add(colConstraint);
                    ITableCell cell = row.get(i);
                    final String textStyleClass = "tableCell";
                    fxUtils.addCellToGridPane(topTablePane, -1, null, i, cell, topTable.getAligenment(i), textStyleClass);
                }
            }
            topTablePane.getStyleClass().add("tableFont");
            Table mainTable = createMainTable();
            GridPane mainTablePane = new GridPane();
            mainTablePane.setLayoutX(2 * FRAME_BORDER);
            mainTablePane.setLayoutY(mainTableYPosition);
            rowConstraint = getRowConstraints();
            mainTablePane.getRowConstraints().add(rowConstraint);
            // Header
            TableHeader header = mainTable.getHeader();
            numberOfColumns = mainTable.getNumberOfColumns();
            int currentCol = 0;
            for (int i = 0; i < numberOfColumns && header.size() > 0; i++) {
                int colWidth = mainTable.getColumnWidth(i);
                ColumnConstraints colConstraint = new ColumnConstraints(colWidth);
                colConstraint.setHalignment(header.getAligenment(i));
                mainTablePane.getColumnConstraints().add(colConstraint);
                ITableCell cell = header.get(i);
                final String textStyleClass = "tableHeader";
                fxUtils.addCellToGridPane(mainTablePane, -1, null, currentCol++, cell, header.getAligenment(i), textStyleClass);
                // jump over spanned columns
                cellConstraint = header.getCellConstraint(cell);
                if (cellConstraint != null) {
                    switch (cellConstraint) {
                        case COLSPAN2:
                            numberOfColumns--;
                            currentCol++;
                            break;
                        case COLSPAN4:
                            numberOfColumns -= 3;
                            currentCol += 3;
                            break;
                        default:
                            throw new RuntimeException("Unhandled case");
                    }
                }
            }
            // Table
            for (int rowNum = 0; rowNum < mainTable.getNumberOfRows(); rowNum++) {
                TableRow row = mainTable.get(rowNum);
                mainTablePane.getRowConstraints().add(rowConstraint);
                final int iterationLimit = Math.min(mainTable.getNumberOfColumns(), row.size());
                for (int col = 0; col < iterationLimit; col++) {

                    ITableCell cell = row.get(col);
                    final HPos hAlignment = mainTable.getAligenment(col);
                    final String textStyleClass = "tableCell";
                    fxUtils.addCellToGridPane(mainTablePane, rowNum, row, col, cell, hAlignment, textStyleClass);
                }
            }
            mainTablePane.getStyleClass().add("tableFont");
            getContent().addAll(topTablePane, mainTablePane);

            Group footerText = createFooterText();
            footerText.setLayoutX(50);
            footerText.setLayoutY(250 + 36 + 7 * 24);

            final int actionButtonX = (WIDTH - 124) / 2;

            final OpenPatricianSmallWaxButton previous = new OpenPatricianSmallWaxButton("<");
            previous.setId("prev");
            previous.getStyleClass().add("actionButton");
            previous.setLayoutX(actionButtonX - 62 - 4);
            previous.setLayoutY(CLOSE_BTN_Y_POS - 48);
            previous.setOnAction(createPreviousAction());
            BooleanProperty disableNavigation = disableNavigation();
            previous.disableProperty().bind(disableNavigation);
            final OpenPatricianSmallWaxButton next = new OpenPatricianSmallWaxButton(">");
            next.setId("next");
            next.getStyleClass().add("actionButton");
            next.setLayoutX(actionButtonX + 124 + 4);
            next.setLayoutY(CLOSE_BTN_Y_POS - 48);
            next.setOnAction(createNextAction());
            next.disableProperty().bind(disableNavigation);

            final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(getActionText());
            action.setId("action");
            action.getStyleClass().add("actionButton");
            action.setOnAction(getAction());
            action.setLayoutX(actionButtonX);
            action.setLayoutY(CLOSE_BTN_Y_POS - 24);
            BooleanBinding actionEnabled = actionEnabledBinding();
            action.setDisable(!actionEnabled.get());
            actionEnabled.addListener((observableValue, oldValue, newValue) -> action.setDisable(!newValue));


            getContent().addAll(footerText, previous, next, action);
        }  else {
            // no ships
            final String template = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseShipyardDialog.noShip", new Object[]{}, locale.getCurrentLocal());
            DecoratedText text = textFactory.createDecoratedText(template, new HashMap<>());

            text.getStyleClass().add("dialogText");
            text.setLayoutX(50);
            text.setLayoutY(250);
            getContent().addAll(text);
        }
    }

    /**
     * Handle for changed selection.
     */
    protected void selectionChanged() {
        String st = getTitleText();
        titleLbl.setText(st);
    }

    /**
     * Representing the title text. The default implementation retrieves the ship type.
     * @return localized dialog title text
     */
    protected String getTitleText() {
        return modelTranslations.getLocalDisplayName(getCurrentShipType());
    }

    /**
     * Initialize the requirements.
      */
    protected abstract void initializeRequirements();

    private EventHandler<? super MouseEvent> createHistoryDisplayHandler() {
        return mouseEvent -> {
            try {
                final Text historyText = getHistoryText(getCurrentShipType());
                Insets insets = new Insets(FRAME_BORDER, FRAME_BORDER, FRAME_BORDER, FRAME_BORDER * 2);
                Font headerFont = fontLoader.createDefaultFont(18 + 6); // bold font not available
                Font defaultFont = fontLoader.createDefaultFont(18);
                final List<Node> firstPageContent = new ArrayList<>(getContent().size());
                firstPageContent.addAll(getContent());
                PaginationV2 pagination = PaginationV2Builder.create()
                        .backButtonLabel(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.CreditsScene.back", new Object[]{}, locale.getCurrentLocal()))
                        .nextButtonLabel(messageSource.getMessage("ch.sahits.game.openpatrician.display.scene.CreditsScene.next", new Object[]{}, locale.getCurrentLocal()))
                        .text(historyText)
                        .contentMaxWidth(WRAPPING_WIDTH)
                        .contentMaxHeight(CLOSE_BTN_Y_POS - 4 * FRAME_BORDER)
                        .padding(insets)
                        .navigationLabelFont(defaultFont)
                        .headerFont(headerFont)
                        .paragraphFont(defaultFont)
                        .centeredFont(defaultFont)
                        .firstLastPageAction(new ReplaceFirstPage(firstPageContent))
                        .build();
                pagination.setLayoutY(FRAME_BORDER * 2);
                replaceContent(pagination);
            } catch (RuntimeException e) {
                logger.error("Failed to display history page", e);
            }
        };
    }
    /**
     * Retrieve the history text.
     * @return localized ship type history text
     */
    private Text getHistoryText(EShipType type) {
        String resourceName = resources.getMessage(getHistoryResourceName(type), new Object[0], locale.getCurrentLocal());
        URL url = getClass().getClassLoader().getResource(resourceName);
        try {
            InputStream is= url.openStream();
            return textParser.parse(is);
        } catch (IOException e) {
            logger.warn("History text could not be found: " + url.toExternalForm());
            return new Text();
        }
    }

    private String getHistoryResourceName(EShipType type) {
        switch (type) {
            case SNAIKKA:
                return "history.snaikka";
            case CRAYER:
                return "history.crayer";
            case COG:
                return "history.cog";
            case HOLK:
                return "history.holk";
            default:
                throw new IllegalStateException("The ship type "+type+" is not handled.");
        }
    }

    protected RowConstraints getRowConstraints() {
        RowConstraints rowConstraint;
        rowConstraint = new RowConstraints(24);
        return rowConstraint;
    }

    /**
     * Move the selection to the next index
     * @return .
     */
    protected EventHandler<MouseEvent> createNextAction() {
        return mouseEvent -> {
            try {
                if (currentShipTypeIndex.get() == shipTypes.length - 1) {
                    currentShipTypeIndex.set(0);
                } else {
                    currentShipTypeIndex.set(currentShipTypeIndex.get() + 1);
                }
            } catch (RuntimeException e) {
                logger.error("Failed to switch to next ship type", e);
            }
        };
    }

    /**
     * If there are no more than one navigable item, the navigation is disabled.
     * @return boolean property indicating the disabled navigation
     */
    protected abstract BooleanProperty disableNavigation();

    /**
     * move the selection to the previous index.
     * @return  .
     */
    protected EventHandler<MouseEvent> createPreviousAction() {
        return mouseEvent -> {
            try {
                if (currentShipTypeIndex.get() == 0) {
                    currentShipTypeIndex.set(shipTypes.length - 1);
                } else {
                    currentShipTypeIndex.set(currentShipTypeIndex.get() - 1);
                }
            } catch (RuntimeException e) {
                logger.error("Failed to switch to previous ship type", e);
            }
        };
    }



    /**
     * Define the action that is executed on the action button.
     * @return action that is executed on the button
     */
    protected abstract EventHandler<MouseEvent> getAction();

    /**
     * Label of the action button
     * @return label on the action button
     */
    protected abstract String getActionText();

    /**
     * Create the group for the footer text;
     * @return bottom footer text group
     */
    protected abstract Group createFooterText();

    /**
     * Create a table representing the main information
     * @return Table model for the main part
     */
    protected abstract Table createMainTable();


    /**
     * Create a table representing the header information
     * @return Table model used for the top line
     */
    protected abstract Table createTopTable();

    /**
     * Boolean binding defining if the action button is enabled.
     * @return  .
     */
    protected abstract BooleanBinding actionEnabledBinding();

    /**
     * Binding for the appropriate side image of the ship.
     * @return .
     */
    private ObjectBinding<Image> shipSideImageBinding() {
        return new ObjectBinding<>() {
            {
                super.bind(currentShipTypeIndex);
            }

            @Override
            protected Image computeValue() {
                switch (getCurrentShipType()) {
                    case SNAIKKA:
                        return imageLoader.getImage("images/schnikka_side");
                    case CRAYER:
                        return imageLoader.getImage("images/crayer_side");
                    case COG:
                        return imageLoader.getImage("images/cog_side");
                    case HOLK:
                        return imageLoader.getImage("images/holk_side");
                    default:
                        throw new RuntimeException("Unknown ships type: " + getCurrentShipType());
                }
            }
        };
    }
    /**
     * Binding for the appropriate front image of the ship.
     * @return  .
     */
    private ObjectBinding<Image> shipFrontImageBinding() {
        return new ObjectBinding<>() {
            {
                super.bind(currentShipTypeIndex);
            }

            @Override
            protected Image computeValue() {
                switch (getCurrentShipType()) {
                    case SNAIKKA:
                        return imageLoader.getImage("images/schnikka_front");
                    case CRAYER:
                        return imageLoader.getImage("images/crayer_front");
                    case COG:
                        return imageLoader.getImage("images/cog_front");
                    case HOLK:
                        return imageLoader.getImage("images/holk_front");
                    default:
                        throw new RuntimeException("Unknown ships type: " + getCurrentShipType());
                }
            }
        };
    }

    /**
     * Retrieve the currently selected ship type.
     * @return  current ship type
     */
    protected final EShipType getCurrentShipType() {
        return shipTypes[currentShipTypeIndex.get()];
    }


    public IntegerProperty currentShipTypeIndexProperty() {
        return currentShipTypeIndex;
    }

    /**
     * Indication if there are ships and the dialog can be rendered.
     * Default value is true. Subclasses should overwrite this method.
      * @return true if there are ships in port or this fact is not relevant.
     */
    protected boolean hasShips() {
        return true;
    }

    @Override
    public void executeOnCloseButtonClicked() {
        clientEventBus.post(new NoticeBoardClose());
        super.executeOnCloseButtonClicked();
    }
    public ICity getCity() {
        return city.getCity();
    }
    public IHumanPlayer getPlayer() {
        return city.getPlayer();
    }


    /**
     * Binding of the amount that has to be bought as it is not stored in a warehouse.
     */
    protected class BuyAmountBinding extends IntegerBinding {
        private final ReadOnlyIntegerProperty stored;
        private final IntegerBinding needed;
        public BuyAmountBinding(IWare ware, IntegerBinding requiredAmount) {
            final Optional<ITradingOffice> office = getPlayer().findTradingOffice(getCity());
            if (office.isPresent()) {
                stored = office.get().getWare(ware).amountProperty();
                needed = requiredAmount;
                super.bind(stored, requiredAmount);
            } else {
                stored = new SimpleIntegerProperty(0);
                needed = requiredAmount;
                super.bind(requiredAmount);
            }
        }

        @Override
        protected int computeValue() {
            return Math.max(0, needed.get() - stored.get());
        }
    }

    private class ReplaceFirstPage implements EventHandler<MouseEvent> {
        private final List<Node> firstPageContent;
        public ReplaceFirstPage(List<Node> firstPageContent) {
            this.firstPageContent = firstPageContent;
        }

        @Override
        public void handle(MouseEvent mouseEvent) {
            replaceContent(firstPageContent);
        }
    }

}
