package ch.sahits.game.openpatrician.display.dialog.shipyard;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.model.WeaponSlotCount;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.ShipFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.display.model.RequiredWareCityStorage;
import ch.sahits.game.openpatrician.event.data.ShipyardOrderBuild;
import ch.sahits.game.openpatrician.javafx.control.BaleAmountAlwaysVisible;
import ch.sahits.game.openpatrician.javafx.control.BarrelAmount;
import ch.sahits.game.openpatrician.javafx.control.BarrelAmountAlwaysVisible;
import ch.sahits.game.openpatrician.javafx.control.CoinPriceAlwaysVisible;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.model.ControlTableCell;
import ch.sahits.game.openpatrician.javafx.model.ECellConstraint;
import ch.sahits.game.openpatrician.javafx.model.StaticTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.Table;
import ch.sahits.game.openpatrician.javafx.model.TableHeader;
import ch.sahits.game.openpatrician.javafx.model.TableRow;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.IWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.impl.ShipWeaponsLocationFactory;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 13, 2013
 */
@Component("shipyardConstructionDialog")
@Scope(value = "prototype")
@ClassCategory({EClassCategory.DIALOG, EClassCategory.SINGLETON_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ShipyardConstructionDialogV2 extends BaseShipyardDialog {
    private final Logger logger = LogManager.getLogger(getClass());

    @Autowired
    private ShipFactory shipFactory;
    @Autowired
    private ShipWeaponsLocationFactory weaponsLocationFactory;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private Locale locale;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private ShipService shipService;

    private static final EWare[] MATERIALS = new EWare[]{EWare.WOOD,
                                                         EWare.CLOTH,
                                                         EWare.IRON,
                                                         EWare.HEMP,
                                                         EWare.PITCH};

    @MapType(key = IWare.class, value = RequiredWareCityStorage.class)
    private HashMap<IWare, RequiredWareCityStorage> requirementsMap = new HashMap<>();

    public ShipyardConstructionDialogV2(ICityPlayerProxyJFX city) {
        super(city);
    }
    @PostConstruct
    private void init() {
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.title", new Object[]{}, locale.getCurrentLocal()));
    }

    /**
     * Initialize the requirementsMap.
     */
    @Override
    protected void initializeRequirements() {
        ICity city = getCity();
        Optional<ITradingOffice> optOffice = getPlayer().findTradingOffice(city);
        for (EWare material : MATERIALS) {
            int requiredAmount = shipFactory.getConstructionAmount(getCurrentShipType(), material);
            ReadOnlyIntegerProperty amountInCity = city.getWare(material).amountProperty();
            ReadOnlyIntegerProperty amountInStorage;
            if (optOffice.isPresent()) {
                ITradingOffice office = optOffice.get();
                amountInStorage = office.getWare(material).amountProperty();
            } else {
                amountInStorage = new SimpleIntegerProperty(0);
            }
            if (requirementsMap.containsKey(material)) {
                // update
                RequiredWareCityStorage requiredWare = requirementsMap.get(material);
                requiredWare.setRequired(requiredAmount);
                requiredWare.inStorageProperty().unbind();
                requiredWare.inStorageProperty().bind(amountInStorage);
                requiredWare.inCityProperty().unbind();
                requiredWare.inCityProperty().bind(amountInCity);
            } else {
                // initialize
                IntegerProperty inCity = new SimpleIntegerProperty();
                inCity.bind(amountInCity);
                IntegerProperty inStorage = new SimpleIntegerProperty();
                inStorage.bind(amountInStorage);
                RequiredWareCityStorage requiredWare = new RequiredWareCityStorage(requiredAmount, inStorage, inCity,
                        material, computablePrice);
                requirementsMap.put(material, requiredWare);
            }
        }
    }

    @Override
    protected EventHandler<MouseEvent> getAction() {
        return mouseEvent -> {
            try {
                ICity city = getCity();
                Optional<ITradingOffice> office = getPlayer().findTradingOffice(getCity());
                Preconditions.checkArgument(office.isPresent(), "Trading office must be present");
                for (EWare ware : MATERIALS) {
                    int buyAmount = requirementsMap.get(ware).getBuyAmount().intValue();
                    city.move(ware, -buyAmount, getPlayer());
                    int needed = requirementsMap.get(ware).requiredProperty().get();
                    int fromWarhouse = needed - buyAmount;
                    office.get().move(ware, -fromWarhouse, 0);
                }
                ShipyardOrderBuild event = new ShipyardOrderBuild(city.getCityState().getShipyardState(), getCurrentShipType(), getPlayer());
                clientServerEventBus.post(event);
                getPlayer().getCompany().updateCash(-calculateTotalBuildCosts());
                executeOnCloseButtonClicked();
            } catch (RuntimeException e) {
                logger.error("Failed to order ship construction", e);
            }
        };
    }

    @Override
    protected String getActionText() {
        return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.build", new Object[]{}, locale.getCurrentLocal());
    }

    @Override
    protected Group createFooterText() {
        IntegerBinding totalPriceBinding = new IntegerBinding() {
            {
                for (EWare ware : MATERIALS) {
                    RequiredWareCityStorage requiredWare = requirementsMap.get(ware);
                    super.bind(requiredWare.buyPriceProperty());
                }
            }
            @Override
            protected int computeValue() {
                return calculateTotalBuildCosts();
            }
        };
        Group g = new Group();
        String t = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.totalSum", new Object[]{}, locale.getCurrentLocal());
        HashMap<String, Object> parameters = new HashMap<>();
        parameters.put("price", totalPriceBinding.get());
        DecoratedText dt = decoratedTextFactory.createDecoratedText(t, parameters);
        dt.setId("totalSum");
        g.getChildren().add(dt);
        g.setId("footerText");
        return g;
    }

    private int calculateTotalBuildCosts() {
        int total = getCity().getCityState().getShipyardState().calculateConstructionCosts(getCurrentShipType());
        for (EWare ware : MATERIALS) {
            RequiredWareCityStorage requiredWare = requirementsMap.get(ware);
            total += requiredWare.getBuyPrice().intValue();
        }
        return total;
    }

    @Override
    protected Table createMainTable() {
        Table model = new Table();
        TableHeader header = new TableHeader(5);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.materials", new Object[]{}, locale.getCurrentLocal())), ECellConstraint.COLSPAN2);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.stored", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.buy", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.ShipyardConstructionDialogV2.costs", new Object[]{}, locale.getCurrentLocal())));
        header.setAligenment(0, HPos.CENTER);
        header.setAligenment(2, HPos.CENTER);
        header.setAligenment(3, HPos.CENTER);
        header.setAligenment(4, HPos.CENTER);
        model.setHeader(header);
        model.setAligenment(0, HPos.RIGHT);
        model.setAligenment(1, HPos.RIGHT);
        model.setAligenment(2, HPos.RIGHT);
        model.setAligenment(3, HPos.RIGHT);
        model.setAligenment(4, HPos.RIGHT);

        model.setColumnWidth(70, 70, 70, 70, 100);
        for (final EWare ware : MATERIALS) {
            RequiredWareCityStorage requiredWare = requirementsMap.get(ware);
            final TableRow row = new TableRow();
            row.add(new StaticTextTableCell(translator.getLocalDisplayName(ware)));

            final IntegerProperty needed = requiredWare.requiredProperty();
            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = new BarrelAmountAlwaysVisible();
                barrelAmount.amountProperty().bind(needed.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = new BaleAmountAlwaysVisible();
                baleAmount.amountProperty().bind(needed.asString());
                row.add(new ControlTableCell(baleAmount));
            }


            final ReadOnlyIntegerProperty stored = requiredWare.inStorageProperty();
            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = new BarrelAmountAlwaysVisible();
                barrelAmount.amountProperty().bind(stored.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = new BaleAmountAlwaysVisible();
                baleAmount.amountProperty().bind(stored.asString());
                row.add(new ControlTableCell(baleAmount));
            }
            final IntegerBinding buyAmount = requiredWare.buyAmountProperty();

            if (ware.isBarrelSizedWare()) {
                BarrelAmountAlwaysVisible barrelAmount = new BarrelAmountAlwaysVisible();
                barrelAmount.amountProperty().bind(buyAmount.asString());
                row.add(new ControlTableCell(barrelAmount));
            } else {
                BaleAmountAlwaysVisible baleAmount = new BaleAmountAlwaysVisible();
                baleAmount.amountProperty().bind(buyAmount.asString());
                row.add(new ControlTableCell(baleAmount));
            }

            IntegerBinding priceBinding = requiredWare.buyPriceProperty();
            CoinPriceAlwaysVisible price = new CoinPriceAlwaysVisible();
            price.amountProperty().bind(priceBinding.asString());
            row.add(new ControlTableCell(price));

            model.add(row);
        }
        return model;
    }


    @Override
    protected Table createTopTable() {
        int capacity = shipFactory.calculateInitialCapacity(getCurrentShipType(), getCity().getCoordinates().getX());
        List<IWeaponSlot> shipWeapons = weaponsLocationFactory.getShipWeaponsLocation(getCurrentShipType(), EShipUpgrade.NONE);
        Table model = new Table();
        // Add dummy header which is not rendered
        TableHeader header = new TableHeader(5);
        model.setHeader(header);
        model.setColumnWidth(70, 70, 70, 70, 70);
        TableRow row = new TableRow();
        BarrelAmount capacityAmount = new BarrelAmount();
        capacityAmount.setAmount(capacity);
        row.add(new ControlTableCell(capacityAmount));
        WeaponSlotCount slotCount = shipService.getWeaponSlotCount(shipWeapons);
        row.add(new StaticTextTableCell(String.valueOf(slotCount.getNbSmallSlots())));
        row.add(new StaticTextTableCell(String.valueOf(slotCount.getNbLargeSlots())));
        row.add(new StaticTextTableCell(String.valueOf(shipFactory.getShipSpeed(getCurrentShipType()))));
        row.add(new StaticTextTableCell(String.valueOf(shipFactory.getMinimalSailors(getCurrentShipType()))));
        model.add(row);
        return model;
    }

    @Override
    protected BooleanProperty disableNavigation() {
        return new SimpleBooleanProperty(false); // there are always at least two ship types
    }

    @Override
    protected BooleanBinding actionEnabledBinding() {
        return new BooleanBinding() {
            {
                for (EWare ware : MATERIALS) {
                    RequiredWareCityStorage requiredWare = requirementsMap.get(ware);
                    super.bind(requiredWare.canBuyProperty());
                }
                super.bind(getPlayer().getCompany().cashProperty());
            }
            @Override
            protected boolean computeValue() {
                long cash = getPlayer().getCompany().getCash();
                long buildCosts = calculateTotalBuildCosts();
                if (cash < buildCosts) {
                    return false;
                }
                for (EWare ware : MATERIALS) {
                    RequiredWareCityStorage requiredWare = requirementsMap.get(ware);
                    if (!requiredWare.canBuyProperty().get()) {
                        return false;
                    }
                }
                return true;
            }
        };
    }
    @Override
    protected void selectionChanged() {
        super.selectionChanged();
        initializeRequirements();
        Group footerText = createFooterText();
        footerText.setLayoutX(50);
        footerText.setLayoutY(250 + 36 + 7 * 24);
        for (Iterator<Node> iterator = getContent().iterator(); iterator.hasNext(); ) {
            Node child =  iterator.next();
            if ("footerText".equals(child.getId())) {
                iterator.remove();
                break;
            }
        }
        getContent().add(footerText);
    }
}
