package ch.sahits.game.openpatrician.display.dialog.tavern;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.display.ClientViewState;
import ch.sahits.game.openpatrician.display.dialog.CloseButtonDialog;
import ch.sahits.game.openpatrician.display.dialog.service.DialogUtil;
import ch.sahits.game.openpatrician.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.openpatrician.event.EViewChangeEvent;
import ch.sahits.game.openpatrician.event.NoticeBoardUpdate;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.openpatrician.javafx.model.ControlTableCell;
import ch.sahits.game.openpatrician.javafx.model.StaticTextTableCell;
import ch.sahits.game.openpatrician.javafx.model.Table;
import ch.sahits.game.openpatrician.javafx.model.TableHeader;
import ch.sahits.game.openpatrician.javafx.model.TableRow;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.building.IWeaponStorage;
import ch.sahits.game.openpatrician.model.javafx.bindings.StaticIntegerBinding;
import ch.sahits.game.openpatrician.model.people.IWeaponsDealer;
import ch.sahits.game.openpatrician.model.product.ComputablePriceV2;
import ch.sahits.game.openpatrician.model.product.ETransferAmount;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.service.TransferUtil;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.geometry.HPos;
import javafx.scene.layout.GridPane;
import javafx.scene.text.Text;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Optional;

/**
 * Dialog for dealing weapons in the tavern.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 21, 2013
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class TavernWeaponsDealerDialog extends CloseButtonDialog {
    private final Logger logger = LogManager.getLogger(getClass());

    @ObjectPropertyType(ETransferAmount.class)
    private ObjectProperty<ETransferAmount> movableAmount = new SimpleObjectProperty<>(ETransferAmount.ONE);
    private final ICityPlayerProxyJFX city;
    private IWeaponStorage storage = null;
    private IntegerProperty currentStorageIndex = new SimpleIntegerProperty(-1);
    private StringBinding storageDestinationBinding;
    @Autowired
    private DialogUtil dialogUtil;
    @Autowired
    private TransferUtil transferUtil;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    private ComputablePriceV2 computablePrice;
    @Autowired
    private Locale locale;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private MessageSource messageSource;

    private EWeapon[] weapons = {EWeapon.SWORD, EWeapon.BOW, EWeapon.CROSSBOW, EWeapon.MUSKET};

    private final IWeaponsDealer weaponsDealer;

    public TavernWeaponsDealerDialog(ICityPlayerProxyJFX city) {
        super();
        this.city = city;
        weaponsDealer = city.getCity().getCityState().getTavernState().getWeaponsDealer();
        initStorageDestionationBinding();

    }

    @PostConstruct
    private void initializeDialog() {
        Optional<ITradingOffice> office = city.getPlayer().findTradingOffice(city.getCity());
        office.ifPresent(iTradingOffice -> storage = iTradingOffice.getWeaponStorage());
        setTitle(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernWeaponsDealerDialog.title", new Object[]{}, locale.getCurrentLocal()));

        String loadedText = messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernWeaponsDealerDialog.dialog1", new Object[0], locale.getCurrentLocal());


        HashMap<String, Object> parameters = new HashMap<>();
        DecoratedText dt = decoratedTextFactory.createDecoratedText(loadedText, parameters);


        getContent().addAll(dt);

        GridPane weapnsTable = createWeaponsTable();
        getContent().add(weapnsTable);

        OpenPatricianSmallWaxButton btn1 = new OpenPatricianSmallWaxButton("1");
        btn1.setId("amountONE");
        btn1.getStyleClass().add("actionButton");
        btn1.setOnAction(arg0 -> {
            try {
                movableAmount.setValue(ETransferAmount.ONE);
            } catch (RuntimeException e) {
                logger.error("Failed to switch to different movable amount", e);
            }
        });
        OpenPatricianSmallWaxButton btn5 = new OpenPatricianSmallWaxButton("5");
        btn5.setId("amountFIVE");
        btn5.getStyleClass().add("actionButton");
        btn5.setOnAction(arg0 -> {
            try {
                movableAmount.setValue(ETransferAmount.FIVE);
            } catch (RuntimeException e) {
                logger.error("Failed to switch to different movable amount", e);
            }
        });
        OpenPatricianSmallWaxButton btnMax = new OpenPatricianSmallWaxButton(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.max", new Object[]{}, locale.getCurrentLocal()));
        btnMax.setId("amountMAX");
        btnMax.getStyleClass().add("actionButton");
        btnMax.setOnAction(arg0 -> {
            try {
                movableAmount.setValue(ETransferAmount.MAX);
            } catch (RuntimeException e) {
                logger.error("Failed to switch to different movable amount", e);
            }
        });
        addButtomControlButtons(btn1, btn5,btnMax);
    }
    private GridPane createWeaponsTable() {

        Table model = new Table();
        TableHeader header = new TableHeader(5);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernWeaponsDealerDialog.weaponType", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernWeaponsDealerDialog.dealer", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.buy", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.BaseTradeDialog.sell", new Object[]{}, locale.getCurrentLocal())));
        Text storageHeader = new Text();
        storageHeader.getStyleClass().add("dialogText");
        storageHeader.textProperty().bind(storageDestinationBinding);
        header.add(new ControlTableCell(storageHeader));
        header.setAligenment(0, HPos.CENTER);
        header.setAligenment(1, HPos.CENTER);
        header.setAligenment(2, HPos.CENTER);
        header.setAligenment(3, HPos.CENTER);
        header.setAligenment(4, HPos.CENTER);
        model.setHeader(header);
        model.setAligenment(0, HPos.RIGHT);
        model.setAligenment(1, HPos.CENTER);
        model.setAligenment(2, HPos.CENTER);
        model.setAligenment(3, HPos.CENTER);
        model.setAligenment(4, HPos.CENTER);
        model.setColumnWidth(100, 70, 70, 70, 100);

        for (final EWeapon weapon : weapons) {
            final IntegerBinding storedAmount = getAmountStoredWeapons(weapon); // amount owned by the player
            final IntegerBinding amountToSell = getAmountBinding(storedAmount);
            final ReadOnlyIntegerProperty dealerOwnedAmount = weaponsDealer.amountAvailableProperty(weapon);

            TableRow row = new TableRow();
            row.add(new StaticTextTableCell(translator.getLocalDisplayName(weapon)));
            Text dealerAmount = new Text();
            dealerAmount.getStyleClass().add("dialogText");
            dealerAmount.textProperty().bind(dealerOwnedAmount.asString());
            row.add(new ControlTableCell(dealerAmount));
            OpenPatricianSmallWaxButton buyButton = new OpenPatricianSmallWaxButton("");
            buyButton.getStyleClass().add("actionButton");
            buyButton.setId("buy"+weapon);
            buyButton.setDisable(storage == null);
            final IntegerBinding buyPriceBinding = buyPriceBinding(movableAmount, weapon);
            buyButton.textProperty().bind(buyPriceBinding.asString());
            buyButton.setOnAction(mouseEvent -> {
                if (storage != null) {
                    try {
                        if (dealerOwnedAmount.get() > 0) {
                            int amountToBuy = transferUtil.calculateAvailableAmount(movableAmount, dealerOwnedAmount.get());
                            int affordableAmount = computablePrice.calculateAffordableAmount(weapon, dealerOwnedAmount.get(), amountToBuy,
                                    city.getPlayer().getCompany().getCash());

                            int avgPrice = buyPriceBinding.get();
                            city.getPlayer().getCompany().updateCash(-affordableAmount * avgPrice);
                            storage.update(weapon, affordableAmount);
                            weaponsDealer.move(weapon, -amountToBuy);
                            storedAmount.invalidate();
                        }
                    } catch (RuntimeException e) {
                        logger.error("Failed to buy from weapons dealer", e);
                    }
                }
            });
            row.add(new ControlTableCell(buyButton));


            OpenPatricianSmallWaxButton sellButton = new OpenPatricianSmallWaxButton("");
            sellButton.getStyleClass().add("actionButton");
            final IntegerBinding sellPriceBinding = sellPriceBinding(weapon, amountToSell);
            sellButton.textProperty().bind(sellPriceBinding.asString());
            sellButton.setOnAction(mouseEvent -> {
                try {
                    int amount = amountToSell.get();
                    int avgPrice = sellPriceBinding.get();
                    city.getPlayer().getCompany().updateCash(amount * avgPrice);
                        storage.update(weapon, -amount);
                    weaponsDealer.move(weapon, amount);
                    storedAmount.invalidate();
                } catch (RuntimeException e) {
                    logger.error("Failed to sell to weapons dealer", e);
                }
            });
            row.add(new ControlTableCell(sellButton));

            Text amountStored = new Text();
            amountStored.getStyleClass().add("dialogText");
            amountStored.textProperty().bind(storedAmount.asString());
            row.add(new ControlTableCell(amountStored));

            model.add(row);
        }

        GridPane pane = dialogUtil.createGridPaneFromModel(model);
        pane.getStyleClass().add("tableFont");
        pane.setLayoutX(2*FRAME_BORDER);
        pane.setLayoutY(350);
        return pane;
    }


    private void initStorageDestionationBinding() {
        if (storageDestinationBinding == null) {
            storageDestinationBinding = new StringBinding() {
                {
                    super.bind(currentStorageIndex);
                }

                @Override
                protected String computeValue() {
                        return messageSource.getMessage("ch.sahits.game.openpatrician.display.dialog.TavernWeaponsDealerDialog.tradingOffice", new Object[]{}, locale.getCurrentLocal());
                }
            };
        }
    }

    /**
     * Retrieve the binding for the amount of the weapon currently stored.
     * @param weapon type of the weapon
     * @return integer binding of the stored amount of the weapon
     */
    private IntegerBinding getAmountStoredWeapons(final EWeapon weapon) {
        return new IntegerBinding() {
            {
                if (storage != null) {
                    super.bind(storage.getWeaponProperty(weapon));
                }
            }
            @Override
            protected int computeValue() {
                if (storage != null) {
                    return storage.getWeapon(weapon);
                } else {
                    return 0;
                }
            }
        };
    }
    /**
     * Bind the amount that can be transferred on the maximal available amount
     * as well as the selected transfer amount. This reflects the amount the
     * dealer owns.
     * @param maxAmount binding of the amount the dealer owns
     * @return integer binding
     */
    private IntegerBinding getAmountBinding(final IntegerBinding maxAmount) {
        return new IntegerBinding() {
            {
                super.bind(movableAmount, maxAmount);
            }

            @Override
            protected int computeValue() {
                return transferUtil.calculateAvailableAmount(movableAmount, maxAmount.get());
            }
        };
    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    public void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_TAVERN);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }

    public IntegerBinding buyPriceBinding(final ObjectProperty<ETransferAmount> amount, final EWeapon weapon) {
        return new IntegerBinding() {
            {
                super.bind(weaponsDealer.amountAvailableProperty(weapon), amount, city.getPlayer().getCompany().cashProperty());
            }
            @Override
            protected int computeValue() {
                int availableAmount = weaponsDealer.amountAvailableProperty(weapon).get();
                int buyAmount = transferUtil.calculateAvailableAmount(amount, availableAmount);
                int affordableAmount = computablePrice.calculateAffordableAmount(weapon, availableAmount, buyAmount,
                        city.getPlayer().getCompany().getCash());
                if (affordableAmount>0) {
                    return computablePrice.buyPrice(weapon, new SimpleIntegerProperty(availableAmount), new StaticIntegerBinding(affordableAmount));
                } else {
                    return 0;
                }
            }
        };
    }

    public IntegerBinding sellPriceBinding(final EWeapon weapon, final IntegerBinding amountToSell) {
        return new IntegerBinding() {
            {
                super.bind(weaponsDealer.amountAvailableProperty(weapon), amountToSell);
            }
            @Override
            protected int computeValue() {
                if (amountToSell.get() > 0) {
                    int availableAmount = weaponsDealer.amountAvailableProperty(weapon).getValue(); // that should be the amount of the dealer
                    return computablePrice.sellPrice(weapon, new SimpleIntegerProperty(availableAmount), amountToSell);  // amountToSell is the amount stored on ship/office
                } else {
                    return 0;
                }
            }
        };
    }


}
